/*
 * JBoss, Home of Professional Open Source
 * Copyright 2007, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.proxy; 

import java.security.AccessController;
import java.security.Principal;
import java.security.PrivilegedAction; 
import java.security.PrivilegedActionException;
import java.security.PrivilegedExceptionAction;

import org.jboss.security.RunAs;  
import org.jboss.security.SecurityAssociation;
import org.jboss.security.SecurityContext; 
import org.jboss.security.SecurityContextFactory;
import org.jboss.security.SecurityContextAssociation;

//$Id$

/**
 * Interface defining the Privileged Blocks 
 *  @author <a href="mailto:Anil.Saldhana@jboss.org">Anil Saldhana</a>
 *  @since  Mar 5, 2007 
 *  @version $Revision$
 */
interface SecurityActions
{ 
   class UTIL
   {
      static SecurityActions getSecurityActions()
      {
         return System.getSecurityManager() == null ? NON_PRIVILEGED : PRIVILEGED;
      }
   }

   SecurityActions NON_PRIVILEGED = new SecurityActions()
   {
      public Principal getPrincipal()
      { 
         Principal p = null; 
         SecurityContext sc = SecurityContextAssociation.getSecurityContext(); 
         if(sc != null)
         { 
            p = sc.getUtil().getUserPrincipal();
         }
         if(p == null && SecurityContextAssociation.isClient())
           p = SecurityAssociation.getPrincipal();
         return p; 
      }

      public Object getCredential()
      {
       //return SecurityAssociation.getCredential();
         Object cred = null;
         SecurityContext sc = SecurityContextAssociation.getSecurityContext(); 
         if(sc != null)
         {  
            cred = sc.getUtil().getCredential();
         }
         if(cred == null && SecurityContextAssociation.isClient())
        	 cred = SecurityAssociation.getCredential();
         return cred; 
      }
      
      public RunAs getCallerRunAsIdentity()
      {
         RunAs rai = null;
         //Pluck the run-as identity from the existing SC if any
         SecurityContext existingSC = getSecurityContext();
         if(existingSC != null)
         { 
            rai = existingSC.getOutgoingRunAs();
         }
         return rai;   
      }

      public SecurityContext getSecurityContext()
      {
         return SecurityContextAssociation.getSecurityContext();
      }

      public void setSecurityContext(SecurityContext sc)
      {
         SecurityContextAssociation.setSecurityContext(sc);
      }
      
      public SecurityContext createSecurityContext(Principal p, Object cred, 
            String sdomain) throws Exception
      {
         return SecurityContextFactory.createSecurityContext(p,cred, null, sdomain);
      }
   };

   SecurityActions PRIVILEGED = new SecurityActions()
   {
      private final PrivilegedAction getPrincipalAction = new PrivilegedAction()
      {
         public Object run()
         {
            //return SecurityAssociation.getPrincipal();
            Principal p = null; 
            SecurityContext sc = SecurityContextAssociation.getSecurityContext(); 
            if(sc != null)
            { 
               p = sc.getUtil().getUserPrincipal();
            }
            return p; 
         }
      };

      private final PrivilegedAction getCredentialAction = new PrivilegedAction()
      {
         public Object run()
         {
            //return SecurityAssociation.getCredential();
            Object cred = null;
            SecurityContext sc = SecurityContextAssociation.getSecurityContext(); 
            if(sc != null)
            { 
               cred = sc.getUtil().getCredential();
            }
            return cred; 
         }
      };

      private final PrivilegedAction getSecurityContextAction = new PrivilegedAction()
      {
         public Object run()
         {
            return SecurityContextAssociation.getSecurityContext();
         }
      };  

      public Principal getPrincipal()
      {
         return (Principal)AccessController.doPrivileged(getPrincipalAction);
      }

      public Object getCredential()
      {
         return AccessController.doPrivileged(getCredentialAction);
      }
      
      public RunAs getCallerRunAsIdentity()
      {
         return (RunAs)AccessController.doPrivileged(new PrivilegedAction(){

            public Object run()
            {
               RunAs rai = null;
               //Pluck the run-as identity from the existing SC if any
               SecurityContext existingSC = getSecurityContext();
               if(existingSC != null)
               { 
                  rai = existingSC.getOutgoingRunAs();
               }
               return rai;  
            }});
         
      }

      public SecurityContext getSecurityContext()
      {
         return (SecurityContext) AccessController.doPrivileged(getSecurityContextAction);
      }

      public void setSecurityContext(final SecurityContext sc)
      {
         AccessController.doPrivileged(new PrivilegedAction(){

            public Object run()
            {
               SecurityContextAssociation.setSecurityContext(sc);
               return null;
            }}); 
      }
      
      public SecurityContext createSecurityContext(final Principal p, final Object cred, 
            final String sdomain) throws PrivilegedActionException
      {
         return (SecurityContext) AccessController.doPrivileged(new PrivilegedExceptionAction()
         { 
            public Object run() throws Exception
            {
               return SecurityContextFactory.createSecurityContext(p,cred, null, sdomain);
            }
            
         }); 
      }
   };

   Principal getPrincipal();

   Object getCredential();
   
   RunAs getCallerRunAsIdentity();

   SecurityContext createSecurityContext( Principal p,  Object cred, 
         String sdomain) throws Exception;
   
   SecurityContext getSecurityContext(); 
   
   void setSecurityContext(SecurityContext sc);  
}
