/*
 * JBoss, Home of Professional Open Source
 * Copyright 2007, Red Hat Middleware LLC, and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.web.deployers;

import java.io.IOException;
import java.lang.reflect.AnnotatedElement;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.jboss.deployers.spi.DeploymentException;
import org.jboss.deployers.spi.annotations.ScanningMetaData;
import org.jboss.deployers.spi.deployer.DeploymentStages;
import org.jboss.deployers.spi.deployer.helpers.AbstractDeployer;
import org.jboss.deployers.structure.spi.DeploymentUnit;
import org.jboss.deployers.vfs.spi.structure.VFSDeploymentUnit;
import org.jboss.deployment.AnnotatedClassFilter;
import org.jboss.metadata.annotation.creator.web.Web30MetaDataCreator;
import org.jboss.metadata.annotation.finder.AnnotationFinder;
import org.jboss.metadata.annotation.finder.DefaultAnnotationFinder;
import org.jboss.metadata.web.spec.WebMetaData;
import org.jboss.vfs.VirtualFile;

/**
 * A POST_CLASSLOADER deployer which generates metadata from
 * annotations
 * 
 * @author Scott.Stark@jboss.org
 * @version $Revision: 93949 $
 */
public class WarAnnotationMetaDataDeployer extends AbstractDeployer
{
   public static final String WEB_ANNOTATED_ATTACHMENT_NAME = "annotated."+WebMetaData.class.getName();

   private boolean metaDataCompleteIsDefault = false;

   public WarAnnotationMetaDataDeployer()
   {
      setStage(DeploymentStages.POST_CLASSLOADER);
      addInput(WebMetaData.class);
      addInput(ScanningMetaData.class);
      addOutput(WEB_ANNOTATED_ATTACHMENT_NAME);
   }

   public boolean isMetaDataCompleteIsDefault()
   {
      return metaDataCompleteIsDefault;
   }
   public void setMetaDataCompleteIsDefault(boolean metaDataCompleteIsDefault)
   {
      this.metaDataCompleteIsDefault = metaDataCompleteIsDefault;
   }

   public void deploy(DeploymentUnit unit) throws DeploymentException
   {
      if (unit instanceof VFSDeploymentUnit == false)
         return;
      
      VFSDeploymentUnit vfsDeploymentUnit = (VFSDeploymentUnit) unit;
      deploy(vfsDeploymentUnit);
   }

   public void undeploy(DeploymentUnit unit)
   {
      if (unit instanceof VFSDeploymentUnit == false)
         return;
      
      VFSDeploymentUnit vfsDeploymentUnit = (VFSDeploymentUnit) unit;
      undeploy(vfsDeploymentUnit);
   }

   /**
    * Process the 
    * 
    * @param unit the unit
    * @throws DeploymentException for any error
    */
   protected void deploy(VFSDeploymentUnit unit)
      throws DeploymentException
   {
      if (!unit.getSimpleName().endsWith(".war"))
      {
         return;
      }

      VirtualFile root = unit.getRoot();
      if(root.isFile())
         return;

      List<VirtualFile> classpath = unit.getClassPath();
      if(classpath == null || classpath.isEmpty())
         return;

      if (log.isTraceEnabled())
         log.trace("Deploying annotations for unit: " + unit + ", classpath: " + classpath);

      try
      {
         processMetaData(unit, classpath);
      }
      catch (Exception e)
      {
         throw DeploymentException.rethrowAsDeploymentException("Cannot process metadata", e);
      }
   }

   /**
    * Process metadata.
    *
    * @param unit the deployment unit
    * @param classpath the classpath
    * @throws Exception for any error
    */
   protected void processMetaData(VFSDeploymentUnit unit, List<VirtualFile> classpath) throws Exception
   {
      ScanningMetaData scanningMetaData = unit.getAttachment(ScanningMetaData.class);
      Map<VirtualFile, Collection<Class<?>>> classesPerJar = new HashMap<VirtualFile, Collection<Class<?>>>();
      boolean foundAnnotations = false;
      for (VirtualFile path : classpath)
      {
         if (scanningMetaData == null 
               || !(scanningMetaData.getPaths() == null 
                     || !scanningMetaData.getPaths().contains(path.getName())))
         {
            Collection<Class<?>> currentClasses = getClasses(unit, path);
            classesPerJar.put(path, currentClasses);
            if (currentClasses.size() > 0)
            {
               foundAnnotations = true;
            }
         }
      }
      if (foundAnnotations)
      {
         AnnotationFinder<AnnotatedElement> finder = new DefaultAnnotationFinder<AnnotatedElement>();
         processJBossWebMetaData(unit, finder, classesPerJar);
      }
   }

   /**
    * Get the classes we want to scan.
    *
    * @param unit the deployment unit
    * @param mainClassName the main class name
    * @param classpath the classpath
    * @return possible classes containing metadata annotations
    * @throws IOException for any error
    */
   protected Collection<Class<?>> getClasses(VFSDeploymentUnit unit, VirtualFile classpath) throws IOException
   {
      AnnotatedClassFilter classVisitor = new AnnotatedClassFilter(unit, unit.getClassLoader(), classpath, null);
      classpath.visit(classVisitor);
      Map<VirtualFile, Class<?>> classes = classVisitor.getAnnotatedClasses();
      if (classes != null && classes.size() > 0)
      {
         if(log.isTraceEnabled())
            log.trace("Annotated classes: " + classes);
      }
      else
      {
         classes = new HashMap<VirtualFile, Class<?>>();
      }
      return classes.values();
   }

   /**
    * Undeploy a vfs deployment
    * 
    * @param unit the unit
    */
   protected void undeploy(VFSDeploymentUnit unit)
   {
      // Nothing
   }

   /**
    * Process annotations.
    *
    * @param unit the deployment unit
    * @param finder the annotation finder
    * @param classes the candidate classes
    */
   protected void processJBossWebMetaData(VFSDeploymentUnit unit,
         AnnotationFinder<AnnotatedElement> finder, Map<VirtualFile, Collection<Class<?>>> classes)
   {
      Web30MetaDataCreator creator = new Web30MetaDataCreator(finder);
      boolean metaData = false;
      for (VirtualFile path : classes.keySet())
      {
         WebMetaData annotationMetaData = creator.create(classes.get(path));
         log.debug("Add annotations: " + WEB_ANNOTATED_ATTACHMENT_NAME + ":" + path.getName());
         if (annotationMetaData != null)
         {
            unit.addAttachment(WEB_ANNOTATED_ATTACHMENT_NAME + ":" + path.getName(), annotationMetaData, WebMetaData.class);
            metaData = true;
         }
      }
      if (metaData)
         unit.addAttachment(WEB_ANNOTATED_ATTACHMENT_NAME, Boolean.TRUE);
   }

}

