/*
* JBoss, Home of Professional Open Source
* Copyright 2006, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.system.deployers;

import java.io.ByteArrayInputStream;
import java.util.Properties;

import javax.management.ObjectName;

import org.jboss.deployers.plugins.classloading.ClassLoading;
import org.jboss.deployers.spi.DeploymentException;
import org.jboss.deployers.spi.deployer.DeploymentStages;
import org.jboss.deployers.spi.deployer.helpers.AbstractRealDeployer;
import org.jboss.deployers.structure.spi.DeploymentUnit;
import org.jboss.deployers.structure.spi.classloading.ClassLoaderMetaData;
import org.jboss.deployers.structure.spi.classloading.ExportAll;
import org.jboss.deployers.structure.spi.classloading.Version;
import org.jboss.mx.loading.LoaderRepositoryFactory.LoaderRepositoryConfig;
import org.jboss.system.metadata.ServiceDeployment;

/**
 * Hacky deployer to create ClassLoaderMetaData and Module from real
 * 
 * This deployer is responsible for creating classloaders for services of
 * type {@link ServiceDeployment}.
 * 
 * @author <a href="adrian@jboss.com">Adrian Brock</a>
 * @version $Revision: 1.1 $
 */
public class HackClassloaderMetaDataDeployer extends AbstractRealDeployer
{
   private ClassLoading classLoading;
   
   public ClassLoading getClassLoading()
   {
      return classLoading;
   }

   public void setClassLoading(ClassLoading classLoading)
   {
      this.classLoading = classLoading;
   }

   public HackClassloaderMetaDataDeployer()
   {
      setStage(DeploymentStages.DESCRIBE);
   }

   public void internalDeploy(DeploymentUnit unit) throws DeploymentException
   {
      // Check the loader repository config
      LoaderRepositoryConfig loaderConfig = unit.getAttachment(LoaderRepositoryConfig.class);
      if (loaderConfig != null)
      {
         log.debug("Using loader repository config: " + loaderConfig.repositoryName);
         ObjectName name = loaderConfig.repositoryName;
         if (name != null)
         {
            String domain = name.getCanonicalName().trim();
            if (domain.length() != 0)
            {
               ClassLoaderMetaData metaData = new ClassLoaderMetaData();
               metaData.setName(unit.getName());
               metaData.setDomain(domain);
               metaData.setExportAll(ExportAll.NON_EMPTY);
               metaData.setImportAll(true);
               metaData.setVersion(Version.DEFAULT_VERSION);

               Properties props = new Properties();
               String config = loaderConfig.repositoryConfig;
               try
               {
                  if (config != null)
                  {
                     ByteArrayInputStream bais = new ByteArrayInputStream(config.getBytes());
                     props.load(bais);
                  }
               }
               catch (Exception e)
               {
                  throw DeploymentException.rethrowAsDeploymentException("Error parsing repository config " + config, e);
               }
               String java2ParentDelegation = props.getProperty("java2ParentDelegation");
               if( java2ParentDelegation == null )
               {
                  // Check for previous mis-spelled property name
                  java2ParentDelegation = props.getProperty("java2ParentDelegaton", "false");
               }
               boolean useParentFirst = Boolean.valueOf(java2ParentDelegation).booleanValue();
               metaData.setJ2seClassLoadingCompliance(useParentFirst);
               
               unit.addAttachment(ClassLoaderMetaData.class, metaData);
            }
         }
      }
      else if(unit.getTopLevel() != unit)
      {
         // Check for a parent ClassLoaderMetaData
         ClassLoaderMetaData parentCMD = unit.getTopLevel().getAttachment(ClassLoaderMetaData.class);
         if(parentCMD != null)
         {
            // Use the parent domain
            ClassLoaderMetaData metaData = new ClassLoaderMetaData();
            metaData.setName(unit.getName());
            metaData.setDomain(parentCMD.getDomain());
            metaData.setExportAll(ExportAll.NON_EMPTY);
            metaData.setImportAll(true);
            metaData.setVersion(Version.DEFAULT_VERSION);
            unit.addAttachment(ClassLoaderMetaData.class, metaData);
         }
      }

      classLoading.addDeploymentUnit(unit);
   }

   public void internalUndeploy(DeploymentUnit unit)
   {
      if(classLoading != null)
         classLoading.removeDeploymentUnit(unit);
   }
}
