/*
* JBoss, Home of Professional Open Source
* Copyright 2006, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.system.deployers;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.Set;

import javax.management.MBeanServer;
import javax.management.ObjectName;

import org.jboss.deployers.spi.DeploymentException;
import org.jboss.deployers.spi.deployer.helpers.AbstractTopLevelClassLoaderDeployer;
import org.jboss.deployers.structure.spi.DeploymentContext;
import org.jboss.deployers.structure.spi.DeploymentUnit;
import org.jboss.deployers.vfs.spi.structure.VFSDeploymentContext;
import org.jboss.deployers.vfs.spi.structure.helpers.ClassPathVisitor;
import org.jboss.mx.loading.LoaderRepositoryFactory;
import org.jboss.mx.loading.RepositoryClassLoader;
import org.jboss.mx.loading.LoaderRepositoryFactory.LoaderRepositoryConfig;
import org.jboss.system.ServiceController;
import org.jboss.system.metadata.ServiceDeployment;
import org.jboss.virtual.VirtualFile;

/**
 * ServiceClassLoaderDeployer.<p>
 * 
 * This deployer is responsible for creating classloaders for services of
 * type {@link ServiceDeployment}.
 * 
 * @author <a href="adrian@jboss.com">Adrian Brock</a>
 * @version $Revision: 1.1 $
 */
public class ServiceClassLoaderDeployer extends AbstractTopLevelClassLoaderDeployer
{
   /** The service controller */
   private final ServiceController controller;
   
   /**
    * Create a new ServiceDeployer.
    * 
    * @param controller the service controller
    * @throws IllegalArgumentException for a null controller
    */
   public ServiceClassLoaderDeployer(ServiceController controller)
   {
      if (controller == null)
         throw new IllegalArgumentException("Null controller");
      this.controller = controller;
   }

   public ClassLoader createTopLevelClassLoader(DeploymentContext context) throws Exception
   {
      MBeanServer server = controller.getMBeanServer();
      DeploymentUnit unit = context.getDeploymentUnit();

      VirtualFile root = null;
      URL url = null;
      if (context instanceof VFSDeploymentContext)
      {
         VFSDeploymentContext vfsContext = (VFSDeploymentContext) context;
         
         // Try to determine a root url
         root = vfsContext.getRoot();
         try
         {
            if (root != null)
               url = trimJARURL(root.toURL());
         }
         catch (Exception ignored)
         {
            log.debug("Unable to get URL for " + context.getName() + " reason=" + ignored);
         }
      }

      // Check the loader repository config
      LoaderRepositoryConfig loaderConfig = unit.getAttachment(LoaderRepositoryConfig.class);
      if (loaderConfig != null)
         log.debug("Using loader repository config: " + loaderConfig.repositoryName);
      
      // Create default loader repository config
      if (loaderConfig == null)
      {
         loaderConfig = new LoaderRepositoryConfig();
         unit.addAttachment(LoaderRepositoryConfig.class, loaderConfig);
         log.trace("Using default loader repository config: " + loaderConfig.repositoryName);
      }

      LoaderRepositoryFactory.createLoaderRepository(server, loaderConfig);

      // Create the classloader
      Object[] args = { url, url, Boolean.TRUE };
      String[] sig =  { "java.net.URL", "java.net.URL", "boolean" };
      RepositoryClassLoader ucl = (RepositoryClassLoader) server.invoke(loaderConfig.repositoryName, "newClassLoader", args, sig);

      // Add in the classpath
      try
      {
         ClassPathVisitor visitor = new ClassPathVisitor();
         context.visit(visitor);
         Set<VirtualFile> classpath = visitor.getClassPath();
         for (VirtualFile path : classpath)
         {
            // TODO Weed duplicates from other deployments
            if (path != root)
               ucl.addURL(trimJARURL(path.toURL()));
         }
         
         // Register the classloader with the MBeanServer
         ObjectName uclName = ucl.getObjectName();
         if (server.isRegistered(uclName) == false)
            server.registerMBean(ucl, uclName);
      }
      catch (Throwable t)
      {
         internalRemoveClassLoader(context, ucl);
         throw DeploymentException.rethrowAsDeploymentException("Error creating classloader: " + context.getName(), t);
      }

      return ucl;
   }

   public void removeTopLevelClassLoader(DeploymentContext context)
   {
      RepositoryClassLoader ucl = (RepositoryClassLoader) context.getClassLoader();
      if (ucl == null)
         return;
      internalRemoveClassLoader(context, ucl);
   }
   
   /**
    * Remove the classloader
    * 
    * @param context the deployment context
    * @param ucl the unified classloader
    */
   private void internalRemoveClassLoader(DeploymentContext context, RepositoryClassLoader ucl)
   {
      MBeanServer server = controller.getMBeanServer();

      // Remove the classloader from the MBeanServer
      try
      {
         ObjectName uclName = ucl.getObjectName();
         if (server.isRegistered(uclName) == true )
            server.unregisterMBean(uclName);
      }
      catch (Throwable t)
      {
         log.warn("Error unregistering classloader mbean: " + ucl + " for " + context.getName(), t);
      }

      // Remove the classloader from the loader repository
      try
      {
         ucl.unregister();
      }
      catch (Throwable t)
      {
         log.warn("Error unregistering ucl: " + ucl + " for " + context.getName(), t);
      }

      // Remove the loader repository
      try
      {

         DeploymentUnit unit = context.getDeploymentUnit();
         ServiceDeployment deployment = unit.getAttachment(ServiceDeployment.class);
         if (deployment != null)
         {
            LoaderRepositoryConfig config = unit.getAttachment(LoaderRepositoryConfig.class);
            if (config != null)
               LoaderRepositoryFactory.destroyLoaderRepository(server, config.repositoryName);
         }
      }
      catch (Throwable t)
      {
         log.warn("Error removing classloader from repository: " + ucl + " for " + context.getName(), t);
      }
   }

   // TODO figure out why this is necessary
   private URL trimJARURL(URL url) throws MalformedURLException
   {
      String temp = url.toString();
      if (temp.startsWith("jar:") && temp.endsWith("!/"))
      {
         temp = temp.substring(4, temp.length()-2);
         return new URL(temp);
      }
      return url;
   }
}
