/*
* JBoss, Home of Professional Open Source
* Copyright 2006, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.system.deployers;

import java.net.URL;
import java.util.ArrayList;
import java.util.List;

import org.jboss.deployers.spi.DeploymentException;
import org.jboss.deployers.vfs.spi.deployer.JAXPDeployer;
import org.jboss.deployers.vfs.spi.structure.VFSDeploymentUnit;
import org.jboss.managed.api.factory.ManagedObjectFactory;
import org.jboss.mx.loading.LoaderRepositoryFactory.LoaderRepositoryConfig;
import org.jboss.system.deployers.managed.ServiceMetaDataICF;
import org.jboss.system.metadata.ServiceDeployment;
import org.jboss.system.metadata.ServiceDeploymentClassPath;
import org.jboss.system.metadata.ServiceDeploymentParser;
import org.jboss.system.metadata.ServiceMetaData;
import org.jboss.system.server.ServerConfig;
import org.jboss.system.server.ServerConfigLocator;
import org.jboss.virtual.VFS;
import org.jboss.virtual.VirtualFile;
import org.w3c.dom.Document;

/**
 * SARDeployer.<p>
 * 
 * This deployer is responsible for looking for -service.xml
 * and creating the metadata object.<p>
 * 
 * The {@link ServiceClassLoaderDeployer} and {@link ServiceDeployer} does the real work of deployment.
 * 
 * @author <a href="adrian@jboss.com">Adrian Brock</a>
 * @version $Revision: 1.1 $
 */
public class SARDeployer extends JAXPDeployer<ServiceDeployment>
{
   private ServiceMetaDataICF icf;

   /**
    * Create a new SARDeployer.
    * 
    * @throws IllegalArgumentException for a null kernel
    */
   public SARDeployer()
   {
      super(ServiceDeployment.class);
      setSuffix("-service.xml");
      // Enable the super class ManagedObjectCreator implementation
      setBuildManagedObject(true);
   }

   public ServiceMetaDataICF getIcf()
   {
      return icf;
   }

   public void setIcf(ServiceMetaDataICF icf)
   {
      this.icf = icf;
      // TODO - should we allow null to handle ICF removal?
      if (icf != null)
      {
         // Have the ServiceMetaData ManagedObject build from the code class
         ManagedObjectFactory mof = ManagedObjectFactory.getInstance();
         mof.setInstanceClassFactory(ServiceMetaData.class, icf);
      }
   }

   /**
    * @param unit - the deployment unit
    * @param file - the vf for the jboss-service.xml descriptor
    * @param document - the jaxp document for the jboss-service.xml descriptor
    */
   @Override
   protected ServiceDeployment parse(VFSDeploymentUnit unit, VirtualFile file, Document document) throws Exception
   {
      ServiceDeploymentParser parser = new ServiceDeploymentParser(document);
      ServiceDeployment parsed = parser.parse();
      String name = file.toURI().toString();
      parsed.setName(name);

      List<ServiceDeploymentClassPath> classPaths = parsed.getClassPaths();
      if (classPaths != null)
         processXMLClasspath(unit, classPaths);

      LoaderRepositoryConfig config = parsed.getLoaderRepositoryConfig();
      if (config != null)
         unit.addAttachment(LoaderRepositoryConfig.class.getName(), config);
      return parsed;
   }

   /**
    * Process the xml classpath
    * 
    * @param unit the unit
    * @param classpaths the classpaths
    * @throws Exception for any error
    */
   private void processXMLClasspath(VFSDeploymentUnit unit, List<ServiceDeploymentClassPath> classpaths) throws Exception
   {
      ArrayList<VirtualFile> classpath = new ArrayList<VirtualFile>();

      for (ServiceDeploymentClassPath path : classpaths)
      {
         String codebase = path.getCodeBase();
         String archives = path.getArchives();

         log.debug("Processing classpath: " + unit.getName() + " codebase=" + codebase + " archives=" + archives);
         VirtualFile codebaseFile = unit.getRoot();
         if (".".equals(codebase) == false)
         {
            ServerConfig config = ServerConfigLocator.locate();
            URL codeBaseURL = new URL(config.getServerHomeURL(), codebase);
            codebaseFile = VFS.getVirtualFile(codeBaseURL, "");
         }

         if (codebaseFile == null)
            throw new DeploymentException("Cannot use classpath without a root: " + unit.getName());

         if (archives == null)
         {
            classpath.add(codebaseFile);
            log.debug("Using codebase as classpath: " + unit.getName());
         }
         else
         {
            SARArchiveFilter filter = new SARArchiveFilter(archives);
            List<VirtualFile> archiveFiles = codebaseFile.getChildren(filter);
            classpath.addAll(archiveFiles);
         }

      }

      List<VirtualFile> origClassPath = unit.getClassPath();
      if (origClassPath != null)
         classpath.addAll(origClassPath);
      unit.setClassPath(classpath);
   }
}
