/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.system.server.profile.repository;

import java.net.URI;
import java.util.Collection;
import java.util.Collections;
import java.util.Map;
import java.util.Set;

import org.jboss.deployers.vfs.spi.client.VFSDeployment;
import org.jboss.managed.api.ManagedComponent;
import org.jboss.managed.api.ManagedDeployment.DeploymentPhase;
import org.jboss.profileservice.spi.DeploymentRepository;
import org.jboss.profileservice.spi.ModificationInfo;
import org.jboss.profileservice.spi.NoSuchDeploymentException;
import org.jboss.profileservice.spi.Profile;
import org.jboss.profileservice.spi.ProfileKey;
import org.jboss.util.JBossObject;
import org.jboss.util.JBossStringBuilder;
import org.jboss.virtual.VFS;
import org.jboss.virtual.VirtualFile;

/**
 * A profile implementation that uses the DeploymentRepository to store
 * deployments and admin metadata.
 * 
 * @author Scott.Stark@jboss.org
 * @version $Revision: 81380 $
 */
public class ProfileImpl extends JBossObject
   implements Profile
{
   private ProfileKey key;
   private String version;
   /** The repository containing the deployment contents */
   private DeploymentRepository repository;
   /** Is hot deployment checking enabled */
   private volatile boolean hotdeployEnabled;

   /**
    * 
    * @param repository the repository
    * @param key the key
    */
   public ProfileImpl(DeploymentRepository repository, ProfileKey key)
   {
      this.key = key;
      this.repository = repository;
      log.info("Using repository:"+repository);
   }

   public String getName()
   {
      return key.getName();
   }

   public ProfileKey getKey()
   {
      return key;
   }

   public String getVersion()
   {
      return version;
   }

   
   public long getLastModified()
   {
      return repository.getLastModified();
   }

   /**
    * Get the names of the deployments in the profile
    * @return names of deployments
    */
   public Set<String> getDeploymentNames()
   {
      return repository.getDeploymentNames();
   }
   public Set<String> getDeploymentNames(DeploymentPhase phase)
   {
      return repository.getDeploymentNames(phase);
   }

   public VirtualFile getRootFile(DeploymentPhase phase)
      throws Exception
   {
      URI uri = repository.getDeploymentURI(phase);
      return VFS.getRoot(uri);
   }

   public Collection<VFSDeployment> getDeployments(DeploymentPhase phase)
      throws Exception
   {
      return repository.getDeployments(phase);
   }
   public Collection<VFSDeployment> getDeployments()
      throws Exception
   {
      return repository.getDeployments();
   }

   /**
    * 
    */
   public Collection<ModificationInfo> getModifiedDeployments()
      throws Exception
   {
      Collection<ModificationInfo> modified = Collections.emptyList();
      if( hotdeployEnabled == true )
         modified = repository.getModifiedDeployments();
      return modified;
   }
   /**
    * Enable/disable getModifiedDeployments results.
    */
   public synchronized void enableModifiedDeploymentChecks(boolean flag)
   {
      this.hotdeployEnabled = flag;
   }

   public void addDeployment(VFSDeployment d, DeploymentPhase phase)
       throws Exception
   {
      String name = d.getName();
      repository.addDeployment(name, d, phase);
   }
   public void updateDeployment(VFSDeployment d, DeploymentPhase phase,
         ManagedComponent mo)
       throws Exception
   {
      repository.updateDeployment(d, phase, mo);
   }
   public VFSDeployment getDeployment(String name, DeploymentPhase phase)
       throws Exception, NoSuchDeploymentException
   {
      return repository.getDeployment(name, phase);
   }
   public VFSDeployment removeDeployment(String name, DeploymentPhase phase)
      throws Exception
   {
      return repository.removeDeployment(name, phase);
   }

   public Map<String, Object> getConfig()
   {
      // TODO Auto-generated method stub
      return null;
   }
   
   public boolean hasDeployment(String name, DeploymentPhase phase)
   {
      // TODO: Wolf: needs refactoring / review
      try
      {
         getDeployment(name, phase);
         return true;
      }
      catch(NoSuchDeploymentException e)
      {
         return false;
      }
      catch(Exception e)
      {
         log.warn("getDeployment return exception", e);
         return false;
      }
   }

   @Override
   protected void toString(JBossStringBuilder buffer)
   {
      buffer.append("key=");
      buffer.append(this.key);
   }
   
}
