/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.web.tomcat.service.deployers;

import java.net.URL;
import java.util.ArrayList;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;
import java.util.StringTokenizer;
import java.util.jar.Attributes;
import java.util.jar.Manifest;

import org.jboss.classloader.spi.ClassLoaderPolicy;
import org.jboss.deployers.plugins.classloading.AbstractClassLoaderSystemDeployer;
import org.jboss.deployers.plugins.classloading.Module;
import org.jboss.deployers.spi.DeploymentException;
import org.jboss.deployers.structure.spi.DeploymentContext;
import org.jboss.deployers.structure.spi.DeploymentUnit;
import org.jboss.deployers.vfs.plugins.classloader.VFSClassLoaderPolicy;
import org.jboss.deployers.vfs.spi.structure.VFSDeploymentContext;
import org.jboss.logging.Logger;
import org.jboss.metadata.web.jboss.JBossWebMetaData;
import org.jboss.virtual.VFSUtils;
import org.jboss.virtual.VirtualFile;

/**
 * The war deployment class loader deployer.
 * TODO: this deployer should not exist. Metadata should be
 * driving the existing top-level class loader deployer.
 * 
 * @author Scott.Stark@jboss.org
 * @author adrian@jboss.org
 * @version $Revision:$
 */
public class WarClassLoaderDeployer extends AbstractClassLoaderSystemDeployer
{
   /** The parent class loader first model flag */
   private boolean java2ClassLoadingCompliance = false;
   private boolean includeWebInfInClasspath;
   /** Package names that should be ignored for class loading */
   private Set<String> filteredPackages;

   /**
    * Create a new WarClassLoaderDeployer.
    */
   public WarClassLoaderDeployer()
   {
      super();
      setInput(JBossWebMetaData.class);
   }

   public boolean isJava2ClassLoadingCompliance()
   {
      return java2ClassLoadingCompliance;
   }
   public void setJava2ClassLoadingCompliance(boolean flag)
   {
      this.java2ClassLoadingCompliance = flag;
   }

   public Set<String> getFilteredPackages()
   {
      return filteredPackages;
   }
   public void setFilteredPackages(Set<String> pkgs)
   {
      this.filteredPackages = pkgs;
   }

   public boolean isIncludeWebInfInClasspath()
   {
      return includeWebInfInClasspath;
   }
   public void setIncludeWebInfInClasspath(boolean includeWebInfInClasspath)
   {
      this.includeWebInfInClasspath = includeWebInfInClasspath;
   }

   
   @Override
   public void deploy(DeploymentUnit unit) throws DeploymentException
   {
      if(unit.isAttachmentPresent(JBossWebMetaData.class))
         super.deploy(unit);
   }

   @Override
   public void undeploy(DeploymentUnit unit)
   {
      if(unit.isAttachmentPresent(JBossWebMetaData.class))
         super.undeploy(unit);
   }

   @Override
   protected ClassLoaderPolicy createClassLoaderPolicy(DeploymentContext context, Module module)
      throws Exception
   {
      VFSClassLoaderPolicy policy = null;
      try
      {
         if(context instanceof VFSDeploymentContext)
         {
            VFSDeploymentContext vfscontext = (VFSDeploymentContext) context;
            ArrayList<VirtualFile> paths = new ArrayList<VirtualFile>();
            if(vfscontext.getClassPath() != null)
               paths.addAll(vfscontext.getClassPath());
            LinkedHashSet<VirtualFile> cpSet = new LinkedHashSet<VirtualFile>();
            // Add the manifest classpath, bug
            VirtualFile war = vfscontext.getRoot();
            // TODO: restore when JBMICROCONT-232 is fixed.
            // VFSUtils.addManifestLocations(war, paths);
            // Add the WEB-INF if requested
            VirtualFile webInf = vfscontext.getFile("WEB-INF");
            if(includeWebInfInClasspath && webInf != null)
               cpSet.add(webInf);
            // Process the war manifest
            addManifestLocations(war, paths, log);
            if(paths != null)
               cpSet.addAll(paths);

            VirtualFile[] warFiles = new VirtualFile[cpSet.size()];
            cpSet.toArray(warFiles);
            policy = new VFSClassLoaderPolicy(module.getName(), warFiles);
            policy.setExportAll(module.getExportAll());
            policy.setImportAll(module.isImportAll());
            policy.setExcludedPackages(filteredPackages);
         }
      }
      catch(Exception e)
      {
         log.warn("Failed to create war ClassLoaderPolicy", e);
      }
      return policy;
   }

   public void removeClassLoader(DeploymentContext context) throws Exception
   {
   }

   public static void addManifestLocations(VirtualFile file, List<VirtualFile> paths, Logger log)
      throws Exception
   {
      Manifest manifest = VFSUtils.getManifest(file);
      if (manifest != null)
      {
         VirtualFile parent = file.getParent();
         URL parentURL = parent != null ? parent.toURL() : file.toURL();

         Attributes mainAttributes = manifest.getMainAttributes();
         String classPath = mainAttributes.getValue(Attributes.Name.CLASS_PATH);
         if(classPath == null)
            return;

         StringTokenizer tokenizer = new StringTokenizer(classPath);
         while (tokenizer.hasMoreTokens())
         {
            String path = tokenizer.nextToken();
            try
            {
               String parentPath = parentURL.toString();
               if(parentPath.endsWith("/") == false)
                  parentPath += "/";
               VirtualFile pathvf = parent.getChild(path);
               if(pathvf != null)
               {
                  paths.add(pathvf);
                  // Recursively process the target
                  addManifestLocations(pathvf, paths, log);
               }
            }
            catch (Exception e)
            {
               log.debug("Manifest Class-Path entry " + path + " ignored for " + file.getPathName() + " reason=" + e);
            }
         }
      }      
   }
}
