/*
* JBoss, Home of Professional Open Source
* Copyright 2005, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.web.tomcat.service.session;

import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import org.jboss.cache.Cache;
import org.jboss.cache.Fqn;
import org.jboss.cache.Node;
import org.jboss.cache.lock.TimeoutException;
import org.jboss.cache.pojo.PojoCache;

public class JBossCacheWrapper 
{
   private static final int RETRY = 3;
   private static final String RETRY_FAIL_MSG = 
      "Continued to catch TimeoutException during " + 
       RETRY + " retry attempts. Giving up.";
   private PojoCache pojoCache_;
   private Cache plainCache_;
   
   JBossCacheWrapper(PojoCache cache)
   {
      pojoCache_ = cache;
      plainCache_ = pojoCache_.getCache();
   }

   /**
    * Wrapper to embed retry logic.
    *
    * @param fqn
    * @param id
    * @return
    */
   Object get(Fqn fqn, String id)
   {
      return get(fqn, id, false);
   }

   /**
    * Wrapper to embed retry logic.
    *
    * @param fqn
    * @param id
    * @return
    */
   Object get(Fqn fqn, String id, boolean gravitate)
   {
      TimeoutException ex = null;
      for (int i = 0; i < RETRY; i++)
      {
         try
         {            
            Object value = null;            
            if (gravitate)
            {            
               plainCache_.getInvocationContext().getOptionOverrides()
                                                 .setForceDataGravitation(true);
               value = plainCache_.get(fqn, id);
            }
            else
            {
               value = plainCache_.get(fqn, id);
            }
            return value;
         }
         catch (TimeoutException e)
         {
            ex = e;
         }
      }
      
      throw new RuntimeException(RETRY_FAIL_MSG, ex);
   }

   /**
    * Wrapper to embed retry logic.
    *
    * @param fqn
    * @param id
    * @param value
    * @return
    */
   void put(Fqn fqn, String id, Object value)
   {
      TimeoutException ex = null;
      for (int i = 0; i < RETRY; i++)
      {
         try
         {
            plainCache_.put(fqn, id, value);
            return;
         }
         catch (TimeoutException e)
         {
            ex = e;
         }
      }
      
      throw new RuntimeException(RETRY_FAIL_MSG, ex);
   }


   /**
    * Wrapper to embed retry logic.
    *
    * @param fqn
    * @param map
    */
   void put(Fqn fqn, Map map)
   {
      TimeoutException ex = null;
      for (int i = 0; i < RETRY; i++)
      {
         try
         {
            plainCache_.put(fqn, map);
            return;
         }
         catch (TimeoutException e)
         {
            ex = e;
         }
      }
      
      throw new RuntimeException(RETRY_FAIL_MSG, ex);
   }

   /**
    * Wrapper to embed retry logic.
    *
    * @param fqn
    * @param id
    * @return
    */
   Object remove(Fqn fqn, String id)
   {
      TimeoutException ex = null;
      for (int i = 0; i < RETRY; i++)
      {
         try
         {
            return plainCache_.remove(fqn, id);
         }
         catch (TimeoutException e)
         {
            ex = e;
         }
      }
      
      throw new RuntimeException(RETRY_FAIL_MSG, ex);
   }

   /**
    * Wrapper to embed retry logic.
    *
    * @param fqn
    */
   void remove(Fqn fqn)
   {
      TimeoutException ex = null;
      for (int i = 0; i < RETRY; i++)
      {
         try
         {
            plainCache_.removeNode(fqn);
            return;
         }
         catch (TimeoutException e)
         {
            ex = e;
         }
      }
      
      throw new RuntimeException(RETRY_FAIL_MSG, ex);
   }
   
   /**
    * Wrapper to embed retry logic.
    *
    * @param fqn
    */
   void removeLocal(Fqn fqn)
   {
      TimeoutException ex = null;
      for (int i = 0; i < RETRY; i++)
      {
         try
         {
            plainCache_.getInvocationContext().getOptionOverrides()
                                              .setCacheModeLocal(true);
            plainCache_.removeNode(fqn);
            return;
         }
         catch (TimeoutException e)
         {
            ex = e;
         }
      }
      
      throw new RuntimeException(RETRY_FAIL_MSG, ex);
   }

   /**
    * Wrapper to embed retry logic.
    *
    * @param fqn
    */
   void evict(Fqn fqn)
   {
      // FIXME remove this method if it remains unused
      
      TimeoutException ex = null;
      for (int i = 0; i < RETRY; i++)
      {
         try
         {
            plainCache_.evict(fqn, false);
            return;
         }
         catch (TimeoutException e)
         {
            ex = e;
         }
      }
      
      throw new RuntimeException(RETRY_FAIL_MSG, ex);
   }
   
   void evictSubtree(Fqn fqn)
   {      
      TimeoutException ex = null;
      for (int i = 0; i < RETRY; i++)
      {
         try
         {
//            // Evict the node itself first, since if it stores a Pojo
//            // that will do everything
//            plainCache_.evict(fqn);
//            
//            // next do a depth first removal; this ensure all nodes
//            // are removed, not just their data map
//            Set children = plainCache_.getChildrenNames(fqn);
//            if (children != null)
//            {
//               for (Iterator it = children.iterator(); it.hasNext(); )
//               {
//                  Fqn child = new Fqn(fqn, it.next());
//                  plainCache_.evict(child);
//               }
//               
//               plainCache_.evict(fqn);
//            }
//            return;
            plainCache_.evict(fqn, true);
            return;
            
         }
         catch (TimeoutException e)
         {
            ex = e;
         }
      }
      
      throw new RuntimeException(RETRY_FAIL_MSG, ex);
   }
   
   void removeLocalSubtree(Fqn fqn)
   {
      // First remove the node itself.  If it is the root of an AOP
      // object tree, this will do everything
      removeLocal(fqn);

      // Next, clear any children      
      TimeoutException ex = null;
      for (int i = 0; i < RETRY; i++)
      {
         try
         {
            // Evict the node itself first, since if it stores a Pojo
            // that will do everything
            removeLocal(fqn);
            
            // next do a depth first removal; this ensures all nodes
            // are removed, not just their data map
            Node base = plainCache_.getRoot().getChild(fqn);
            if (base != null)
            {
               Set children = base.getChildren();
               if (children != null)
               {
                  for (Iterator it = children.iterator(); it.hasNext(); )
                  {
                     Node child = (Node) it.next();
                     removeLocalSubtree(child.getFqn());
                  }
                  
                  removeLocal(fqn);
               }
            }
            return;
            
         }
         catch (TimeoutException e)
         {
            ex = e;
         }
      }
      
      throw new RuntimeException(RETRY_FAIL_MSG, ex);
   }

}
