/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.web.tomcat.service.deployers;

import java.io.File;
import java.util.Iterator;

import javax.management.Attribute;
import javax.management.MBeanServer;
import javax.management.Notification;
import javax.management.NotificationListener;
import javax.management.ObjectInstance;
import javax.management.ObjectName;
import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.security.jacc.PolicyContext;

import org.apache.catalina.Lifecycle;
import org.apache.catalina.connector.Connector;
import org.apache.catalina.startup.CatalinaProperties;
import org.apache.tomcat.util.modeler.Registry;
import org.jboss.kernel.spi.dependency.KernelController;
import org.jboss.kernel.spi.dependency.KernelControllerContext;
import org.jboss.security.plugins.JaasSecurityManagerServiceMBean;
import org.jboss.system.ServiceMBeanSupport;
import org.jboss.system.server.Server;
import org.jboss.system.server.ServerImplMBean;
import org.jboss.web.tomcat.security.HttpServletRequestPolicyContextHandler;

/**
 * Temporary workaround to support controlling the lifecycle of the webserver runtime portion of TomcatDeployer via a
 * JMX service in the deploy directory. We want it in deploy so dependencies on services in deploy can be properly
 * expressed. We want it as a JMX service so the ServiceBindingManager can alter the connector ports.
 * <p>
 * A more long term solution involves:
 * <ol>
 * <li>separating out the JBossWeb runtime aspects from TomcatDeployer and putting them in a separate class</li>
 * <li>developing a ProfileService-based alternative to ServiceBindingManager</li>
 * </ol>
 * </p>
 * 
 * @author <a href="brian.stansberry@jboss.com">Brian Stansberry</a>
 * @version $Revision: 1.1 $
 */
public class TomcatService extends ServiceMBeanSupport implements NotificationListener, TomcatServiceMBean
{
   
   /** The associated Tomcat deployer * */
   private TomcatDeployer tomcatDeployer;

   // Dependency inject the TomcatDeployer pojo

   public TomcatDeployer getTomcatDeployer()
   {
      return tomcatDeployer;
   }

   public void setTomcatDeployer(TomcatDeployer tomcatDeployer)
   {
      this.tomcatDeployer = tomcatDeployer;
   }

   // In our lifecycle, we invoke the webserver lifecycle-related operations
   // in the TomcatDeployer

   @Override
   protected void startService() throws Exception
   {
      if (tomcatDeployer == null)
         throw new IllegalStateException("Must set TomcatDeployer before starting");

      tomcatDeployer.setServiceClassLoader(getClass().getClassLoader());

      // Load Catalina properties
      CatalinaProperties.getProperty("");

      log.debug("Starting tomcat deployer");
      MBeanServer server = super.getServer();
      System.setProperty("catalina.ext.dirs", (System.getProperty("jboss.server.home.dir") + File.separator + "lib"));

      String objectNameS = tomcatDeployer.getDomain() + ":type=server";
      ObjectName objectName = new ObjectName(objectNameS);

      // Set the modeler Registry MBeanServer to the that of the tomcat service
      Registry.getRegistry().setMBeanServer(server);

      Registry.getRegistry().registerComponent(Class.forName("org.apache.catalina.startup.Catalina").newInstance(),
            objectName, "org.apache.catalina.startup.Catalina");

      server.setAttribute(objectName, new Attribute("catalinaHome", System.getProperty("jboss.server.home.dir")));
      server.setAttribute(objectName, new Attribute("configFile", tomcatDeployer.getConfigFile()));
      server.setAttribute(objectName, new Attribute("useNaming", new Boolean(false)));
      server.setAttribute(objectName, new Attribute("useShutdownHook", new Boolean(false)));
      server.setAttribute(objectName, new Attribute("await", new Boolean(false)));
      server.setAttribute(objectName, new Attribute("redirectStreams", new Boolean(false)));

      server.invoke(objectName, "create", new Object[]{}, new String[]{});

      server.invoke(objectName, "start", new Object[]{}, new String[]{});

      // Set up the authenticators in JNDI such that they can be configured for web apps
      InitialContext ic = new InitialContext();
      try
      {
         ic.bind("TomcatAuthenticators", tomcatDeployer.getAuthenticators());
      }
      catch (NamingException ne)
      {
         if (log.isTraceEnabled())
            log.trace("Binding Authenticators to JNDI failed", ne);
      }
      finally
      {
         try
         {
            ic.close();
         }
         catch (NamingException nee)
         {
         }
      }

      // Register the web container JACC PolicyContextHandlers
      HttpServletRequestPolicyContextHandler handler = new HttpServletRequestPolicyContextHandler();
      PolicyContext.registerHandler(HttpServletRequestPolicyContextHandler.WEB_REQUEST_KEY, handler, true);

      // If we are hot-deployed *after* the overall server is started
      // we'll never receive Server.START_NOTIFICATION_TYPE, so check
      // with the Server and start the connectors immediately, if this is the case.
      // Otherwise register to receive the server start-up notification.
      Boolean started = (Boolean) server.getAttribute(ServerImplMBean.OBJECT_NAME, "Started");
      if (started.booleanValue() == true)
      {
         log.debug("Server '" + ServerImplMBean.OBJECT_NAME + "' already started, starting connectors now");

         startConnectors();
      }
      else
      {
         // Register for notification of the overall server startup
         log.debug("Server '" + ServerImplMBean.OBJECT_NAME + "' not started, registering for start-up notification");

         server.addNotificationListener(ServerImplMBean.OBJECT_NAME, this, null, null);
      }

   }

   @Override
   protected void stopService() throws Exception
   {

      if (tomcatDeployer == null)
         throw new IllegalStateException("Must set TomcatDeployer before stopping");

      // Hot undeploy
      Boolean inShutdown = (Boolean) server.getAttribute(ServerImplMBean.OBJECT_NAME, "InShutdown");
      if (inShutdown.booleanValue() == false)
      {
         log.debug("Server '" + ServerImplMBean.OBJECT_NAME + "' already started, stopping connectors now");

         stopConnectors();
      }

      MBeanServer server = super.getServer();
      String objectNameS = tomcatDeployer.getDomain() + ":type=server";
      ObjectName objectName = new ObjectName(objectNameS);

      server.invoke(objectName, "stop", new Object[]{}, new String[]{});

      server.invoke(objectName, "destroy", new Object[]{}, new String[]{});

      server.unregisterMBean(objectName);

      MBeanServer server2 = server;

      // Unregister any remaining jboss.web or Catalina MBeans
      ObjectName queryObjectName = new ObjectName(tomcatDeployer.getDomain() + ":*");
      Iterator iterator = server2.queryMBeans(queryObjectName, null).iterator();
      while (iterator.hasNext())
      {
         ObjectInstance oi = (ObjectInstance) iterator.next();
         ObjectName toRemove = oi.getObjectName();
         // Exception: Don't unregister the service right now
         if (!"WebServer".equals(toRemove.getKeyProperty("service")))
         {
            if (server2.isRegistered(toRemove))
            {
               server2.unregisterMBean(toRemove);
            }
         }
      }
      queryObjectName = new ObjectName("Catalina:*");
      iterator = server2.queryMBeans(queryObjectName, null).iterator();
      while (iterator.hasNext())
      {
         ObjectInstance oi = (ObjectInstance) iterator.next();
         ObjectName name = oi.getObjectName();
         server2.unregisterMBean(name);
      }

   }

   // Expose the TomcatDeployer MBean interface

   public String getConfigFile()
   {
      return tomcatDeployer == null ? null : tomcatDeployer.getConfigFile();
   }

   public String getContextMBeanCode()
   {
      return tomcatDeployer == null ? null : tomcatDeployer.getContextMBeanCode();
   }

   public boolean getUseJBossWebLoader()
   {
      return tomcatDeployer == null ? false : tomcatDeployer.getUseJBossWebLoader();
   }

   public String getDomain()
   {
      return tomcatDeployer == null ? null : tomcatDeployer.getDomain();
   }

   public String[] getFilteredPackages()
   {
      return tomcatDeployer == null ? null : tomcatDeployer.getFilteredPackages();
   }

   public String getManagerClass()
   {
      return tomcatDeployer == null ? null : tomcatDeployer.getManagerClass();
   }

   public String getSessionIdAlphabet()
   {
      return tomcatDeployer == null ? null : tomcatDeployer.getSessionIdAlphabet();
   }

   public String getSubjectAttributeName()
   {
      return tomcatDeployer == null ? null : tomcatDeployer.getSubjectAttributeName();
   }

   public boolean getDeleteWorkDirOnContextDestroy()
   {
      return tomcatDeployer == null ? false : tomcatDeployer.getDeleteWorkDirOnContextDestroy();
   }

   public boolean isAllowSelfPrivilegedWebApps()
   {
      return tomcatDeployer == null ? false : tomcatDeployer.isAllowSelfPrivilegedWebApps();
   }

   public void setAllowSelfPrivilegedWebApps(boolean flag)
   {
      if (tomcatDeployer != null)
         tomcatDeployer.setAllowSelfPrivilegedWebApps(flag);
   }

   public void setConfigFile(String configFile)
   {
      if (tomcatDeployer != null)
         tomcatDeployer.setConfigFile(configFile);
   }

   public void setContextMBeanCode(String className)
   {
      if (tomcatDeployer != null)
         tomcatDeployer.setContextMBeanCode(className);
   }

   public void setDeleteWorkDirOnContextDestroy(boolean flag)
   {
      if (tomcatDeployer != null)
         tomcatDeployer.setDeleteWorkDirOnContextDestroy(flag);
   }

   public void setDomain(String domainName)
   {
      if (tomcatDeployer != null)
         tomcatDeployer.setDomain(domainName);
   }

   public void setFilteredPackages(String[] pkgs)
   {
      if (tomcatDeployer != null)
         tomcatDeployer.setFilteredPackages(pkgs);
   }

   public void setManagerClass(String managerClass)
   {
      if (tomcatDeployer != null)
         tomcatDeployer.setManagerClass(managerClass);
   }

   public void setSecurityManagerService(JaasSecurityManagerServiceMBean mgr)
   {
      if (tomcatDeployer != null)
         tomcatDeployer.setSecurityManagerService(mgr);
   }

   public void setSessionIdAlphabet(String sessionIdAlphabet)
   {
      if (tomcatDeployer != null)
         tomcatDeployer.setSessionIdAlphabet(sessionIdAlphabet);
   }

   public void setSubjectAttributeName(String name)
   {
      if (tomcatDeployer != null)
         tomcatDeployer.setSubjectAttributeName(name);
   }

   public void setUseJBossWebLoader(boolean flag)
   {
      if (tomcatDeployer != null)
         tomcatDeployer.setUseJBossWebLoader(flag);
   }

   public void startConnectors() throws Exception
   {
      if (tomcatDeployer == null)
         throw new IllegalStateException("Must set TomcatDeployer before starting connectors");

      MBeanServer server = super.getServer();
      ObjectName service = new ObjectName(tomcatDeployer.getDomain() + ":type=Service,serviceName=jboss.web");
      Object[] args = {};
      String[] sig = {};
      Connector[] connectors = (Connector[]) server.invoke(service, "findConnectors", args, sig);
      for (int n = 0; n < connectors.length; n++)
      {
         Lifecycle lc = connectors[n];
         lc.start();
      }
      /*
       * TODO: // Notify listeners that connectors have started processing requests sendNotification(new
       * Notification(TOMCAT_CONNECTORS_STARTED, this, getNextNotificationSequenceNumber()));
       */
   }

   public void stopConnectors() throws Exception
   {
      if (tomcatDeployer == null)
         throw new IllegalStateException("Must set TomcatDeployer before stopping connectors");

      MBeanServer server = super.getServer();
      ObjectName service = new ObjectName(tomcatDeployer.getDomain() + ":type=Service,serviceName=jboss.web");
      Object[] args = {};
      String[] sig = {};
      Connector[] connectors = (Connector[]) server.invoke(service, "findConnectors", args, sig);
      for (int n = 0; n < connectors.length; n++)
      {
         connectors[n].pause();
         connectors[n].stop();
      }
   }

   /**
    * Used to receive notification of the server start msg so the tomcat connectors can be started after all web apps
    * are deployed.
    */
   public void handleNotification(Notification msg, Object handback)
   {
      String type = msg.getType();
      if (type.equals(Server.START_NOTIFICATION_TYPE))
      {
         log.debug("Saw " + type + " notification, starting connectors");
         try
         {
            startConnectors();
         }
         catch (Exception e)
         {
            log.warn("Failed to startConnectors", e);
         }
      }
      if (type.equals(Server.STOP_NOTIFICATION_TYPE))
      {
         log.debug("Saw " + type + " notification, stopping connectors");
         try
         {
            stopConnectors();
         }
         catch (Exception e)
         {
            log.warn("Failed to stopConnectors", e);
         }
      }
   }

   public String getDefaultSecurityDomain()
   {
      return tomcatDeployer == null ? null : tomcatDeployer.getDefaultSecurityDomain();
   }

   public boolean getJava2ClassLoadingCompliance()
   {
      return tomcatDeployer == null ? false : tomcatDeployer.getJava2ClassLoadingCompliance();
   }

   public boolean getLenientEjbLink()
   {
      return tomcatDeployer == null ? false : tomcatDeployer.getLenientEjbLink();
   }

   public boolean getUnpackWars()
   {
      return tomcatDeployer == null ? false : tomcatDeployer.getUnpackWars();
   }

   public void setDefaultSecurityDomain(String defaultSecurityDomain)
   {
      if (tomcatDeployer != null)
         tomcatDeployer.setDefaultSecurityDomain(defaultSecurityDomain);
   }

   public void setJava2ClassLoadingCompliance(boolean flag)
   {
      if (tomcatDeployer != null)
         tomcatDeployer.setJava2ClassLoadingCompliance(flag);
   }

   public void setLenientEjbLink(boolean flag)
   {
      if (tomcatDeployer != null)
         tomcatDeployer.setLenientEjbLink(flag);
   }

   public void setUnpackWars(boolean flag)
   {
      if (tomcatDeployer != null)
         tomcatDeployer.setUnpackWars(flag);
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.jboss.web.tomcat.service.deployers.TomcatDeployerMBean#getHttpHeaderForSSOAuth()
    */
   public String getHttpHeaderForSSOAuth()
   {
      return tomcatDeployer == null ? null : tomcatDeployer.getHttpHeaderForSSOAuth();
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.jboss.web.tomcat.service.deployers.TomcatDeployerMBean#setHttpHeaderForSSOAuth(java.lang.String)
    */
   public void setHttpHeaderForSSOAuth(String httpHeaderForSSOAuth)
   {
      if (this.tomcatDeployer != null)
         this.tomcatDeployer.setHttpHeaderForSSOAuth(httpHeaderForSSOAuth);
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.jboss.web.tomcat.service.deployers.TomcatDeployerMBean#getSessionCookieForSSOAuth()
    */
   public String getSessionCookieForSSOAuth()
   {
      return tomcatDeployer == null ? null : tomcatDeployer.getSessionCookieForSSOAuth();
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.jboss.web.tomcat.service.deployers.TomcatDeployerMBean#setSessionCookieForSSOAuth(java.lang.String)
    */
   public void setSessionCookieForSSOAuth(String sessionCookieForSSOAuth)
   {
      if (this.tomcatDeployer != null)
         this.tomcatDeployer.setSessionCookieForSSOAuth(sessionCookieForSSOAuth);
   }
   
   /**
    * {@inheritDoc}
    * 
    * Overrides the superclass version to inject the <code>KernelController</code>
    * into {@link JBossWebMicrocontainerBeanLocator}.
    */
   @Override
   public void setKernelControllerContext(KernelControllerContext controllerContext) throws Exception
   {
      super.setKernelControllerContext(controllerContext);
      KernelController kernelController = controllerContext == null ? null : controllerContext.getKernel().getController();
      JBossWebMicrocontainerBeanLocator.setKernelController(kernelController);
   }
   
   /**
    * {@inheritDoc}
    * 
    * Overrides the superclass version to clear the <code>KernelController</code>
    * from {@link JBossWebMicrocontainerBeanLocator}.
    */
   @Override
   public void unsetKernelControllerContext(KernelControllerContext controllerContext) throws Exception
   {
      super.unsetKernelControllerContext(controllerContext);
      JBossWebMicrocontainerBeanLocator.setKernelController(null);
   }
}
