/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.web.jsf.integration.config;

import java.lang.reflect.Field;
import java.lang.reflect.Member;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.logging.Logger;

/**
 * This class provides access to the loggers used by Facelets version 1.1.15.
 *
 * @author Stan Silvert
 * @author Pete Muir
 */
public class FaceletsLoggers 
{
    
   // Don't allow an instance of this class
   private FaceletsLoggers() {}
    
   /**
    * Determine if Facelets is in the classpath.
    *
    * @return <code>true</code> if Facelets is available,
    *         <code>false</code> otherwise.
    */
   public static boolean isFaceletsAvailable()
   {
      try
      {
         classForName("com.sun.facelets.Facelet");
         return true;
      } 
      catch (ClassNotFoundException e)
      {
         return false;
      }
   }
   
   /**
    * Return an Iterator of all the java.util.logging.Logger objects used
    * in Facelets.
    *
    * @return The Loggers, or an empty Iterator if Facelets is not available.
    */
   public static Iterator<Logger> getLoggers() throws Exception
   {
      List<Logger> loggers = new ArrayList<Logger>();
      
      if (!isFaceletsAvailable()) return loggers.iterator();
      
      // Gah have to do this by reflection as the loggers are protected
         
      // And some aren't static, so this really is best effort
      loggers.add(getPrivateStaticLogger("com.sun.facelets.compiler.TagLibraryConfig", "log"));
      loggers.add(getPrivateStaticLogger("com.sun.facelets.compiler.Compiler", "log"));
      loggers.add(getPrivateStaticLogger("com.sun.facelets.impl.DefaultFaceletFactory", "log"));
      loggers.add(getPrivateStaticLogger("com.sun.facelets.tag.jsf.ComponentHandler", "log"));
      loggers.add(getPrivateStaticLogger("com.sun.facelets.util.Resource", "log"));
      loggers.add(getPrivateStaticLogger("com.sun.facelets.FaceletViewHandler", "log"));

      // These ones are in a package-scoped class
      loggers.add(getPrivateStaticLogger("com.sun.facelets.compiler.CompilationManager", "log"));
      loggers.add(getPrivateStaticLogger("com.sun.facelets.tag.jsf.ComponentRule", "log"));
      loggers.add(getPrivateStaticLogger("com.sun.facelets.tag.MetaRulesetImpl", "log"));
         
      return loggers.iterator();
   }
   
   private static Logger getPrivateStaticLogger(Class clazz, String fieldName) throws Exception 
   {
      Field field = getField(clazz, fieldName);
      field.setAccessible(true);
      return (Logger) get(field, new Object());
   }
   
   private static Logger getPrivateStaticLogger(String className, String fieldName) throws Exception
   {
      return getPrivateStaticLogger(classForName(className), fieldName);
   }
   
   // Code copied from org.jboss.seam.util.Reflctions
   private static Object get(Field field, Object target) throws Exception
   {
      try
      {
         return field.get(target);
      }
      catch (IllegalArgumentException iae)
      {
         String message = "Could not get field value by reflection: " + toString(field) + 
            " on: " + target.getClass().getName();
         throw new IllegalArgumentException(message, iae);
      }
   }
   
   // Code copied from org.jboss.seam.util.Reflctions
   private static Field getField(Class clazz, String name)
   {
      for ( Class superClass = clazz; superClass!=Object.class; superClass=superClass.getSuperclass() )
      {
         try
         {
            return superClass.getDeclaredField(name);
         }
         catch (NoSuchFieldException nsfe) {}
      }
      throw new IllegalArgumentException("no such field: " + clazz.getName() + '.' + name);
   }
   
   // Code copied from org.jboss.seam.util.Reflctions
   private static Class classForName(String name) throws ClassNotFoundException
   {
      try 
      {
         return Thread.currentThread().getContextClassLoader().loadClass(name);
      }
      catch (Exception e)
      {
         return Class.forName(name);
      }
   }
   
   // Code copied from org.jboss.seam.util.Reflctions
   private static String toString(Member member)
   {
      return unqualify( member.getDeclaringClass().getName() ) + 
            '.' + 
            member.getName();
   }
   
   // Code copied from org.jboss.seam.util.Strings
   private static String unqualify(String name)
   {
      return unqualify(name, '.');
   }
   
   // Code copied from org.jboss.seam.util.Strings
   private static String unqualify(String name, char sep)
   {
      return name.substring( name.lastIndexOf(sep)+1, name.length() );
   }
   
}
