/*
* JBoss, Home of Professional Open Source
* Copyright 2005, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.services.binding;

import java.net.InetAddress;
import java.net.UnknownHostException;

import javax.management.MalformedObjectNameException;
import javax.management.ObjectName;

/** A ServiceBinding is a {name,virtualHost,port,interfaceAddress}
 * quad specifying a named binding for a service.
 *
 * @author <a href="mailto:bitpushr@rochester.rr.com">Mike Finn</a>.
 * @author Scott.Stark@jboss.org
 * @version $Revision: 76997 $
 */
public class ServiceBinding
{
   /**
    * Checks if <code>serviceName</code> can be converted into an
    * ObjectName; if it can, converts it and returns its canonical form.
    * 
    * @param serviceName the service name
    * @return the canonicalized form, or <code>serviceName</code> if it
    *         cannot be converted into an ObjectName.
    */
   public static String canonicalizeServiceName(String serviceName)
   {      
      // If the serviceName looks like an object name, canonicalize it
      try
      {
         ObjectName oname = new ObjectName(serviceName);
         return oname.getCanonicalName();
      }
      catch (MalformedObjectNameException e)
      {
         return serviceName;
      }      
   }
   
   // ----------------------------------------------------------------- Fields
   
   /**
    * The name of the service to which the binding applies.
    */
   private final String serviceName;
   
   /** 
    * The name of the binding. A null or empty name implies the default
    * binding for a service.
    */
   private final String bindingName;
   
   /** The virtual host name. This is the interface name used to
    construct the bindAddress value. A null value implies bind on any
    interface.
    */
   private final String hostName;
   
   /** The port the service should listen on. A 0 value implies an
    anonymous port.
    */
   private final int port;
   
   /** The interface on which the service should bind its listening port. A
    null address implies bind on any interface.
    */
   private final InetAddress bindAddress;
   
   /** The ServiceBindingValueSource implementation class
    */
   private String serviceBindingValueSourceClassName;
   
   /** The ServiceBindingValueSource 
    */
   private ServiceBindingValueSource serviceBindingValueSource;
   
   /** An aribtrary object used to configure the behavior of
    the ServiceBindingValueSource. An example would be an XML Element.
    */
   private Object serviceBindingValueSourceConfig;

   // -----------------------------------------------------------  Constructors

   /**
    * Creates a new instance of ServiceBinding
    *
    * @param bindingName The name of the binding. A null or empty name
    * implies that default binding for a service.
    * @param port The port the service should listen on. A 0 value implies an
    * anonymous port.
    *
    * @exception UnknownHostException
    */
   public ServiceBinding(String serviceName, int port)
      throws UnknownHostException
   {
      this(serviceName, null, port);
   }   

   /**
    * Creates a new instance of ServiceBinding
    *
    * @param bindingName The name of the binding. A null or empty name
    * implies that default binding for a service.
    * @param port The port the service should listen on. A 0 value implies an
    * anonymous port.
    *
    * @exception UnknownHostException
    */
   public ServiceBinding(String serviceName, String hostName, int port)
      throws UnknownHostException
   {
      this(serviceName, null, hostName, port);
   }

   /**
    * Creates a new instance of ServiceBinding
    *
    * @param config the ServiceConfig associated with this binding. May be
    *               <code>null</code>, in which case {@link #setServiceConfig(ServiceConfig)}
    *               should be promptly called.
    * @param bindingName The name of the binding. A null or empty name
    * implies the default binding for a service.
    * @param hostName The virtual host name. This is the interface name used to
    * construct the bindAddress value. A null value implies bind on the loopback interface.
    * @param port The port the service should listen on. A 0 value implies an
    * anonymous port.
    *
    * @exception UnknownHostException If hostName is not resolvable.
    */
   public ServiceBinding(String serviceName, String bindingName, String hostName, int port)
      throws UnknownHostException
   {
      if (serviceName == null)
         throw new IllegalArgumentException("serviceName is null");
      
      // If the serviceName looks like an object name, canonicalize it      
      this.serviceName = canonicalizeServiceName(serviceName);
      this.bindingName = bindingName;
      this.port = port;
      this.hostName = hostName;
      this.bindAddress = InetAddress.getByName(hostName);
   } 

   // -------------------------------------------------------------  Properties
   
   public String getServiceName()
   {
      return serviceName;
   }

   /**
    * Getter for property name.
    *
    * @return The name of the binding
    */
   public String getBindingName()
   {
      return this.bindingName;
   }

   /**
    * Returns host name
    *
    * @return the hostname or address
    */
   public String getHostName()
   {
      return this.hostName;
   }

   /**
    * Gets the port attribute of the ServiceDescriptor object
    *
    * @return The listen port number
    */
   public int getPort()
   {
      return this.port;
   }

   /**
    * Gets the bindAddress attribute of the ServiceDescriptor object
    *
    * @return  The listen address
    */
   public InetAddress getBindAddress()
   {
      return this.bindAddress;
   }

   public synchronized ServiceBindingValueSource getServiceBindingValueSource() 
      throws ClassNotFoundException, InstantiationException, IllegalAccessException
   {
      if (this.serviceBindingValueSource == null && this.serviceBindingValueSourceClassName != null)
      {
         ClassLoader loader = Thread.currentThread().getContextClassLoader();
         Class<?> delegateClass = loader.loadClass(serviceBindingValueSourceClassName);
         this.serviceBindingValueSource = (ServiceBindingValueSource) delegateClass.newInstance();
      }
      return this.serviceBindingValueSource;
   }

   public void setServiceBindingValueSource(ServiceBindingValueSource serviceBindingValueSource)
   {
      this.serviceBindingValueSource = serviceBindingValueSource;
      if (serviceBindingValueSource != null)
      {
         setServiceBindingValueSourceClassName(serviceBindingValueSource.getClass().getName());
      }
   }

   /** 
    * Getter for property serviceBindingValueSourceClassName.
    */
   public String getServiceBindingValueSourceClassName()
   {
      return serviceBindingValueSourceClassName;
   }

   public void setServiceBindingValueSourceClassName(String serviceBindingValueSourceClassName)
   {
      this.serviceBindingValueSourceClassName = serviceBindingValueSourceClassName;
   }

   /** 
    * Getter for property serviceBindingValueSourceConfig.
    */
   public Object getServiceBindingValueSourceConfig()
   {
      return serviceBindingValueSourceConfig;
   }
   
   public void setServiceBindingValueSourceConfig(Object serviceBindingValueSourceConfig)
   {
      this.serviceBindingValueSourceConfig = serviceBindingValueSourceConfig;
   }

   // ----------------------------------------------------------------   Public
   
   public ServiceBinding getOffsetBinding(int offset) throws UnknownHostException
   {
      ServiceBinding result = new ServiceBinding(serviceName, bindingName, hostName, port + offset);
      result.setServiceBindingValueSourceClassName(serviceBindingValueSourceClassName);
      result.setServiceBindingValueSource(serviceBindingValueSource);
      result.setServiceBindingValueSourceConfig(serviceBindingValueSourceConfig);
      return result;
   }

   // --------------------------------------------------------------  Overrides
   
   /**
    * Equality is based on our ServiceConfig's serviceName and our bindingName.
    */
   @Override
   public boolean equals(Object obj)
   {
      if (this == obj)
         return true;
      
      if (obj instanceof ServiceBinding)
      {
         ServiceBinding other = (ServiceBinding) obj;
         return (this.serviceName.equals(other.serviceName)
                 && safeEquals(this.bindingName, other.bindingName));
      }
      
      return false;
   }

   /**
    * Hashcode is based on our ServiceConfig's serviceName and our bindingName.
    */
   @Override
   public int hashCode()
   {
      int result = 19;
      result += 29 * this.serviceName.hashCode();
      result += 29 * (this.bindingName == null ? 0 : this.bindingName.hashCode());
      return result;
   }

   /**
    * Create string representation of the service descriptor
    *
    * @return  String containing service descriptor properties
    */
   public String toString()
   {
      StringBuffer sBuf = new StringBuffer("ServiceBinding [serviceName=");
      sBuf.append(this.serviceName);
      sBuf.append(";bindingName=");

      sBuf.append(this.getBindingName());
      sBuf.append(";hostName=");
      String host = getHostName();

      if (hostName == null)
      {
         host = "<ANY>";
      }
      sBuf.append(host);
      sBuf.append(";bindAddress=");
      sBuf.append(this.getBindAddress().toString());
      sBuf.append(";port=");
      sBuf.append(this.getPort());
      sBuf.append("]");
      return sBuf.toString();
   }

   // ----------------------------------------------------------------  Private
   
   private boolean safeEquals(Object a, Object b)
   {
      return (a == b || (a != null && a.equals(b)));
   }
}
