/*
* JBoss, Home of Professional Open Source
* Copyright 2005, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.services.binding;


import java.net.InetAddress;
import java.net.URL;
import java.net.UnknownHostException;

import org.jboss.services.binding.impl.SimpleServiceBindingValueSourceImpl;
import org.jboss.services.binding.impl.StringReplacementServiceBindingValueSourceImpl;
import org.jboss.services.binding.impl.Util;
import org.jboss.services.binding.impl.XSLTServiceBindingValueSourceConfig;
import org.jboss.services.binding.impl.XSLTServiceBindingValueSourceImpl;
import org.w3c.dom.Element;

/** 
 * The services configuration binding manager implementation.
 *
 * <p>The ServiceBindingManager enables the centralized management
 * of ports, by service. The port configuration store is abstracted out
 * using the ServiceBindingStore interface.
 *
 * @version $Revision: 76997 $
 * @author  <a href="mailto:bitpushr@rochester.rr.com">Mike Finn</a>
 * @author Scott.Stark@jboss.org
 * @author <a href="mailto:dimitris@jboss.org">Dimitris Andreadis</a>
 * @author Brian Stansberry
 *
 * @jmx:mbean
 */
public class ServiceBindingManager
   implements ServiceBindingManagerMBean
{  
   // -----------------------------------------------------------------  Static
   
   /** Enumeration of types of binding requests */
   public enum BindingType { INT, INETADDRESS, STRING, ELEMENT, URL, RESOURCE, GENERIC };
   
   /**
    * Algorithm for obtaining a ServiceBindingValueSource given a particular
    * binding and binding type.
    * 
    * @param binding the binding
    * @param bindingType the binding type
    * @return the appropriate ServiceBindingValueSource. Will not return <code>null</code>.
    * 
    * @throws ClassNotFoundException if any {@link ServiceBinding#getServiceBindingValueSourceClassName()} cannot be found
    * @throws InstantiationException if any {@link ServiceBinding#getServiceBindingValueSourceClassName()} cannot be instantiated
    * @throws IllegalAccessException if any {@link ServiceBinding#getServiceBindingValueSourceClassName()} is not public
    * @throws IllegalStateException if no appropriate ServiceBindingValueSource can be identified
    */
   public static ServiceBindingValueSource getServiceBindingValueSource(ServiceBinding binding, BindingType bindingType)
         throws ClassNotFoundException, InstantiationException, IllegalAccessException
   {
      ServiceBindingValueSource source = binding.getServiceBindingValueSource();
      if (source == null)
      {
         switch (bindingType)
         {
            case INT:
            case INETADDRESS:
               source = new SimpleServiceBindingValueSourceImpl();
               break;
            case STRING:
               source = new StringReplacementServiceBindingValueSourceImpl();
               break;
            case ELEMENT:
            case URL:
            case RESOURCE:
               Object config = binding.getServiceBindingValueSourceConfig();
               if (config instanceof XSLTServiceBindingValueSourceConfig)
               {
                  source = new XSLTServiceBindingValueSourceImpl();
               }
               else
               {
                  source = new StringReplacementServiceBindingValueSourceImpl();
               }
               break;
            default:
               throw new IllegalStateException("No ServiceBindingValueSource configured for " + 
                                               binding + " and no default source available for binding of type " +
                                               bindingType);
         }         
      }
      return source;
   }

   // ----------------------------------------------------------------  Fields
   
   /** 
    * The name of the config set this manager is associated with. This is a
    * logical name used to lookup ServiceConfigs from the ServiceBindingStore.
    */
   private final String serverName;
   
   /** The ServiceConfig store instance
    */
   private final ServiceBindingStore store;

   // -----------------------------------------------------------  Constructors
   
   public ServiceBindingManager(String serverName, ServiceBindingStore store)
   {
      if (serverName == null)
         throw new IllegalArgumentException("serverName is null");
      if (store == null)
         throw new IllegalArgumentException("store is null");
      
      this.serverName = serverName;
      this.store = store;
   }

   // -------------------------------------------------------------  Properties
   
   /**
    * @jmx:managed-attribute
    */
   public String getServerName()
   {
      return this.serverName;
   }

   // ----------------------------------------------------------------- Public
   
   public int getIntBinding(String serviceName) throws Exception
   {
      return getIntBinding(serviceName, null);
   }
   
   public int getIntBinding(String serviceName, String bindingName) throws Exception
   {
      ServiceBinding binding = store.getServiceBinding(serverName, serviceName, bindingName);
      ServiceBindingValueSource source = getServiceBindingValueSource(binding, BindingType.INT);
      if (source instanceof IntServiceBindingValueSource)
      {
         return ((IntServiceBindingValueSource) source).getIntServiceBindingValue(binding);
      }
      else
      {
         return Util.getBindingValue(source, binding, Number.class).intValue();             
      }
   }
   
   public int getIntBinding(String serviceName, String bindingName, String hostName, int basePort) throws Exception
   {
      try
      {
         return getIntBinding(serviceName, bindingName);
      }
      catch (NoSuchBindingException e)
      {
         createBindingFromDefaults(serviceName, bindingName, hostName, basePort);
         return getIntBinding(serviceName, bindingName);
      }
   }

   private void createBindingFromDefaults(String serviceName, String bindingName, String hostName, int basePort)
         throws UnknownHostException, DuplicateServiceException
   {
      ServiceBinding sb = new ServiceBinding(serviceName, bindingName, hostName, basePort);
      store.addServiceBinding(sb);
   }
   
   public InetAddress getInetAddressBinding(String serviceName) throws Exception
   {
      return getInetAddressBinding(serviceName, null);
   }
   
   public InetAddress getInetAddressBinding(String serviceName, String bindingName) throws Exception
   {
      ServiceBinding binding = store.getServiceBinding(serverName, serviceName, bindingName);
      ServiceBindingValueSource source = getServiceBindingValueSource(binding, BindingType.INETADDRESS);
      if (source instanceof InetAddressServiceBindingValueSource)
      {
         return ((InetAddressServiceBindingValueSource) source).getInetAddressServiceBindingValue(binding);
      }
      else
      {
         return Util.getBindingValue(source, binding, InetAddress.class);
      }
   }
   
   public InetAddress getInetAddressBinding(String serviceName, String bindingName, String hostName, int basePort) throws Exception
   {
      try
      {
         return getInetAddressBinding(serviceName, bindingName);
      }
      catch (NoSuchBindingException e)
      {
         createBindingFromDefaults(serviceName, bindingName, hostName, basePort);
         return getInetAddressBinding(serviceName, bindingName);
      }
   }
   
   public String getStringBinding(String serviceName, String input) throws Exception
   {
      return getStringBinding(serviceName, null, input);
   }
   
   public String getStringBinding(String serviceName, String bindingName, String input) throws Exception
   {
      ServiceBinding binding = store.getServiceBinding(serverName, serviceName, bindingName);
      ServiceBindingValueSource source = getServiceBindingValueSource(binding, BindingType.STRING);
      if (source instanceof StringServiceBindingValueSource)
      {
         return ((StringServiceBindingValueSource) source).getStringServiceBindingValue(binding, input);
      }
      else
      {
         return Util.getBindingValueWithInput(source, binding, input, String.class);             
      }
   }
   
   public String getStringBinding(String serviceName, String bindingName, String input, String hostName, int basePort) throws Exception
   {
      try
      {
         return getStringBinding(serviceName, bindingName, input);
      }
      catch (NoSuchBindingException e)
      {
         createBindingFromDefaults(serviceName, bindingName, hostName, basePort);
         return getStringBinding(serviceName, bindingName, input);
      }
   }
   
   public Element getElementBinding(String serviceName, Element input) throws Exception
   {
      return getElementBinding(serviceName, null, input);
   }
   
   public Element getElementBinding(String serviceName, String bindingName, Element input) throws Exception
   {
      ServiceBinding binding = store.getServiceBinding(serverName, serviceName, bindingName);
      ServiceBindingValueSource source = getServiceBindingValueSource(binding, BindingType.ELEMENT);
      if (source instanceof ElementServiceBindingValueSource)
      {
         return ((ElementServiceBindingValueSource) source).getElementServiceBindingValue(binding, input);
      }
      else
      {
         return Util.getBindingValueWithInput(source, binding, input, Element.class);              
      }
   }
   
   public Element getElementBinding(String serviceName, String bindingName, Element input, String hostName, int basePort) throws Exception
   {
      try
      {
         return getElementBinding(serviceName, bindingName, input);
      }
      catch (NoSuchBindingException e)
      {
         createBindingFromDefaults(serviceName, bindingName, hostName, basePort);
         return getElementBinding(serviceName, bindingName, input);
      }
   }
   
   public URL getURLBinding(String serviceName, URL input) throws Exception
   {
      return getURLBinding(serviceName, null, input);
   }
   
   public URL getURLBinding(String serviceName, String bindingName, URL input) throws Exception
   {
      ServiceBinding binding = store.getServiceBinding(serverName, serviceName, bindingName);
      ServiceBindingValueSource source = getServiceBindingValueSource(binding, BindingType.URL);
      if (source instanceof URLServiceBindingValueSource)
      {
         return ((URLServiceBindingValueSource) source).getURLServiceBindingValue(binding, input);
      }
      else
      {
         return Util.getBindingValueWithInput(source, binding, input, URL.class);                 
      }
   }
   
   public URL getURLBinding(String serviceName, String bindingName, URL input, String hostName, int basePort) throws Exception
   {
      try
      {
         return getURLBinding(serviceName, bindingName, input);
      }
      catch (NoSuchBindingException e)
      {
         createBindingFromDefaults(serviceName, bindingName, hostName, basePort);
         return getURLBinding(serviceName, bindingName, input);
      }
   }
   
   public String getResourceBinding(String serviceName, String input) throws Exception
   {
      return getResourceBinding(serviceName, null, input);
   }
   
   public String getResourceBinding(String serviceName, String bindingName, String input) throws Exception
   {
      ServiceBinding binding = store.getServiceBinding(serverName, serviceName, bindingName);
      ServiceBindingValueSource source = getServiceBindingValueSource(binding, BindingType.RESOURCE);
      if (source instanceof URLServiceBindingValueSource)
      {
         return ((URLServiceBindingValueSource) source).getResourceServiceBindingValue(binding, input);
      }
      else
      {
         return Util.getBindingValueWithInput(source, binding, input, String.class);             
      }
   }
   
   public String getResourceBinding(String serviceName, String bindingName, String input, String hostName, int basePort) throws Exception
   {
      try
      {
         return getResourceBinding(serviceName, bindingName, input);
      }
      catch (NoSuchBindingException e)
      {
         createBindingFromDefaults(serviceName, bindingName, hostName, basePort);
         return getResourceBinding(serviceName, bindingName, input);
      }
   }
   
   public Object getGenericBinding(String serviceName, Object ... params) throws Exception
   {
      return getGenericBinding(serviceName, null, params);
   }
   
   /**
    * Generic extension point, allowing arbitrary return types based
    * on arbitrary sets of parameters.
    * 
    * @param serviceName
    * @param bindingName
    * @param params
    * @return
    * @throws Exception
    */
   public Object getGenericBinding(String serviceName, String bindingName, Object ... params) throws Exception
   {
      ServiceBinding binding = store.getServiceBinding(serverName, serviceName, bindingName);      
      ServiceBindingValueSource source = getServiceBindingValueSource(binding, BindingType.GENERIC);
      return source.getServiceBindingValue(binding, params); 
   }   
   
}
