/*
* JBoss, Home of Professional Open Source
* Copyright 2005, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.services.binding;

import java.net.InetAddress;


/** Interface for API to persist, read, and look up service configs
 *
 * @version $Revision: 37459 $
 * @author <a href="mailto:bitpushr@rochester.rr.com">Mike Finn</a>.
 * @author Scott.Stark@jboss.org
 * @author Brian Stansberry
 */
public interface ServiceBindingStore 
{
   /** Obtain a ServiceBinding object for the given server instance, target
    * service and binidng name. This is called by the JBoss service configuration
    * layer to obtain service attribute binding overrides.
    *
    * @param serverName the name identifying the JBoss server instance in
    *    which the service is running.
    * @param serviceName the name of the service
    * @param bindingName the name of the binding, or <code>null</code> to indicate
    *                    the default binding.
    * @return the ServiceBinding if one exists for the <serverName, serviceName, bindingName>
    *         tuple.
    *         
    * @throws NoSuchBindingException if no matching binding exists
    */
   ServiceBinding getServiceBinding(String serverName, String serviceName, String bindingName) 
      throws NoSuchBindingException;

   /** 
    * Add a ServiceBinding to the store. This is an optional method not used
    * by the JBoss service configuration layer.
    *
    * @param serverName the name identifying the JBoss server instance in
    *    which the service is running.
    * @param serviceName the JMX ObjectName of the service
    * @param serviceConfig the configuration to add
    * 
    * @throws DuplicateServiceException thrown if a configuration for the
    *    <serverName, serviceName> pair already exists.
    */
   void addServiceBinding(String serverName, ServiceBinding binding)
      throws DuplicateServiceException;

   /** 
    * Remove a service configuration from the store. This is an optonal method
    * not used by the JBoss service configuration layer.
    *
    * @param serverName the name identifying the JBoss server instance in
    *    which the service is running.
    * @param serviceBinding the binding
    */
   void removeServiceBinding(String serverName, ServiceBinding binding);

   /** 
    * Remove a service configuration from the store. This is an optonal method
    * not used by the JBoss service configuration layer.
    *
    * @param serverName the name identifying the JBoss server instance in
    *    which the service is running.
    * @param serviceBinding the binding
    */
   void removeServiceBinding(String serverName, String serviceName, String bindingName);

   /** 
    * Add a ServiceBinding to all binding sets in the store. For each binding 
    * set, a new ServiceBinding is added whose serviceName and bindingName
    * properties match the passed binding. The new binding's hostName matches
    * the target set's {@link #getDefaultHostName(String) default host name}.
    * The new binding's port is derived by taking the port from the passed 
    * binding and incrementing it by the target set's 
    * {@link #getDefaultPortOffset(String) default port offset}.
    *
    * @param serviceName the JMX ObjectName of the service
    * @param serviceConfig the configuration to add
    * 
    * @throws DuplicateServiceException thrown if a configuration for the
    *    <serverName, serviceName> pair already exists.
    */
   void addServiceBinding(ServiceBinding binding)
      throws DuplicateServiceException;

   /** 
    * Remove a service configuration from all binding sets in the store.
    *
    * @param serviceBinding the binding
    */
   void removeServiceBinding(ServiceBinding binding);

   /** 
    * Remove a service configuration from all binding sets in the store.
    *
    * @param serviceBinding the binding
    */
   void removeServiceBinding(String serviceName, String bindingName);
   
   /**
    * Gets the offset from a base value that by default should be added to
    * port values for a given serverName.
    * 
    * @param serverName the name of the binding set
    * @return the offset
    */
   int getDefaultPortOffset(String serverName);
   
   /**
    * Gets the default value to use as the host name for the given serverName.
    * 
    * @param serverName the name of the binding set
    * @return the host name
    */
   String getDefaultHostName(String serverName);
   
   /**
    * Value of <code>InetAddress.getByHost({@link #getDefaultHostName(String) getDefaultHost(serverName)})</code>.
    * 
    * @param serverName the name of the binding set
    * @return the host name
    */
   InetAddress getDefaultBindAddress(String serverName);
   
}
