/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.services.binding.impl;

import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

import org.jboss.services.binding.DuplicateServiceException;
import org.jboss.services.binding.NoSuchBindingException;
import org.jboss.services.binding.ServiceBinding;
import org.jboss.services.binding.ServiceBindingStore;

/**
 * A PojoServiceBindingStore.
 * 
 * @author Brian Stansberry
 * @version $Revision$
 */
public class PojoServiceBindingStore implements ServiceBindingStore
{
   private static final Object VALUE = new Object();
   
   private final ConcurrentMap<ServiceBindingKey, ServiceBinding> bindings = 
      new ConcurrentHashMap<ServiceBindingKey, ServiceBinding>(16, (float) .75, 4);
   
   private final ConcurrentMap<String, Object> bindingSetNames = 
      new ConcurrentHashMap<String, Object>(16, (float) .75, 4);
   
   private final Map<String, String> defaultHostNames = new HashMap<String, String>();
   
   private final Map<String, InetAddress> defaultBindAddresses = new HashMap<String, InetAddress>();
   
   private final Map<String, Integer> defaultPortOffsets = new HashMap<String, Integer>();
   
   public PojoServiceBindingStore(Map<String, ServiceBindingSet> configSets)
     throws DuplicateServiceException
   {
      if (configSets == null)
         throw new IllegalArgumentException("configSets is null");
      
      for (Map.Entry<String, ServiceBindingSet> entry : configSets.entrySet())
      {
         String serverName = entry.getKey();
         
         bindingSetNames.put(serverName, VALUE);
         
         ServiceBindingSet bindingSet = entry.getValue();
         
         defaultHostNames.put(serverName, bindingSet.getDefaultHostName());
         defaultBindAddresses.put(serverName, bindingSet.getDefaultBindAddress());
         defaultPortOffsets.put(serverName, bindingSet.getPortOffset());
         
         for (ServiceBinding binding : bindingSet)
         {
            addServiceBinding(serverName, binding);
         }
      }
   }
   
   public void addServiceBinding(String serverName, ServiceBinding binding) throws DuplicateServiceException
   {      
      validateServerName(serverName);
      Object oldBinding = bindings.putIfAbsent(new ServiceBindingKey(serverName, binding), binding);
      if (oldBinding != null)
      {
         throw new DuplicateServiceException(serverName, binding);
      }
   }

   public ServiceBinding getServiceBinding(String serverName, String serviceName, String bindingName)
         throws NoSuchBindingException
   {
      ServiceBinding binding = bindings.get(new ServiceBindingKey(serverName, serviceName, bindingName));
      if (binding == null)
      {
         throw new NoSuchBindingException(serverName, serviceName, bindingName);
      }
      return binding;
   }

   public void removeServiceBinding(String serverName, ServiceBinding binding)
   {
      validateServerName(serverName);
      bindings.remove(new ServiceBindingKey(serverName, binding));
   }

   public void removeServiceBinding(String serverName, String serviceName, String bindingName)
   {
      validateServerName(serverName);
      bindings.remove(new ServiceBindingKey(serverName, serviceName, bindingName));
   }
   
   public void addServiceBinding(ServiceBinding binding) throws DuplicateServiceException
   {
      for (String serverName : bindingSetNames.keySet())
      {
         int port = binding.getPort() + getDefaultPortOffset(serverName);
         String hostName = binding.getHostName();
         if (hostName == null)
            hostName = getDefaultHostName(serverName);
         
         try
         {
            addServiceBinding(serverName, new ServiceBinding(binding.getServiceName(), binding.getBindingName(), hostName, port));
         }
         catch (UnknownHostException e)
         {
            throw new IllegalStateException("Cannot convert " + hostName + " into an InetAddress");
         }
      }
   }

   public void removeServiceBinding(ServiceBinding binding)
   {
      for (String serverName : bindingSetNames.keySet())
      {
         removeServiceBinding(serverName, binding);
      }
   }

   public void removeServiceBinding(String serviceName, String bindingName)
   {
      for (String serverName : bindingSetNames.keySet())
      {
         removeServiceBinding(serverName, serviceName, bindingName);
      }
   }

   public InetAddress getDefaultBindAddress(String serverName)
   {
      validateServerName(serverName);
      return defaultBindAddresses.get(serverName);
   }

   public String getDefaultHostName(String serverName)
   {
      validateServerName(serverName);
      return defaultHostNames.get(serverName);
   }

   public int getDefaultPortOffset(String serverName)
   {
      validateServerName(serverName);
      return defaultPortOffsets.get(serverName).intValue();
   }

   private void validateServerName(String serverName)
   {
      if (bindingSetNames.containsKey(serverName) == false)
         throw new IllegalArgumentException("unknown serverName " +serverName);
   }

   private static class ServiceBindingKey
   {
      private final String serverName;
      private final String serviceName;
      private final String bindingName;
      
      private ServiceBindingKey(String serverName, ServiceBinding binding)
      {
         this(serverName, binding.getServiceName(), binding.getBindingName());
      }
      
      private ServiceBindingKey(String serverName, String serviceName, String bindingName)
      {
         assert serverName != null : "serverName is null";
         assert serviceName != null : "serviceName is null";
         
         this.serverName  = serverName;
         this.serviceName = ServiceBinding.canonicalizeServiceName(serviceName);
         this.bindingName = bindingName;
      }

      @Override
      public boolean equals(Object obj)
      {
         if (obj instanceof ServiceBindingKey)
         {
            ServiceBindingKey other = (ServiceBindingKey) obj;
            return (this.serverName.equals(other.serverName)
                    && this.serviceName.equals(other.serviceName)
                    && safeEquals(this.bindingName, other.bindingName));
         }
         return false;
      }

      @Override
      public int hashCode()
      {
         int result = 17;
         result += 23 * this.serverName.hashCode();
         result += 23 * this.serviceName.hashCode();
         result += 23 * (this.bindingName == null ? 0 : this.bindingName.hashCode());
         return result;
      }
      
      private boolean safeEquals(Object a, Object b)
      {
         return (a == b || (a != null && a.equals(b)));
      }
      
   }

}
