/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.services.binding.impl;

import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.HashSet;
import java.util.Set;

import org.jboss.services.binding.ServiceBinding;

/**
 * Set<ServiceBinding> that populates itself by taking a base set
 * and duplicating its binding with a fixed offset to each port value.
 * 
 * @author Brian Stansberry
 * @version $Revision$
 */
public class ServiceBindingSet extends HashSet<ServiceBinding>
{   
   /** The serialVersionUID */
   private static final long serialVersionUID = 765380451233486038L;

   private final String defaultHostName;
   private final InetAddress defaultBindAddress;
   private final int portOffset;
   
   public ServiceBindingSet(Set<ServiceBinding> base) throws UnknownHostException
   {
      this(base, 0, null, null);
   }

   public ServiceBindingSet(Set<ServiceBinding> base, int offset) throws UnknownHostException
   {
      this(base, offset, null, null);
   }

   public ServiceBindingSet(Set<ServiceBinding> base, int offset, String defaultHostName) throws UnknownHostException
   {
      this(base, offset, null, defaultHostName);
   }
   
   public ServiceBindingSet(Set<ServiceBinding> base, int offset, Set<ServiceBinding> overrides, String defaultHostName) throws UnknownHostException
   {
      super(getOffsetBindings(base, offset));
      if (overrides != null)
      {
         // HashSet.add() is not a replace operation, so must remove first
         for (ServiceBinding binding : overrides)
         {
            remove(binding);
            add(binding);
         }
      }
      
      this.defaultHostName = defaultHostName;
      this.defaultBindAddress = defaultHostName == null ? null : InetAddress.getByName(defaultHostName);
      this.portOffset = offset;
   }
   
   private static Set<ServiceBinding> getOffsetBindings(Set<ServiceBinding> base, int offset) throws UnknownHostException
   {
      Set<ServiceBinding> altered = new HashSet<ServiceBinding>(base.size());
      for (ServiceBinding binding : base)
      {
         altered.add(binding.getOffsetBinding(offset));
      }
      return altered;
   }

   public String getDefaultHostName()
   {
      return defaultHostName;
   }

   public InetAddress getDefaultBindAddress()
   {
      return defaultBindAddress;
   }

   public int getPortOffset()
   {
      return portOffset;
   }
   
}
