/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2010, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.weld.integration.deployer;

import org.jboss.classloading.spi.RealClassLoader;
import org.jboss.weld.bean.proxy.util.SimpleProxyServices;
import org.jboss.weld.exceptions.WeldException;
import org.jboss.weld.logging.messages.BeanMessage;
import org.jboss.weld.serialization.spi.ProxyServices;

import java.security.AccessController;
import java.security.PrivilegedAction;
import java.security.PrivilegedActionException;
import java.security.PrivilegedExceptionAction;

/**
 * A {@link ProxyServices} implementation for JBoss AS. This needs to be used if Weld is shared.
 *
 * @author Marius Bogoevici
 */
public class JBossProxyServices extends SimpleProxyServices
{

   public ClassLoader getClassLoader(final Class<?> proxiedBeanType)
   {
      SecurityManager sm = System.getSecurityManager();
      if (sm != null)
      {
         return AccessController.doPrivileged(new PrivilegedAction<ClassLoader>()
         {
            public ClassLoader run()
            {
               return _getClassLoader(proxiedBeanType);
            }
         });
      }
      else
      {
         return _getClassLoader(proxiedBeanType);
      }
   }

   /**
    * Return the thread context classloader or its first {@link org.jboss.classloading.spi.RealClassLoader}
    * parent (WARs WebCtxLoader$ENCLoader cannot be used for defining new proxy classes)
    *
    * If no {@link org.jboss.classloading.spi.RealClassLoader} is found, we return whatever we found.
    *
    * If there is no thread context classloader, then just fall back to the default Weld behaviour.
    *
    */
   private ClassLoader _getClassLoader(Class<?> proxiedBeanType)
   {
      ClassLoader threadContextClassLoader = getThreadContextClassLoader();
      ClassLoader actualClassLoader = threadContextClassLoader;
      while (actualClassLoader != null && !(actualClassLoader instanceof RealClassLoader))
      {
         actualClassLoader = actualClassLoader.getParent();
      }
      if (actualClassLoader != null)
      {
         return actualClassLoader;
      }
      else if (threadContextClassLoader != null)
      {
         return threadContextClassLoader;
      }
      else
      {
          // fall back to the default Weld behaviour
          return super.getClassLoader(proxiedBeanType);
      }
   }

   private static ClassLoader getThreadContextClassLoader()
   {
      SecurityManager sm = System.getSecurityManager();
      if (sm != null)
      {
         return AccessController.doPrivileged(new PrivilegedAction<ClassLoader>()
         {
            public ClassLoader run()
            {
               return Thread.currentThread().getContextClassLoader();
            }
         });
      }
      else
      {
         return Thread.currentThread().getContextClassLoader();
      }
   }
}