/*
 * JBoss, Home of Professional Open Source
 * Copyright 2010, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.jsf.deployer;

import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;

import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.jboss.deployers.spi.DeploymentException;
import org.jboss.deployers.spi.deployer.DeploymentStages;
import org.jboss.deployers.vfs.spi.deployer.AbstractSimpleVFSRealDeployer;
import org.jboss.deployers.vfs.spi.structure.VFSDeploymentUnit;
import org.jboss.metadata.javaee.spec.ParamValueMetaData;
import org.jboss.metadata.web.spec.ServletMetaData;
import org.jboss.metadata.web.spec.ServletsMetaData;
import org.jboss.metadata.web.spec.WebFragmentMetaData;
import org.jboss.metadata.web.spec.WebMetaData;
import org.jboss.util.StringPropertyReplacer;
import org.jboss.vfs.VirtualFile;
import org.jboss.vfs.VirtualFileFilter;
import org.jboss.vfs.util.SuffixMatchFilter;

/**
 * The Deployer is responsible for building metadata for the available JSF
 * implementations.
 *
 * Also, for each WAR deployment, decide which (if any) JSF impl will be used.
 *
 * @author Stan Silvert
 * @since 1.0
 */
public class JSFImplManagementDeployer extends AbstractSimpleVFSRealDeployer<WebMetaData>
{    
    private static final String JSF_CONFIG_PARAM = "org.jboss.jbossfaces.JSF_CONFIG_NAME";
    private static final String JSF_CONFIG_FILES_PARAM = "javax.faces.CONFIG_FILES";
    private static final String WAR_BUNDLES_JSF_PARAM = "org.jboss.jbossfaces.WAR_BUNDLES_JSF_IMPL";

    private static final VirtualFileFilter JAR_FILTER = new SuffixMatchFilter(".jar");
    private static final VirtualFileFilter FACES_CONFIG_FILTER = new SuffixMatchFilter(".faces-config.xml");

    private Collection<String> facesServlets;

    private String defaultJSFConfiguration;
    private Map<String,URL> jsfConfigurations;
    private boolean jsfDisabled = false;
    private boolean alwaysAddJSF = false;

    private Map<String, JSFImplMetaData> jsfImplMetatData = new HashMap<String, JSFImplMetaData>();

    /**
     * Create a new deployer.
     */
    public JSFImplManagementDeployer()
    {
        super(WebMetaData.class);
        setStage(DeploymentStages.POST_PARSE);
        setOutput(JSFImplMetaData.class);
    }

    public void start() throws DeploymentException
    {
        createJSFImplMetaData();
    }
    
    protected void createJSFImplMetaData() throws DeploymentException
    {
        if (jsfConfigurations == null)
        {
            log.warn("No JSF implementations specified in jsf-integration-deployer-jboss-beans.xml. Integrated JSF is disabled.");
            jsfDisabled = true;
            return;
        }

       for (Map.Entry<String, URL> implEntry : jsfConfigurations.entrySet())
       {
          try
          {
             jsfImplMetatData.put(implEntry.getKey(), new JSFImplMetaData(implEntry.getKey(), implEntry.getValue()));
          }
          catch (Exception e)
          {
             throw DeploymentException.rethrowAsDeploymentException("Malformed URL", e);
          }
       }

        log.info("Initialized " + jsfImplMetatData.size() + " JSF configurations: " + jsfImplMetatData.keySet());
    }

    /**
     * Should we add JSF to every WAR no matter what?
     * 
     * @param alwaysAddJSF Tells if we should always add JSF to a WAR.
     */
    public void setAlwaysAddJSF(boolean alwaysAddJSF)
    {
       this.alwaysAddJSF = alwaysAddJSF;
    }

    /**
     * Set the collection of servlets that signal this deployer to add JSF to a
     * WAR.
     *
     * @param facesServlets The servlet classes that signal this deployer to add
     *                    JSF to a WAR.
     */
    public void setFacesServlets(Collection<String> facesServlets)
    {
        this.facesServlets = Collections.unmodifiableCollection(facesServlets);
    }

    /**
     * Set the default JSF Configuration.
     *
     * @param defaultJSFConfig The name of the default JSF Configuration.
     */
    public void setDefaultJSFConfig(String defaultJSFConfig)
    {
        this.defaultJSFConfiguration = defaultJSFConfig;
    }

    /**
     *  Set the Map<ConfigName, URL> of all JSF Configurations.
     *
     * @param jsfConfigs The Map of all JSF Configurations.
     *
     * @throws MalformedURLException
     */
    public void setJsfConfigurations(Map<String, String> jsfConfigs) throws MalformedURLException
    {
        this.jsfConfigurations = new HashMap<String, URL>(jsfConfigs.size());

       for (Map.Entry<String, String> configEntry : jsfConfigs.entrySet())
       {
          String strURL = StringPropertyReplacer.replaceProperties(configEntry.getValue());
          jsfConfigurations.put(configEntry.getKey(), new URL(strURL));
       }
    }

    /**
     * Get JSF MetaData for a JSF Coniguration.
     *
     * @param configName
     *
     * @return The JSF MetaData
     */
    public JSFImplMetaData getJSFImpl(String configName)
    {
       return jsfImplMetatData.get(configName);
    }

    private boolean isJSFDeployment(VFSDeploymentUnit unit, WebMetaData metaData) throws DeploymentException
    {
       return
          alwaysAddJSF ||

          (getContextParam(JSF_CONFIG_PARAM, metaData) != null) ||

          (getContextParam(JSF_CONFIG_FILES_PARAM, metaData) != null) ||

          (unit.getFile("WEB-INF/faces-config.xml") != null) ||
       
          isJSFServletInWebInfWebXml(metaData.getServlets()) ||

          isJSFServletInWebFragment(unit) ||

          isFacesConfigInWebInfLibJar(unit);
    }

    private boolean isFacesConfigInWebInfLibJar(VFSDeploymentUnit unit) throws DeploymentException
    {
       VirtualFile webInfLib = unit.getFile("WEB-INF/lib");
       if (webInfLib == null) return false;

       try
       {
          for (VirtualFile jarFile : webInfLib.getChildren(JAR_FILTER))
          {
             VirtualFile metaInf = jarFile.getChild("META-INF");
             if (!metaInf.exists()) return false;

             VirtualFile facesConfig = metaInf.getChild("faces-config.xml");
             if (facesConfig.exists()) return true;

             if (!metaInf.getChildren(FACES_CONFIG_FILTER).isEmpty()) return true;
          }
       }
       catch (IOException e)
       {
          throw DeploymentException.rethrowAsDeploymentException("Unable to search WEB-INF/lib for faces-config files", e);
       }

       return false;
    }

    private boolean isJSFServletInWebInfWebXml(ServletsMetaData servlets)
    {
       if (servlets == null) return false;

       for (ServletMetaData servletMetaData : servlets)
       {
          String servletClass = servletMetaData.getServletClass();
          if (this.facesServlets.contains(servletClass))
          {
             return true;
          }
       }

       return false;
    }

    private boolean isJSFServletInWebFragment(VFSDeploymentUnit unit)
    {
       String fragmentName = WebFragmentMetaData.class.getName();
       Map<String, Object> attachments = unit.getAttachments();

       for (Map.Entry<String, Object> entry : attachments.entrySet())
       {
          WebFragmentMetaData fragment = null;
          if (entry.getKey().startsWith(fragmentName))
          {
             fragment = (WebFragmentMetaData)entry.getValue();
             if (isJSFServletInWebInfWebXml(fragment.getServlets())) return true;
          }
       }

       return false;
    }

    private String getContextParam(String paramName, WebMetaData metaData)
    {
       List<ParamValueMetaData> contextParams = metaData.getContextParams();
       if (contextParams == null) return null;

       for (ParamValueMetaData param : contextParams)
       {
          if (param.getParamName().equals(paramName))
          {
             return param.getParamValue();
          }
       }

       return null;
    }

    private boolean isWarBundlesJSF(WebMetaData metaData)
    {
        String warBundlesJSF = getContextParam(WAR_BUNDLES_JSF_PARAM, metaData );
        return (warBundlesJSF != null) && warBundlesJSF.equalsIgnoreCase("true");
    }

    @Override
    public void deploy(VFSDeploymentUnit unit, WebMetaData metaData) throws DeploymentException
    {
        if ( jsfDisabled || !isJSFDeployment(unit, metaData) || isWarBundlesJSF(metaData)) return;
        
        JSFImplMetaData jsfImpl = findWhichJSFImplToUse(metaData);
        unit.addAttachment(JSFImplMetaData.class, jsfImpl);
    }

    private JSFImplMetaData findWhichJSFImplToUse(WebMetaData metaData)
    {
        String implToUse = getContextParam(JSF_CONFIG_PARAM, metaData);
        if (implToUse != null)
        {
            return this.jsfImplMetatData.get(implToUse);
        }

        return this.jsfImplMetatData.get(this.defaultJSFConfiguration);
    }

}
