/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2010, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.jsf.deployer;

import java.io.IOException;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import org.jboss.deployers.spi.DeploymentException;
import org.jboss.metadata.web.spec.TldMetaData;
import org.jboss.metadata.web.spec.Web30MetaData;
import org.jboss.vfs.VFS;
import org.jboss.vfs.VirtualFile;
import org.jboss.vfs.VirtualFileFilter;
import org.jboss.vfs.VisitorAttributes;
import org.jboss.vfs.util.SuffixMatchFilter;
import org.jboss.xb.binding.JBossXBException;
import org.jboss.xb.binding.Unmarshaller;
import org.jboss.xb.binding.UnmarshallerFactory;
import org.jboss.xb.binding.sunday.unmarshalling.SchemaBinding;
import org.jboss.xb.binding.sunday.unmarshalling.SchemaBindingResolver;
import org.jboss.xb.binding.sunday.unmarshalling.SingletonSchemaResolverFactory;
import org.jboss.xb.builder.JBossXBBuilder;

/**
 * Bean that represents a single JSF implementation configuration (jars + web.xml)
 *
 * @author Stan Silvert
 * @since 1.0
 */
public class JSFImplMetaData
{
    private static final VirtualFileFilter JAR_FILTER = new SuffixMatchFilter(".jar");
    private static final VirtualFileFilter TLD_FILTER = new SuffixMatchFilter(".tld", VisitorAttributes.DEFAULT);

    /**
    * Unmarshall factory used for parsing shared web.xml.
    */
   private static final UnmarshallerFactory factory = UnmarshallerFactory.newInstance();

    private String implName;
    private URL implURL;
    private Set<URL> jsfJars;
    private List<TldMetaData> sharedTldMetaData;
    private Web30MetaData sharedWebXml;

    public JSFImplMetaData(String implName, URL implURL) throws Exception
    {
        this.implName = implName;
        this.implURL = implURL;
        findJSFJars();
        parseJSFImplWebMetaData();
    }

    private void findJSFJars() throws Exception
    {
        VirtualFile vFile = VFS.getChild(implURL);
        VirtualFile jsfLibsDir = vFile.getChild("jsf-libs");

        if (!jsfLibsDir.isDirectory())
        {
            throw new Exception(this.implName + ": jsf-libs must be a directory");
        }

        Set<URL> jars = new HashSet<URL>();
        for (VirtualFile jarFile : jsfLibsDir.getChildren(JAR_FILTER))
        {
            jars.add(jarFile.asFileURL());
        }

        this.jsfJars = Collections.unmodifiableSet(jars);
    }

    // This method will fail unless called late in the deployment lifecycle.
    // That's why it is called from getTldMetaData()
    private void findTLDMetaData() throws  URISyntaxException, IOException, JBossXBException
    {
        List<TldMetaData> sharedTlds = new ArrayList<TldMetaData>();

        if (this.jsfJars.isEmpty()) return;

        Unmarshaller unmarshaller = factory.newUnmarshaller();
        SchemaBindingResolver resolver = SingletonSchemaResolverFactory.getInstance().getSchemaBindingResolver();

        for (URL tldJar : jsfJars)
        {
            VirtualFile virtualFile = VFS.getChild(tldJar);
            VirtualFile metaInf = virtualFile.getChild("META-INF");
            if (metaInf == null) continue;
            List<VirtualFile> tlds = metaInf.getChildren(TLD_FILTER);
            for (VirtualFile tld : tlds)
            {
               // temporary fix because this tld can't be parsed
                if (tld.getName().endsWith("mojarra_ext.tld")) continue;

                TldMetaData tldMetaData = (TldMetaData) unmarshaller.unmarshal(tld.toURL().toString(), resolver);
                sharedTlds.add(tldMetaData);
            }
        }

        this.sharedTldMetaData = Collections.unmodifiableList(sharedTlds);
    }


   protected void parseJSFImplWebMetaData() throws DeploymentException
   {
      Unmarshaller unmarshaller = factory.newUnmarshaller();

      String webXml = implURL.toString() + "/META-INF/web.xml";

      SchemaBinding schema = JBossXBBuilder.build(Web30MetaData.class);

      try
      {
         this.sharedWebXml = (Web30MetaData) unmarshaller.unmarshal(webXml, schema);
      }
      catch (JBossXBException e)
      {
         throw new DeploymentException("Unable to parse web.xml at " + webXml, e);
      }
   }

    public String getImplName()
    {
        return this.implName;
    }

    public URL getImplURL()
    {
        return this.implURL;
    }

    public Web30MetaData getSharedWebXml()
    {
       return this.sharedWebXml;
    }

    public Set<URL> getJars()
    {
       return this.jsfJars;
    }

    public List<TldMetaData> getTldMetaData() throws DeploymentException
    {
        try
        {
            synchronized (this)
            {
                if (sharedTldMetaData == null) findTLDMetaData();
            }
        }
        catch (Exception e)
        {
            DeploymentException.rethrowAsDeploymentException("Unable to read TLD metadata for JSF implementation " + this.implName, e);
        }

        return this.sharedTldMetaData;
    }

    @Override
    public String toString()
    {
        return "JSF Impl " + this.implName + ": " + this.implURL.toString();
    }
}