/*
* JBoss, Home of Professional Open Source
* Copyright 2010, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.jsf.deployer;


import java.net.MalformedURLException;
import java.net.URL;
import java.util.Collection;
import java.util.HashSet;
import java.util.Set;
import org.jboss.deployers.spi.DeploymentException;
import org.jboss.deployers.vfs.plugins.classloader.UrlIntegrationDeployer;
import org.jboss.deployers.vfs.spi.structure.VFSDeploymentUnit;
import org.jboss.util.StringPropertyReplacer;

/**
 * The JSFUrlIntegrationDeployer adds the jars from the given JSF implementation
 * to the DeploymentUnit's classpath.
 *
 * @author Stan Silvert
 * @since 1.0
 */
public class JSFUrlIntegrationDeployer extends UrlIntegrationDeployer<JSFImplMetaData>
{
   private String jsfImplName;
   private JSFImplManagementDeployer implManager;
   private Collection<String> additionalURLs = new HashSet<String>(0);

   /**
    * Create a new deployer.
    */
   public JSFUrlIntegrationDeployer()
   {
      super(JSFImplMetaData.class);
      setDisableOptional(true);
   }

   public void setJSFImplName(String jsfImplName)
   {
      this.jsfImplName = jsfImplName;
   }

   public String getJSFImplName()
   {
      return this.jsfImplName;
   }

   public void setJSFImplManagementDeployer(JSFImplManagementDeployer implManager)
   {
      this.implManager = implManager;
   }

   public void setAdditionalURLs(Collection<String> additionalURLs)
   {
      this.additionalURLs = additionalURLs;
   }

   @Override
   public void start()
   {
      JSFImplMetaData jsfImpl = implManager.getJSFImpl(jsfImplName);
      if (jsfImpl == null)
         throw new IllegalStateException("JSF implementation " + jsfImplName + " not defined in JSFImplManagementDeployer.");
      
      Set<URL> integrationURLs = new HashSet<URL>();
      integrationURLs.addAll(jsfImpl.getJars());
      for (String url : this.additionalURLs)
      {
         try
         {
            integrationURLs.add(new URL(StringPropertyReplacer.replaceProperties(url)));
         }
         catch (MalformedURLException e)
         {
            throw new IllegalArgumentException("additionalURL " + url + " is invalid.", e);
         }
      }
      setIntegrationURLs(integrationURLs);
   } 

   @Override
   public void deploy(VFSDeploymentUnit unit, JSFImplMetaData jsfImpl) throws DeploymentException
   {
      if (!isIntegrationDeployment(unit, jsfImpl)) return;

      super.deploy(unit, jsfImpl);

      // Add JSF to any ancestor deployments
      while (!unit.isTopLevel())
      {
         unit = unit.getParent();
         super.deploy(unit, jsfImpl);
      } 
    }

   @Override
   public void undeploy(VFSDeploymentUnit unit, JSFImplMetaData jsfImpl)
   {
      if (!isIntegrationDeployment(unit, jsfImpl)) return;

      super.undeploy(unit, jsfImpl);

      // Remove JSF from any ancestor deployments
      while (!unit.isTopLevel())
      {
         unit = unit.getParent();
         super.undeploy(unit, jsfImpl);
      }
   }

   
   @Override
   protected boolean isIntegrationDeployment(VFSDeploymentUnit unit, JSFImplMetaData jsfImpl)
   {
      return jsfImpl.getImplName().equals(this.jsfImplName);
   }

   @Override
   protected boolean isIntegrationDeployment(VFSDeploymentUnit unit)
   {
      // If we got this far, we know that we need to return true
       return true;
   }
}