/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2009, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.logging.jdk;

import org.jboss.logging.AbstractLoggerPluginInstance;
import org.jboss.logging.LoggerPluginInstance;
import org.jboss.logging.LoggerPlugin;

import java.util.logging.Level;
import java.util.logging.LogRecord;
import java.util.logging.Logger;

import java.util.Map;
import java.util.EnumMap;

/**
 * An example LoggerPlugin which uses the JDK java.util.logging framework.
 *
 * @author Scott.Stark@jboss.org
 * @author David M. Lloyd
 * @version $Revison:$
 */
public class JDK14LoggerPluginInstance extends AbstractLoggerPluginInstance implements LoggerPluginInstance {

    @SuppressWarnings({ "NonConstantLogger" })
    private final Logger log;
    private final String name;

    public JDK14LoggerPluginInstance(final String name, final String resourceBundleName, final LoggerPlugin loggerPlugin) {
        super(name, resourceBundleName, loggerPlugin);
        this.name = name;
        log = resourceBundleName == null ? Logger.getLogger(name) : Logger.getLogger(name, resourceBundleName);
    }

    private static final Map<org.jboss.logging.Logger.Level, Level> LEVEL_MAP;

    static {
        final EnumMap<org.jboss.logging.Logger.Level, Level> map = new EnumMap<org.jboss.logging.Logger.Level, Level>(org.jboss.logging.Logger.Level.class);
        map.put(org.jboss.logging.Logger.Level.TRACE, JBossLevel.TRACE);
        map.put(org.jboss.logging.Logger.Level.DEBUG, JBossLevel.DEBUG);
        map.put(org.jboss.logging.Logger.Level.INFO, JBossLevel.INFO);
        map.put(org.jboss.logging.Logger.Level.WARN, JBossLevel.WARN);
        map.put(org.jboss.logging.Logger.Level.ERROR, JBossLevel.ERROR);
        map.put(org.jboss.logging.Logger.Level.FATAL, JBossLevel.FATAL);
        LEVEL_MAP = map;
    }

    public boolean isEnabled(final org.jboss.logging.Logger.Level level) {
        return log.isLoggable(LEVEL_MAP.get(level));
    }

    public void log(final org.jboss.logging.Logger.Level level, final String loggerFqcn, final Object message, final Object[] params, final Throwable t) {
        LogRecord record = new LogRecord(LEVEL_MAP.get(level), String.valueOf(message));
        record.setParameters(params);
        record.setLoggerName(name);
        record.setThrown(t);
        record.setSourceMethodName(null); // prevent expensive, yet guaranteed to be incorrect lookup
        log.log(record);
    }

    protected void log(final org.jboss.logging.Logger.Level level, final String loggerFqcn, final String message, final Throwable t) {
        LogRecord record = new LogRecord(LEVEL_MAP.get(level), message);
        record.setLoggerName(name);
        record.setThrown(t);
        record.setSourceMethodName(null); // prevent expensive, yet guaranteed to be incorrect lookup
        log.log(record);
    }
}
