/*
 * JBoss, Home of Professional Open Source
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.managed.plugins.factory;

import java.io.Serializable;
import java.lang.annotation.Annotation;
import java.util.Map;

import org.jboss.beans.info.spi.BeanInfo;
import org.jboss.config.spi.Configuration;
import org.jboss.logging.Logger;
import org.jboss.managed.api.Fields;
import org.jboss.managed.api.ManagedObject;
import org.jboss.managed.api.ManagedProperty;
import org.jboss.managed.api.annotation.ManagementObjectID;
import org.jboss.managed.api.annotation.ManagementRuntimeRef;
import org.jboss.managed.plugins.ManagedObjectImpl;
import org.jboss.managed.spi.factory.InstanceClassFactory;
import org.jboss.managed.spi.factory.ManagedObjectPopulator;
import org.jboss.metatype.api.values.MetaValue;
import org.jboss.metatype.api.values.SimpleValue;

/**
 * A default ManagedObjectPopulator implementation that relies on mc configuration
 * for obtaining BeanInfos for a class, and InstanceClassFactorys for
 * class, name and values from a value to be managed.
 * 
 * @param <T> the type to populate
 * @author Scott.Stark@jboss.org
 * @version $Revision: 78207 $
 */
public class AbstractManagedObjectPopulator<T extends Serializable>
   implements ManagedObjectPopulator<T>
{
   private static Logger log = Logger.getLogger(AbstractManagedObjectPopulator.class);

   /** The configuration */
   private Configuration configuration;
   private InstanceClassFactory<? extends Serializable> defaultInstanceFactory;
   /** The instance to class factories */
   private Map<Class<?>, InstanceClassFactory<? extends Serializable>> instanceFactories;


   public AbstractManagedObjectPopulator(Configuration configuration)
   {
      this(configuration, null, null);
   }
   public AbstractManagedObjectPopulator(
         Configuration configuration,
         InstanceClassFactory<? extends Serializable> defaultInstanceFactory,
         Map<Class<?>, InstanceClassFactory<? extends Serializable>> instanceFactories)
   {
      super();
      this.configuration = configuration;
      this.defaultInstanceFactory = defaultInstanceFactory;
      this.instanceFactories = instanceFactories;
   }
   
   public Configuration getConfiguration()
   {
      return configuration;
   }
   public void setConfiguration(Configuration configuration)
   {
      this.configuration = configuration;
   }

   public InstanceClassFactory<? extends Serializable> getDefaultInstanceFactory()
   {
      return defaultInstanceFactory;
   }
   public void setDefaultInstanceFactory(InstanceClassFactory<? extends Serializable> defaultInstanceFactory)
   {
      this.defaultInstanceFactory = defaultInstanceFactory;
   }

   public Map<Class<?>, InstanceClassFactory<? extends Serializable>> getInstanceFactories()
   {
      return instanceFactories;
   }
   public void setInstanceFactories(Map<Class<?>, InstanceClassFactory<? extends Serializable>> instanceFactories)
   {
      this.instanceFactories = instanceFactories;
   }

   public void createObject(ManagedObject managedObject, Class<T> clazz)
   {
      if (managedObject == null)
         throw new IllegalArgumentException("Null managed object");
      
      if (managedObject instanceof ManagedObjectImpl == false)
         throw new IllegalStateException("Unable to create object " + managedObject.getClass().getName());
      
      ManagedObjectImpl managedObjectImpl = (ManagedObjectImpl) managedObject;
      T object = createUnderlyingObject(managedObjectImpl, clazz);
      populateManagedObject(managedObject, object);
   }

   public void populateManagedObject(ManagedObject managedObject, T object)
   {
      if (managedObject instanceof ManagedObjectImpl == false)
         throw new IllegalStateException("Unable to populate managed object " + managedObject.getClass().getName());
      
      ManagedObjectImpl managedObjectImpl = (ManagedObjectImpl) managedObject;
      managedObjectImpl.setAttachment(object);
      populateValues(managedObjectImpl, object);
   }

   /**
    * Get the instance factory for a class
    * 
    * @param <T> the class to get an instance factory for
    * @param clazz the class
    * @return the InstanceClassFactory
    */
   @SuppressWarnings("unchecked")
   public <X extends Serializable> InstanceClassFactory<X> getInstanceClassFactory(Class<X> clazz)
   {
      synchronized (instanceFactories)
      {
         InstanceClassFactory factory = instanceFactories.get(clazz);
         if (factory != null)
            return factory;
      }
      InstanceClassFactory<X> factory = (InstanceClassFactory<X>) defaultInstanceFactory;
      return factory;
   }

   /**
    * Create the underlying object
    * 
    * @param managedObject the managed object
    * @param clazz the class
    * @return the object
    */
   protected T createUnderlyingObject(ManagedObjectImpl managedObject, Class<T> clazz)
   {
      BeanInfo beanInfo = configuration.getBeanInfo(clazz);
      try
      {
         Object result = beanInfo.newInstance();
         return clazz.cast(result);
      }
      catch (Throwable t)
      {
         throw new RuntimeException("Unable to create new object for " + managedObject + " clazz=" + clazz, t);
      }
   }

   /**
    * Populate the values
    * 
    * @param managedObject the managed object
    * @param object the object
    */
   @SuppressWarnings("unchecked")
   protected void populateValues(ManagedObjectImpl managedObject, T object)
   {
      InstanceClassFactory icf = getInstanceClassFactory(object.getClass());
      Class moClass;
      try
      {
         moClass = icf.getManagedObjectClass(object);
      }
      catch(ClassNotFoundException e)
      {
         throw new IllegalStateException(e);
      }
      BeanInfo beanInfo = configuration.getBeanInfo(moClass);

      Object componentName = null;
      Map<String, ManagedProperty> properties = managedObject.getProperties();
      if (properties != null && properties.size() > 0)
      {
         for (ManagedProperty property : properties.values())
         {
            MetaValue value = null;
            try
            {
               value = icf.getValue(beanInfo, property, object);
            }
            catch(Throwable t)
            {
               if(log.isTraceEnabled())
                  log.trace("Failed to access value for property: "+property, t);
            }

            if (value != null)
               property.setField(Fields.VALUE, value);
            /* Need to look for a ManagementObjectID at the property level which
               defines the ManagedObject id name from the property value.
             */
            Map<String, Annotation> annotations = property.getAnnotations();
            if (annotations == null)
               continue;
            ManagementObjectID id = (ManagementObjectID) annotations.get(ManagementObjectID.class.getName());
            if (id != null)
            {
               if (value == null || value.getMetaType().isSimple() == false)
               {
                  log.warn("Cannot create String name from non-Simple property: " + property + ", value=" + value);
                  continue;
               }
               SimpleValue svalue = (SimpleValue) value;
               String name = "" + svalue.getValue();
               managedObject.setName(name);
            }
            ManagementRuntimeRef runtimeRef = (ManagementRuntimeRef) annotations.get(ManagementRuntimeRef.class.getName());
            if (runtimeRef != null)
            {
               componentName = icf.getComponentName(beanInfo, property, object, value);
               if(componentName == null && defaultInstanceFactory != null)
               {
                  InstanceClassFactory dicf = defaultInstanceFactory;
                  componentName = dicf.getComponentName(beanInfo, property, object, value);
               }
            }
         }
      }
      if (componentName == null)
         componentName = icf.getComponentName(null, null, object, null);
      // set it, even if it's null
      managedObject.setComponentName(componentName);
   }

}
