/*
* JBoss, Home of Professional Open Source
* Copyright 2007, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.managed.plugins.factory;

import java.io.Serializable;
import java.lang.annotation.Annotation;
import java.lang.reflect.Constructor;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.WeakHashMap;

import org.jboss.beans.info.spi.BeanInfo;
import org.jboss.beans.info.spi.PropertyInfo;
import org.jboss.config.spi.Configuration;
import org.jboss.logging.Logger;
import org.jboss.managed.api.Fields;
import org.jboss.managed.api.ManagedObject;
import org.jboss.managed.api.ManagedOperation;
import org.jboss.managed.api.ManagedParameter;
import org.jboss.managed.api.ManagedProperty;
import org.jboss.managed.api.MutableManagedObject;
import org.jboss.managed.api.ManagedOperation.Impact;
import org.jboss.managed.api.annotation.AnnotationDefaults;
import org.jboss.managed.api.annotation.ManagementComponent;
import org.jboss.managed.api.annotation.ManagementConstants;
import org.jboss.managed.api.annotation.ManagementDeployment;
import org.jboss.managed.api.annotation.ManagementObject;
import org.jboss.managed.api.annotation.ManagementObjectID;
import org.jboss.managed.api.annotation.ManagementObjectRef;
import org.jboss.managed.api.annotation.ManagementOperation;
import org.jboss.managed.api.annotation.ManagementParameter;
import org.jboss.managed.api.annotation.ManagementProperties;
import org.jboss.managed.api.annotation.ManagementProperty;
import org.jboss.managed.api.annotation.ManagementRuntimeRef;
import org.jboss.managed.api.factory.ManagedObjectFactory;
import org.jboss.managed.plugins.DefaultFieldsImpl;
import org.jboss.managed.plugins.ManagedObjectImpl;
import org.jboss.managed.plugins.ManagedOperationImpl;
import org.jboss.managed.plugins.ManagedParameterImpl;
import org.jboss.managed.plugins.WritethroughManagedPropertyImpl;
import org.jboss.managed.spi.factory.InstanceClassFactory;
import org.jboss.managed.spi.factory.ManagedObjectBuilder;
import org.jboss.managed.spi.factory.ManagedObjectPopulator;
import org.jboss.managed.spi.factory.ManagedParameterConstraintsPopulator;
import org.jboss.managed.spi.factory.ManagedParameterConstraintsPopulatorFactory;
import org.jboss.managed.spi.factory.ManagedPropertyConstraintsPopulator;
import org.jboss.managed.spi.factory.ManagedPropertyConstraintsPopulatorFactory;
import org.jboss.metadata.spi.MetaData;
import org.jboss.metatype.api.types.ArrayMetaType;
import org.jboss.metatype.api.types.CollectionMetaType;
import org.jboss.metatype.api.types.GenericMetaType;
import org.jboss.metatype.api.types.MetaType;
import org.jboss.metatype.api.types.MetaTypeFactory;
import org.jboss.metatype.api.values.MetaValueFactory;
import org.jboss.reflect.spi.AnnotatedInfo;
import org.jboss.reflect.spi.ClassInfo;
import org.jboss.reflect.spi.MethodInfo;
import org.jboss.reflect.spi.ParameterInfo;
import org.jboss.reflect.spi.TypeInfo;

/**
 * The base ManagedObjectFactory implementation.
 * 
 * @author <a href="adrian@jboss.com">Adrian Brock</a>
 * @author Scott.Stark@jboss.org
 * @version $Revision: 81334 $
 */
public class AbstractManagedObjectFactory extends ManagedObjectFactory
   implements ManagedObjectBuilder
{
   private static final Logger log = Logger.getLogger(AbstractManagedObjectFactory.class);

   /** The configuration */
   private static final Configuration configuration = PropertyConfigurationAccess.getConfiguration();

   /** The managed object meta type */
   public static final GenericMetaType MANAGED_OBJECT_META_TYPE = new GenericMetaType(ManagedObject.class.getName(), ManagedObject.class.getName());
   
   /** The meta type factory */
   private MetaTypeFactory metaTypeFactory = MetaTypeFactory.getInstance(); 

   /** The meta value factory */
   private MetaValueFactory metaValueFactory = MetaValueFactory.getInstance();
   /** A default InstanceClassFactory used when there is no explicit ICF for a given class */
   private InstanceClassFactory<?> defaultInstanceFactory;
   /** A default ManagedObjectPopulator used when there is no explicit ManagedObjectBuilder for a given class */
   private ManagedObjectPopulator<?> defaultManagedObjectPopulator;

   /** The managed object builders */
   private Map<Class<?>, ManagedObjectBuilder> builders = new WeakHashMap<Class<?>, ManagedObjectBuilder>();

   /** The instance to class factories */
   private Map<Class<?>, InstanceClassFactory<?>> instanceFactories = new WeakHashMap<Class<?>, InstanceClassFactory<?>>();

   /**
    * Create a ManagedProperty by looking to the factory for ctor(Fields)
    * @param factory - the ManagedProperty implementation class
    * @param fields - the fields to pass to the ctor
    * @return the managed property if successful, null otherwise
    */
   public static ManagedProperty createManagedProperty(Class<? extends ManagedProperty> factory, Fields fields)
   {
      ManagedProperty property = null;
      try
      {
         Class<?>[] sig = {Fields.class};
         Constructor<? extends ManagedProperty> ctor = factory.getConstructor(sig);
         Object[] args = {fields};
         property = ctor.newInstance(args);
      }
      catch(Exception e)
      {
         log.debug("Failed to create ManagedProperty", e);
      }
      return property;
   }

   /**
    * Create an AbstractManagedObjectFactory that uses an AbstractInstanceClassFactory
    * as the defaultInstanceFactory and AbstractManagedObjectPopulator as the
    * defaultManagedObjectPopulator. The MetaTypeFactory, MetaValueFactory are
    * obtained from the respective getInstance() factory methods.
    */
   public AbstractManagedObjectFactory()
   {
      // Create an AbstractInstanceClassFactory as the default ICF
      DefaultInstanceClassFactory icf = new DefaultInstanceClassFactory();
      icf.setMof(this);
      defaultInstanceFactory = icf;
      // Create an AbstractManagedObjectPopulator as the default
      defaultManagedObjectPopulator = new AbstractManagedObjectPopulator<Serializable>(configuration, icf, instanceFactories);
   }
   /**
    * Create an AbstractManagedObjectFactory the given factories, supporting
    * information.
    * 
    * @param metaTypeFactory
    * @param metaValueFactory
    * @param defaultInstanceFactory
    * @param defaultManagedObjectPopulator
    * @param builders
    * @param instanceFactories
    */
   public AbstractManagedObjectFactory(MetaTypeFactory metaTypeFactory,
         MetaValueFactory metaValueFactory,
         InstanceClassFactory<?> defaultInstanceFactory,
         ManagedObjectPopulator<?> defaultManagedObjectPopulator,
         Map<Class<?>, ManagedObjectBuilder> builders,
         Map<Class<?>, InstanceClassFactory<?>> instanceFactories)
   {
      this.metaTypeFactory = metaTypeFactory;
      this.metaValueFactory = metaValueFactory;
      this.defaultInstanceFactory = defaultInstanceFactory;
      this.defaultManagedObjectPopulator = defaultManagedObjectPopulator;
      this.builders = builders;
      this.instanceFactories = instanceFactories;
   }

   /**
    * Get the configuration
    * 
    * @return the configuration
    */
   public Configuration getConfiguration()
   {
      return configuration;
   }

   
   public MetaTypeFactory getMetaTypeFactory()
   {
      return metaTypeFactory;
   }

   public void setMetaTypeFactory(MetaTypeFactory metaTypeFactory)
   {
      this.metaTypeFactory = metaTypeFactory;
   }

   public MetaValueFactory getMetaValueFactory()
   {
      return metaValueFactory;
   }

   public void setMetaValueFactory(MetaValueFactory metaValueFactory)
   {
      this.metaValueFactory = metaValueFactory;
   }

   public Map<Class<?>, ManagedObjectBuilder> getBuilders()
   {
      return builders;
   }

   public void setBuilders(Map<Class<?>, ManagedObjectBuilder> builders)
   {
      this.builders = builders;
   }

   public Map<Class<?>, InstanceClassFactory<?>> getInstanceFactories()
   {
      return instanceFactories;
   }

   public void setInstanceFactories(
         Map<Class<?>, InstanceClassFactory<?>> instanceFactories)
   {
      this.instanceFactories = instanceFactories;
   }

   /**
    * Get the default InstanceClassFactory
    * @return the current default InstanceClassFactory
    */
   public InstanceClassFactory<?> getDefaultInstanceFactory()
   {
      return defaultInstanceFactory;
   }
   /**
    * Set the default InstanceClassFactory. This is used when there is not
    * match an exact match by the {@linkplain #getInstanceClassFactory(Class)}
    * factory method.
    * 
    * @param defaultInstanceFactory the default InstanceClassFactory to fall
    * back to. It may be null if no default should be used.
    */
   public void setDefaultInstanceFactory(
         InstanceClassFactory<? extends Serializable> defaultInstanceFactory)
   {
      this.defaultInstanceFactory = defaultInstanceFactory;
   }

   
   public ManagedObjectPopulator<?> getDefaultManagedObjectPopulator()
   {
      return defaultManagedObjectPopulator;
   }

   /**
    * A default implementation of ManagedObjectPopulator that is used when
    * there is no ManagedObjectBuilder registered for a given type.
    * @see #getBuilder(Class)
    * @param defaultManagedObjectPopulator
    */
   public void setDefaultManagedObjectPopulator(
         ManagedObjectPopulator<? extends Serializable> defaultManagedObjectPopulator)
   {
      this.defaultManagedObjectPopulator = defaultManagedObjectPopulator;
   }

   @Override
   public <T> ManagedObject createManagedObject(Class<T> clazz, MetaData metaData)
   {
      if (clazz == null)
         throw new IllegalArgumentException("Null class");

      ManagedObject result = createSkeletonManagedObject(clazz, metaData);
      ManagedObjectPopulator<T> populator = getPopulator(clazz);
      populator.createObject(result, clazz, metaData);
      
      return result;
   }

   public ManagedObject initManagedObject(Object instance, Class<?> instanceType,
         MetaData metaData, String name, String nameType)
   {
      if (instance == null)
         throw new IllegalArgumentException("instance cannot be null");

      Class<?> clazz = instance.getClass();
      InstanceClassFactory icf = defaultInstanceFactory;
      if(instanceType != null && instanceType != clazz)
         icf = getInstanceClassFactory(instanceType, metaData);
      if(icf == defaultInstanceFactory)
         icf = getInstanceClassFactory(clazz, metaData);

      Class<Object> moClass;
      try
      {
         moClass = icf.getManagedObjectClass(instance);
      }
      catch(ClassNotFoundException e)
      {
         log.debug("Failed to load class for ManagedObject", e);
         return null;
      }
      if(moClass == null)
      {
         log.debug("ICF returned null class: "+instance);
         return null;
      }

      ManagedObject result = createSkeletonManagedObject(moClass, metaData);
      if (result == null)
      {
         log.debug("Null ManagedObject created for: "+moClass);
         return null;
      }
      log.debug("Created skeleton ManagedObject: "+result);

      if(result instanceof MutableManagedObject)
      {
         MutableManagedObject mmo = (MutableManagedObject) result;
         ManagedObjectPopulator<Object> populator = getPopulator(moClass);
         populator.populateManagedObject(mmo, instance, metaData);
      }

      return result;
   }

   @Override
   public void setBuilder(Class<?> clazz, ManagedObjectBuilder builder)
   {
      synchronized (builders)
      {
         if (builder == null)
            builders.remove(clazz);
         else
            builders.put(clazz, builder);
      }
   }

   @Override
   public <T> void setInstanceClassFactory(Class<T> clazz, InstanceClassFactory<T> factory)
   {
      synchronized (instanceFactories)
      {
         if (factory == null)
         {
            instanceFactories.remove(clazz);
            log.debug("Removed ICF for: "+clazz);
         }
         else
         {
            instanceFactories.put(clazz, factory);
            log.debug("Set ICF for: "+clazz+", to: "+factory);
         }
      }      
   }

   /**
    * Create a skeleton managed object
    * 
    * @param <T> the type
    * @param clazz the clazz
    * @return the skeleton managed object, null if clazz is not
    *    marked as a ManagementObject.
    * {@linkplain ManagementObject}
    */
   protected <T> ManagedObject createSkeletonManagedObject(Class<T> clazz, MetaData metaData)
   {
      if (clazz == null)
         throw new IllegalArgumentException("Null class");

      ManagedObjectBuilder builder = getBuilder(clazz);
      return builder.buildManagedObject(clazz, metaData);
   }
   
   /**
    * The ManagedObjectBuilder.buildManagedObject implementation. This is based
    * on the org.jboss.managed.api.annotation.* package annotations.
    * @param clazz the attachment class
    * @param metaData - the optional metadata repository accessor used to query
    * for management annotation overrides/additions to the clazz
    * @return the ManagementObject if clazz is properly annotated, null if
    *    it does not have a ManagementObject annotation on the class or metaData
    */
   @SuppressWarnings("unchecked")
   public ManagedObject buildManagedObject(Class<?> clazz, MetaData metaData)
   {
      boolean trace = log.isTraceEnabled();
      BeanInfo beanInfo = configuration.getBeanInfo(clazz);
      ClassInfo classInfo = beanInfo.getClassInfo();

      ManagementObject managementObject = getAnnotation(ManagementObject.class, classInfo, metaData);
      if( managementObject == null )
      {
         if (trace)
            log.trace("No ManagementObject annotation, skipping ManagedObject for class: "+clazz);
         // Skip the ManagedObject creation
         return null;
      }

      HashMap<String, Annotation> moAnnotations = new HashMap<String, Annotation>();
      moAnnotations.put(ManagementObject.class.getName(), managementObject);
      ManagementDeployment mnagementDeployment = getAnnotation(ManagementDeployment.class, classInfo, metaData);
      if(mnagementDeployment != null)
         moAnnotations.put(ManagementDeployment.class.getName(), mnagementDeployment);
      ManagementObjectID moID = getAnnotation(ManagementObjectID.class, classInfo, metaData);
      if (moID != null)
         moAnnotations.put(ManagementObjectID.class.getName(), moID);

      // Process the ManagementObject fields
      boolean isRuntime = managementObject.isRuntime();
      String name = classInfo.getName();
      String nameType = null;
      String attachmentName = classInfo.getName();
      Class<? extends Fields> moFieldsFactory = null;
      Class<? extends ManagedPropertyConstraintsPopulatorFactory> moConstraintsFactory = null;
      Class<? extends ManagedProperty> moPropertyFactory = null;
      if (managementObject != null)
      {
         name = managementObject.name();
         if (name.length() == 0 || name.equals(ManagementConstants.GENERATED))
            name = classInfo.getName();
         nameType = managementObject.type();
         if (nameType.length() == 0)
            nameType = null;
         attachmentName = managementObject.attachmentName();
         if (attachmentName.length() == 0)
            attachmentName = classInfo.getName();
         // Check for a component specification
         ManagementComponent mc = managementObject.componentType();
         // Work around JBMDR-51 by checking type/subtype
         // if (mc.equals(AnnotationDefaults.COMP_TYPE) == false)
         if (mc.type().length() > 0 || mc.subtype().length() > 0)
         {
            log.debug("ManagedObject("+name+") is ManagedComponent type: "+mc);
            moAnnotations.put(ManagementComponent.class.getName(), mc);
         }
         // ManagementObject level default factory classes
         moFieldsFactory = managementObject.fieldsFactory();
         moConstraintsFactory = managementObject.constraintsFactory();
         moPropertyFactory = managementObject.propertyFactory();
      }

      if (trace)
      {
         log.trace("Building MangedObject(name="+name+",nameType="+nameType
               +",attachmentName="+attachmentName+",isRuntime="+isRuntime+")");
      }

      ManagementProperties propertyType = ManagementProperties.ALL;
      Set<String> classProperties = null;
      if (managementObject != null)
      {
         propertyType = managementObject.properties();
         if(propertyType == ManagementProperties.CLASS || propertyType == ManagementProperties.CLASS_AND_EXPLICIT)
         {
            classProperties = new HashSet<String>();
            for(ManagementProperty mp : managementObject.classProperties())
            {
               if(mp.name().length() > 0)
                  classProperties.add(mp.name());
               if(mp.mappedName().length() > 0)
                  classProperties.add(mp.mappedName());
            }
         }
      }

      // Build the ManagedProperties
      Set<ManagedProperty> properties = new HashSet<ManagedProperty>();

      Set<PropertyInfo> propertyInfos = beanInfo.getProperties();
      if (propertyInfos != null && propertyInfos.isEmpty() == false)
      {
         for (PropertyInfo propertyInfo : propertyInfos)
         {
            // Ignore the "class" property
            if ("class".equals(propertyInfo.getName()))
               continue;

            ManagementProperty managementProperty = getAnnotation(ManagementProperty.class, propertyInfo, metaData);
            ManagementObjectID id = getAnnotation(ManagementObjectID.class, propertyInfo, metaData);
            ManagementObjectRef ref = getAnnotation(ManagementObjectRef.class, propertyInfo, metaData);
            ManagementRuntimeRef runtimeRef = getAnnotation(ManagementRuntimeRef.class, propertyInfo, metaData);
            HashMap<String, Annotation> propAnnotations = new HashMap<String, Annotation>();
            if (managementProperty != null)
               propAnnotations.put(ManagementProperty.class.getName(), managementProperty);
            if (id != null)
            {
               propAnnotations.put(ManagementObjectID.class.getName(), id);
               // This overrides the MO nameType
               nameType = id.type();
            }
            if (ref != null)
               propAnnotations.put(ManagementObjectRef.class.getName(), ref);
            if (runtimeRef != null)
               propAnnotations.put(ManagementRuntimeRef.class.getName(), runtimeRef);

            // Check whether this property should be included
            boolean includeProperty = false;
            switch(propertyType)
            {
               // Only if the property as a ManagementProperty
               case EXPLICIT:
                  includeProperty = managementProperty != null &&
                  (managementProperty.ignored() == false);
               break;
               // Only if the property is listed in the classProperties
               case CLASS:
                  includeProperty = classProperties.contains(propertyInfo.getName());
               break;
               // Only if the property is listed in the classProperties
               case CLASS_AND_EXPLICIT:
                  includeProperty = classProperties.contains(propertyInfo.getName())
                     || (managementProperty != null && managementProperty.ignored() == false);
               break;
               // Any property that is not ignored
               case ALL:
                  includeProperty = managementProperty == null
                     || managementProperty.ignored() == false;
               break;
            }

            if (includeProperty)
            {
               Fields fields = null;
               if (managementProperty != null)
               {
                  Class<? extends Fields> factory = moFieldsFactory;
                  if (factory == ManagementProperty.NULL_FIELDS_FACTORY.class)
                     factory = managementProperty.fieldsFactory();
                  if (factory != ManagementProperty.NULL_FIELDS_FACTORY.class)
                  {
                     try
                     {
                        fields = factory.newInstance();
                     }
                     catch (Exception e)
                     {
                        log.debug("Failed to created Fields", e);
                     }
                  }
               }
               if (fields == null)
                  fields = new DefaultFieldsImpl();

               if( propertyInfo instanceof Serializable )
               {
                  Serializable info = Serializable.class.cast(propertyInfo);
                  fields.setField(Fields.PROPERTY_INFO, info);
               }

               String propertyName = propertyInfo.getName();
               if (managementProperty != null)
                  propertyName = managementProperty.name();
               if( propertyName.length() == 0 )
                  propertyName = propertyInfo.getName();
               fields.setField(Fields.NAME, propertyName);

               // This should probably always the the propertyInfo name?
               String mappedName = propertyInfo.getName();
               if (managementProperty != null)
                  mappedName = managementProperty.mappedName();
               if( mappedName.length() == 0 )
                  mappedName = propertyInfo.getName();
               fields.setField(Fields.MAPPED_NAME, mappedName);

               String description = ManagementConstants.GENERATED;
               if (managementProperty != null)
                  description = managementProperty.description();
               if (description.equals(ManagementConstants.GENERATED))
                  description = propertyName;
               fields.setField(Fields.DESCRIPTION, description);

               if (trace)
               {
                  log.trace("Building MangedProperty(name="+propertyName
                        +",mappedName="+mappedName
                        +") ,annotations="+propAnnotations);
               }

               boolean mandatory = false;
               if (managementProperty != null)
                  mandatory = managementProperty.mandatory();
               if (mandatory)
                  fields.setField(Fields.MANDATORY, Boolean.TRUE);
               
               boolean managed = false;
               if (managementProperty != null)
                  managed = managementProperty.managed();
               
               MetaType metaType;
               if (managed)
               {
                  TypeInfo typeInfo = propertyInfo.getType();
                  if(typeInfo.isArray())
                     metaType = new ArrayMetaType(1, MANAGED_OBJECT_META_TYPE);
                  else if (typeInfo.isCollection())
                     metaType = new CollectionMetaType(typeInfo.getName(), MANAGED_OBJECT_META_TYPE);
                  else
                     metaType = MANAGED_OBJECT_META_TYPE;
               }
               else
               {
                  metaType = metaTypeFactory.resolve(propertyInfo.getType());
               }
               fields.setField(Fields.META_TYPE, metaType);
               if (propAnnotations.isEmpty() == false)
                  fields.setField(Fields.ANNOTATIONS, propAnnotations);

               // Delegate others (legal values, min/max etc.) to the constraints factory
               try
               {
                  Class<? extends ManagedPropertyConstraintsPopulatorFactory> factoryClass = moConstraintsFactory;
                  if (factoryClass == ManagementProperty.NULL_CONSTRAINTS.class)
                  {
                     if (managementProperty != null)
                        factoryClass = managementProperty.constraintsFactory();
                  }
                  ManagedPropertyConstraintsPopulatorFactory factory = factoryClass.newInstance();
                  ManagedPropertyConstraintsPopulator populator = factory.newInstance();
                  if (populator != null)
                     populator.populateManagedProperty(clazz, propertyInfo, fields);
               }
               catch(Exception e)
               {
                  log.debug("Failed to populate constraints for: "+propertyInfo, e);
               }

               ManagedProperty property = null;
               if (managementProperty != null)
               {
                  Class<? extends ManagedProperty> factory = moPropertyFactory;
                  if (factory == ManagementProperty.NULL_PROPERTY_FACTORY.class)
                     factory = managementProperty.propertyFactory();
                  if (factory != ManagementProperty.NULL_PROPERTY_FACTORY.class)
                     property = getManagedProperty(factory, fields);
               }
               // we should have write-through by default
               // use factory to change this default behavior
               if (property == null)
                  property = createDefaultManagedProperty(fields);
               properties.add(property);
            }
            else if (trace)
               log.trace("Ignoring property: " + propertyInfo);
         }
      }

      /* TODO: Operations. In general the bean metadata does not contain
         operation information.
      */
      Set<ManagedOperation> operations = new HashSet<ManagedOperation>();
      
      Set<MethodInfo> methodInfos = beanInfo.getMethods();
      if (methodInfos != null && methodInfos.isEmpty() == false)
      {
         for (MethodInfo methodInfo : methodInfos)
         {
            ManagementOperation managementOp = getAnnotation(ManagementOperation.class, methodInfo, metaData);
            if (managementOp == null)
               continue;

            ManagedOperation op = getManagedOperation(methodInfo, managementOp);
            operations.add(op);
         }
      }

      ManagedObjectImpl result = new ManagedObjectImpl(name, properties);
      result.setAnnotations(moAnnotations);
      if (nameType != null)
         result.setNameType(nameType);
      if (attachmentName != null)
         result.setAttachmentName(attachmentName);
      if (operations.size() > 0 )
         result.setOperations(operations);
      for (ManagedProperty property : properties)
         property.setManagedObject(result);
      return result;
   }

   /**
    * Create default MangedProperty instance.
    * Override this method for different default.
    *
    * @param fields the fields
    * @return new ManagedProperty instance
    */
   protected ManagedProperty createDefaultManagedProperty(Fields fields)
   {
      return new WritethroughManagedPropertyImpl(fields, metaValueFactory, this);
   }

   /**
    * Get the property name.
    *
    * @param property managed property
    * @return property name
    */
   protected String getPropertyName(ManagedProperty property)
   {
      // First look to the mapped name
      String name = property.getMappedName();
      if (name == null)
         property.getName();
      return name;
   }

   /**
    * 
    * @param methodInfo
    * @param opAnnotation
    * @return the managed operation
    */
   protected ManagedOperation getManagedOperation(MethodInfo methodInfo, ManagementOperation opAnnotation)
   {
      String name = methodInfo.getName();
      String description = opAnnotation.description();
      Impact impact = opAnnotation.impact();
      ManagementParameter[] params = opAnnotation.params();
      ParameterInfo[] paramInfo = methodInfo.getParameters();
      TypeInfo returnInfo = methodInfo.getReturnType();
      MetaType returnType = metaTypeFactory.resolve(returnInfo);
      ArrayList<ManagedParameter> mparams = new ArrayList<ManagedParameter>();
      Class<? extends ManagedParameterConstraintsPopulatorFactory> opConstraintsFactor = opAnnotation.constraintsFactory();

      if( paramInfo != null )
      {
         for(int i = 0; i < paramInfo.length; i ++)
         {
            ParameterInfo pinfo = paramInfo[i];
            String pname = pinfo.getName();
            String pdescription = null;
            ManagementParameter mpa = null;
            // Look to ManagementParameter for info
            if (i < params.length)
            {
               mpa = params[i];
               if (mpa.name().equals(AnnotationDefaults.EMPTY_STRING) == false)
                  pname = mpa.name();
               if (mpa.description().equals(AnnotationDefaults.EMPTY_STRING) == false)
                  pdescription = mpa.description();
            }
            // Generate a name if there is none
            if (pname == null)
               pname = "arg#" + i;
            Fields fields =  new DefaultFieldsImpl(pname);
            if (pdescription != null)
               fields.setField(Fields.DESCRIPTION, pdescription);
            MetaType metaType = metaTypeFactory.resolve(pinfo.getParameterType());
            fields.setField(Fields.META_TYPE, metaType);
            // Delegate others (legal values, min/max etc.) to the constraints factory
            try
            {
               Class<? extends ManagedParameterConstraintsPopulatorFactory> factoryClass = opConstraintsFactor;
               if (factoryClass == ManagementParameter.NULL_CONSTRAINTS.class)
               {
                  if (mpa != null)
                     factoryClass = mpa.constraintsFactory();
               }
               ManagedParameterConstraintsPopulatorFactory factory = factoryClass.newInstance();
               ManagedParameterConstraintsPopulator populator = factory.newInstance();
               if (populator != null)
                  populator.populateManagedParameter(name, pinfo, fields);
            }
            catch(Exception e)
            {
               log.debug("Failed to populate constraints for: "+pinfo, e);
            }

            ManagedParameterImpl mp = new ManagedParameterImpl(fields);
            mparams.add(mp);
         }
      }
      ManagedParameter[] parameters = new ManagedParameter[mparams.size()];
      mparams.toArray(parameters);

      return new ManagedOperationImpl(name, description, impact, parameters, returnType);
   }

   /**
    * Get the builder for a class
    * 
    * @param clazz the class
    * @return the builder
    */
   protected ManagedObjectBuilder getBuilder(Class<?> clazz)
   {
      synchronized (builders)
      {
         ManagedObjectBuilder builder = builders.get(clazz);
         if (builder != null)
            return builder;
      }
      return this;
   }

   /**
    * Get the instance factory for a class
    * 
    * @param clazz the class
    * @return the InstanceClassFactory
    */
   @SuppressWarnings("unchecked")
   public <X> InstanceClassFactory<X> getInstanceClassFactory(Class<X> clazz,
         MetaData metaData)
   {
      InstanceClassFactory defaultFactory = defaultInstanceFactory;
      if(metaData != null)
      {
         InstanceClassFactory mdrFactory = metaData.getMetaData(InstanceClassFactory.class);
         if(mdrFactory != null)
            defaultFactory = mdrFactory;
      }
      InstanceClassFactory<X> factory = (InstanceClassFactory<X>)
         Utility.getInstanceClassFactory(clazz, instanceFactories,
               defaultFactory);
      return factory;
   }

   /**
    * Get the populator for a class
    * 
    * @param clazz the class
    * @return the populator
    */
   @SuppressWarnings("unchecked")
   protected <X> ManagedObjectPopulator<X> getPopulator(Class<X> clazz)
   {
      ManagedObjectBuilder builder = getBuilder(clazz);
      if (builder instanceof ManagedObjectPopulator)
         return (ManagedObjectPopulator) builder;
      ManagedObjectPopulator<X> mop = (ManagedObjectPopulator<X>) defaultManagedObjectPopulator;
      return mop;
   }

   protected Collection<?> getAsCollection(Object value)
   {
      if( value.getClass().isArray() )
         return Arrays.asList(value);
      else if (value instanceof Collection)
         return Collection.class.cast(value);
      return null;
   }

   /**
    * Look for ctor(Fields)
    * @param factory - the ManagedProperty implementation class
    * @param fields - the fields to pass to the ctor
    * @return the managed property
    */
   protected ManagedProperty getManagedProperty(Class<? extends ManagedProperty> factory, Fields fields)
   {
      return createManagedProperty(factory, fields);
   }

   protected <X extends Annotation> X getAnnotation(Class<X> annotationType,
      AnnotatedInfo info, MetaData metaData)
   {
      X annotation = null;
      if(metaData != null)
      {
         annotation = metaData.getAnnotation(annotationType);
         if(annotation != null)
            log.trace("Loaded "+annotationType+" from MetaData");
      }
      if(annotation == null)
         annotation = info.getUnderlyingAnnotation(annotationType);
      return annotation;
   }
}
