/*
 * JBoss, Home of Professional Open Source
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.managed.plugins.factory;

import java.util.Map;

import org.jboss.managed.spi.factory.InstanceClassFactory;
import org.jboss.metadata.spi.MetaData;

/**
 * Common utility methods used by the factory plugins.
 * 
 * @author Scott.Stark@jboss.org
 * @version $Revision: 84862 $
 */
public class Utility
{
   /**
    * Get the instance factory for a class
    * 
    * @param clazz the class
    * @param instanceFactories - the registered mapping of classes to InstanceClassFactorys
    * @param defaultInstanceFactory - the default InstanceClassFactory to use
    *    if no class match is found.
    * @return the InstanceClassFactory
    */
   @SuppressWarnings("unchecked")
   public static InstanceClassFactory getInstanceClassFactory(Class<?> clazz,
         Map<Class<?>, InstanceClassFactory<?>> instanceFactories,
         InstanceClassFactory defaultInstanceFactory)
   {
      return getInstanceClassFactory(clazz, instanceFactories, defaultInstanceFactory, null);
   }
   /**
    * Get the instance factory for a class
    * 
    * @param clazz the class
    * @param instanceFactories - the registered mapping of classes to InstanceClassFactorys
    * @param defaultInstanceFactory - the default InstanceClassFactory to use
    *    if no class match is found.
    * @param metaData - the possibly null metdata repository accessor. Its used to
    * query for an mdr InstanceClassFactory.class as an override to the defaultInstanceFactory
    * @return the InstanceClassFactory
    */
   @SuppressWarnings("unchecked")
   public static InstanceClassFactory getInstanceClassFactory(Class<?> clazz,
         Map<Class<?>, InstanceClassFactory<?>> instanceFactories,
         InstanceClassFactory defaultInstanceFactory, MetaData metaData)
   {
      InstanceClassFactory defaultFactory = defaultInstanceFactory;
      if(metaData != null)
      {
         InstanceClassFactory mdrFactory = metaData.getMetaData(InstanceClassFactory.class);
         if(mdrFactory != null)
            defaultFactory = mdrFactory;
      }
      
      synchronized (instanceFactories)
      {
         Class<?> c = clazz;
         InstanceClassFactory factory = instanceFactories.get(c);
         while(factory == null && c != null && c != Object.class)
         {
            // First check for interfaces
            Class<?>[] ifaces = c.getInterfaces();
            if(ifaces != null)
            {
               for(Class<?> i : ifaces)
               {
                  factory = instanceFactories.get(i);
                  if(factory != null)
                     break;
               }
            }
            // Next look at superclasses
            if(factory == null)
            {
               c = c.getSuperclass();
               factory = instanceFactories.get(c);
            }
         }
         if (factory != null)
            return factory;
      }
      InstanceClassFactory factory = defaultFactory;
      return factory;
   }

}
