/*
 * JBoss, Home of Professional Open Source
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.managed.plugins.factory;

import java.lang.annotation.Annotation;
import java.util.Map;

import org.jboss.beans.info.spi.BeanInfo;
import org.jboss.config.spi.Configuration;
import org.jboss.logging.Logger;
import org.jboss.managed.api.Fields;
import org.jboss.managed.api.ManagedObject;
import org.jboss.managed.api.ManagedProperty;
import org.jboss.managed.api.MutableManagedObject;
import org.jboss.managed.api.annotation.ManagementObjectID;
import org.jboss.managed.api.annotation.ManagementRuntimeRef;
import org.jboss.managed.spi.factory.InstanceClassFactory;
import org.jboss.managed.spi.factory.ManagedObjectPopulator;
import org.jboss.metadata.spi.MetaData;
import org.jboss.metatype.api.values.MetaValue;
import org.jboss.metatype.api.values.SimpleValue;

/**
 * A default ManagedObjectPopulator implementation that relies on mc configuration
 * for obtaining BeanInfos for a class, and InstanceClassFactorys for
 * class, name and values from a value to be managed.
 * 
 * @param <T> the type to populate
 * @author Scott.Stark@jboss.org
 * @version $Revision: 85283 $
 */
public class AbstractManagedObjectPopulator<T>
   implements ManagedObjectPopulator<T>
{
   private static Logger log = Logger.getLogger(AbstractManagedObjectPopulator.class);

   /** The configuration */
   private Configuration configuration;
   private InstanceClassFactory<?> defaultInstanceFactory;
   /** The instance to class factories */
   private Map<Class<?>, InstanceClassFactory<?>> instanceFactories;


   public AbstractManagedObjectPopulator(Configuration configuration)
   {
      this(configuration, null, null);
   }
   public AbstractManagedObjectPopulator(
         Configuration configuration,
         InstanceClassFactory<?> defaultInstanceFactory,
         Map<Class<?>, InstanceClassFactory<?>> instanceFactories)
   {
      super();
      this.configuration = configuration;
      this.defaultInstanceFactory = defaultInstanceFactory;
      this.instanceFactories = instanceFactories;
   }
   
   public Configuration getConfiguration()
   {
      return configuration;
   }
   public void setConfiguration(Configuration configuration)
   {
      this.configuration = configuration;
   }

   public InstanceClassFactory<?> getDefaultInstanceFactory()
   {
      return defaultInstanceFactory;
   }
   public void setDefaultInstanceFactory(InstanceClassFactory<?> defaultInstanceFactory)
   {
      this.defaultInstanceFactory = defaultInstanceFactory;
   }

   public Map<Class<?>, InstanceClassFactory<?>> getInstanceFactories()
   {
      return instanceFactories;
   }
   public void setInstanceFactories(Map<Class<?>, InstanceClassFactory<?>> instanceFactories)
   {
      this.instanceFactories = instanceFactories;
   }

   public void createObject(ManagedObject managedObject, Class<T> clazz)
   {
      createObject(managedObject, clazz, null);
   }
   public void createObject(ManagedObject managedObject, Class<T> clazz, MetaData metaData)
   {
      if (managedObject == null)
         throw new IllegalArgumentException("Null managed object");
      
      if (managedObject instanceof MutableManagedObject == false)
         throw new IllegalStateException("Unable to create object " + managedObject.getClass().getName());
      
      MutableManagedObject managedObjectImpl = (MutableManagedObject) managedObject;
      T object = createUnderlyingObject(managedObjectImpl, clazz);
      populateManagedObject(managedObjectImpl, object, null);
   }

   public void populateManagedObject(MutableManagedObject managedObject, T object)
   {
      populateManagedObject(managedObject, object);
   }
   public void populateManagedObject(MutableManagedObject managedObject, T object,
         MetaData metaData)
   {
      managedObject.setAttachment(object);
      populateValues(managedObject, object, metaData);
   }

   /**
    * Get the instance factory for a class
    * 
    * @param <T> the class to get an instance factory for
    * @param clazz the class
    * @return the InstanceClassFactory
    */
   @SuppressWarnings("unchecked")
   public <X> InstanceClassFactory<X> getInstanceClassFactory(Class<X> clazz, MetaData metaData)
   {
      InstanceClassFactory<X> factory = (InstanceClassFactory<X>)
      Utility.getInstanceClassFactory(clazz, instanceFactories,
         defaultInstanceFactory, metaData);
      return factory;
   }

   /**
    * Create the underlying object
    * 
    * @param managedObject the managed object
    * @param clazz the class
    * @return the object
    */
   protected T createUnderlyingObject(MutableManagedObject managedObject, Class<T> clazz)
   {
      BeanInfo beanInfo = configuration.getBeanInfo(clazz);
      try
      {
         Object result = beanInfo.newInstance();
         return clazz.cast(result);
      }
      catch (Throwable t)
      {
         throw new RuntimeException("Unable to create new object for " + managedObject + " clazz=" + clazz, t);
      }
   }

   /**
    * Populate the values
    * 
    * @param managedObject the managed object
    * @param object the object
    */
   @SuppressWarnings("unchecked")
   protected void populateValues(MutableManagedObject managedObject, T object,
         MetaData metaData)
   {
      InstanceClassFactory icf = getInstanceClassFactory(object.getClass(), metaData);
      Class moClass;
      try
      {
         moClass = icf.getManagedObjectClass(object);
      }
      catch(ClassNotFoundException e)
      {
         throw new IllegalStateException(e);
      }
      
      BeanInfo beanInfo = managedObject.getTransientAttachment(BeanInfo.class);
      if(beanInfo == null)
         beanInfo = configuration.getBeanInfo(moClass);

      Object componentName = null;
      Map<String, ManagedProperty> properties = managedObject.getProperties();
      if (properties != null && properties.size() > 0)
      {
         for (ManagedProperty property : properties.values())
         {
            MetaValue value = null;
            try
            {
               value = icf.getValue(beanInfo, property, metaData, object);
            }
            catch(Throwable t)
            {
               if(log.isTraceEnabled())
                  log.trace("Failed to access value for property: "+property, t);
            }

            if (value != null)
               property.setField(Fields.VALUE, value);
            /* Need to look for a ManagementObjectID at the property level which
               defines the ManagedObject id name from the property value.
             */
            Map<String, Annotation> annotations = property.getAnnotations();
            if (annotations == null && metaData == null)
               continue;
            
            ManagementObjectID id = getAnnotation(ManagementObjectID.class, annotations, metaData);
            if (id != null)
            {
               if (value == null || value.getMetaType().isSimple() == false)
               {
                  log.warn("Cannot create String name from non-Simple property: " + property + ", value=" + value);
                  continue;
               }
               SimpleValue svalue = (SimpleValue) value;
               String name = id.prefix() + svalue.getValue() + id.suffix();
               log.debug("Created name: "+name+" from property: "+property.getName());
               managedObject.setName(name);
            }
            ManagementRuntimeRef runtimeRef = getAnnotation(ManagementRuntimeRef.class, annotations, metaData);
            if (runtimeRef != null)
            {
               componentName = icf.getComponentName(beanInfo, property, object, value);
               if(componentName == null && defaultInstanceFactory != null)
               {
                  InstanceClassFactory dicf = defaultInstanceFactory;
                  componentName = dicf.getComponentName(beanInfo, property, object, value);
               }
            }
         }
      }
      if (componentName == null)
         componentName = icf.getComponentName(null, null, object, null);
      // set it, even if it's null, but not if its already been set
      if (managedObject.getComponentName() == null)
         managedObject.setComponentName(componentName);
   }

   private static <X extends Annotation> X getAnnotation(Class<X> clazz, Map<String, Annotation> annotations,
         MetaData metaData)
   {
      X annotation = null;
      if(metaData != null)
         annotation = metaData.getAnnotation(clazz);
      if(annotation == null && annotations != null)
         annotation = (X) annotations.get(clazz.getName());
      return annotation;
   }
}
