/*
* JBoss, Home of Professional Open Source
* Copyright 2006, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.managed.api;

import java.lang.annotation.Annotation;
import java.util.Map;

import org.jboss.managed.api.annotation.ManagementRuntimeRef;

/**
 * ManagedObject is an interface for a manageable element. It
 * consists of:
 * - a name/name type for a registry/references
 * - an attachment name to associate the ManagedObject with a
 *    deployment attachment
 * - annotations from the metadata making up the ManagedObject
 * - the attachment instance
 * - the ManagedPropertys for the interface
 * - the ManagedOperations for the interface 
 * 
 * @author <a href="adrian@jboss.com">Adrian Brock</a>
 * @author Scott.Stark@jboss.org
 * @version $Revision: 86324 $
 */
public interface ManagedObject extends ManagedCommon
{
   /**
    * Get the underlying object. Note that this is only useful in the server
    * environment where the ManagedObject is associated with a deployment
    * attachment. In a client environment (admin tool for example), this is
    * most likely null.
    * 
    * @return the underlying object
    */
   Object getAttachment();

   /**
    * Get the annotations associated with the managed object
    * @return the annotations associated with the managed object
    */
   public Map<String, Annotation> getAnnotations();

   /**
    * Get the parent ManagedObject
    * @return the parent if one exists, null otherwise
    */
   public ManagedObject getParent();

   /**
    * Get a transient attachment from the property.
    * 
    * @see #setTransientAttachment(String, Object)
    * 
    * @param name the name
    * @return the attachment
    */
   Object getTransientAttachment(String name);
   
   /**
    * Get an attachment from the property,
    * uses the expected type as both the name
    * and to cast the resulting object.
    * 
    * @param <T> the expected type
    * @param expectedType the expected type
    * @return the attachment
    * @throws ClassCastException when the object is not of the expected type
    */
   <T> T getTransientAttachment(Class<T> expectedType);

   /**
    * Set a transient attachment against the property. A transient attachment
    * is one that will not be available to clients of the property, typically
    * admin tools. Such attachments are used by the server side where the
    * underlying metadata to which the property is associated is available.
    * 
    * @param name the name
    * @param attachment the attachment, pass null to remove an attachment
    * @throws IllegalArgumentException for a null name
    */
   void setTransientAttachment(String name, Object attachment);
}
