/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2010, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.mc.servlet.vdf.api;

import javax.servlet.Servlet;
import javax.servlet.ServletConfig;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;

import java.io.IOException;
import java.util.UUID;

import org.jboss.beans.metadata.spi.builder.BeanMetaDataBuilder;
import org.jboss.dependency.spi.ControllerState;
import org.jboss.kernel.spi.dependency.KernelController;
import org.jboss.kernel.spi.dependency.KernelControllerContext;
import org.jboss.mc.servlet.vdf.spi.VDFConnector;

/**
 * MC based servlet.
 *
 * @author <a href="mailto:ales.justin@jboss.org">Ales Justin</a>
 */
public class MicrocontainerServlet implements Servlet
{
   private Servlet delegate;
   private KernelController controller;
   private String name;
   private ServletConfig config;

   public void init(ServletConfig config) throws ServletException
   {
      VDFConnector<KernelController> connector = new KernelControllerVDFConnector(config.getServletContext());
      if (connector.isValid() == false)
         throw new ServletException("Invalid VDF connector: " + connector);

      String className = config.getInitParameter("servlet-class");
      if (className == null)
         throw new ServletException("Missing 'servlet-class' init parameter.");

      controller = connector.getUtility();
      name = UUID.randomUUID().toString();
      BeanMetaDataBuilder builder = BeanMetaDataBuilder.createBuilder(name, className);
      builder.ignoreDestroy(); // leave it to servlet lifecycle
      try
      {
         KernelControllerContext context = controller.install(builder.getBeanMetaData());
         if (ControllerState.INSTALLED.equals(context.getState()) == false)
         {
            //noinspection ThrowableResultOfMethodCallIgnored
            throw new IllegalArgumentException(
                  "Servlet [" + className + "] not fully installed, missing dependency? - " +
                  context.getDependencyInfo().getUnresolvedDependencies(null),
                  context.getError()
            );
         }
         Object result = context.getTarget();
         if (Servlet.class.isInstance(result) == false)
            throw new ServletException("Classname [" + className + "] doesn't map Servlet instance: " + result);

         delegate = (Servlet) result;
      }
      catch (ServletException t)
      {
         throw t;
      }
      catch (Throwable t)
      {
         throw new ServletException(t);
      }

      this.config = config;
   }

   public ServletConfig getServletConfig()
   {
      return config;
   }

   public void service(ServletRequest req, ServletResponse res) throws ServletException, IOException
   {
      delegate.service(req, res);
   }

   public String getServletInfo()
   {
      return delegate.getServletInfo();
   }

   public void destroy()
   {
      try
      {
         if (delegate != null)
         {
            delegate.destroy();
            delegate = null;
         }
      }
      finally
      {
         if (controller != null)
         {
            controller.uninstall(name);
            controller = null;
         }
      }
   }
}
