/*
* JBoss, Home of Professional Open Source
* Copyright 2006, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.beans.metadata.plugins.factory;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import javax.xml.bind.annotation.XmlAttribute;
import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlNsForm;
import javax.xml.bind.annotation.XmlRootElement;
import javax.xml.bind.annotation.XmlTransient;
import javax.xml.bind.annotation.XmlType;

import org.jboss.beans.metadata.plugins.AbstractBeanMetaData;
import org.jboss.beans.metadata.plugins.AbstractClassLoaderMetaData;
import org.jboss.beans.metadata.plugins.AbstractConstructorMetaData;
import org.jboss.beans.metadata.plugins.AbstractDependencyValueMetaData;
import org.jboss.beans.metadata.plugins.AbstractLifecycleMetaData;
import org.jboss.beans.metadata.plugins.AbstractMapMetaData;
import org.jboss.beans.metadata.plugins.AbstractParameterMetaData;
import org.jboss.beans.metadata.plugins.AbstractPropertyMetaData;
import org.jboss.beans.metadata.plugins.AbstractValueMetaData;
import org.jboss.beans.metadata.spi.ClassLoaderMetaData;
import org.jboss.beans.metadata.spi.ConstructorMetaData;
import org.jboss.beans.metadata.spi.LifecycleMetaData;
import org.jboss.beans.metadata.spi.ParameterMetaData;
import org.jboss.beans.metadata.spi.PropertyMetaData;
import org.jboss.beans.metadata.spi.ValueMetaData;
import org.jboss.kernel.plugins.bootstrap.basic.KernelConstants;
import org.jboss.kernel.spi.config.KernelConfigurator;
import org.jboss.xb.annotations.JBossXmlSchema;

/**
 * GenericBeanFactoryMetaData.
 * 
 * @author <a href="ales.justin@jboss.com">Ales Justin</a>
 * @author <a href="adrian@jboss.com">Adrian Brock</a>
 * @version $Revision: 68642 $
 */
@JBossXmlSchema(namespace="urn:jboss:bean-deployer:2.0", elementFormDefault= XmlNsForm.QUALIFIED)
@XmlRootElement(name="beanfactory")
@XmlType(propOrder={"aliases", "annotations", "classLoader", "constructor", "properties", "create", "start", "depends", "demands", "supplies", "installs", "uninstalls"})
public class GenericBeanFactoryMetaData extends AbstractBeanMetaData
{
   private static final long serialVersionUID = 2L;

   /**
    * Create a new GenericBeanFactoryMetaData.
    */
   public GenericBeanFactoryMetaData()
   {
      setBean(GenericBeanFactory.class.getName());
      AbstractConstructorMetaData constructor = new AbstractConstructorMetaData();
      ArrayList<ParameterMetaData> parameters = new ArrayList<ParameterMetaData>();
      parameters.add(new AbstractParameterMetaData(KernelConfigurator.class.getName(), new AbstractDependencyValueMetaData(KernelConstants.KERNEL_CONFIGURATOR_NAME)));
      constructor.setParameters(parameters);
      setConstructor(constructor);
      setProperties(new HashSet<PropertyMetaData>());
   }
   
   /**
    * Create a new GenericBeanFactoryMetaData.
    * 
    * @param name the name
    */
   public GenericBeanFactoryMetaData(String name)
   {
      this();
      setName(name);
   }
   
   /**
    * Create a new GenericBeanFactoryMetaData.
    * 
    * @param name the name
    * @param bean the bean class name
    */
   public GenericBeanFactoryMetaData(String name, String bean)
   {
      this(name);
      setBeanClass(bean);
   }

   /**
    * Get the class of the bean created
    * 
    * @return the class
    */
   public String getBeanClass()
   {
      AbstractPropertyMetaData prop = (AbstractPropertyMetaData) getProperty("bean");
      if (prop != null)
      {
         AbstractValueMetaData value = (AbstractValueMetaData) prop.getValue();
         if (value != null)
            return (String) value.getUnderlyingValue();
      }
      return null;
   }

   /**
    * Set the class of the bean created
    * 
    * @param beanClass the class
    */
   @XmlAttribute(name="class")
   public void setBeanClass(String beanClass)
   {
      Set<PropertyMetaData> properties = getProperties();
      properties.add(new AbstractPropertyMetaData("bean", new AbstractValueMetaData(beanClass)));
   }

   @XmlElement(name="classloader", type=AbstractClassLoaderMetaData.class)
   public void setClassLoader(ClassLoaderMetaData classLoader)
   {
      super.setClassLoader(classLoader);
      Set<PropertyMetaData> properties = getProperties();
      properties.add(new AbstractPropertyMetaData("classLoader", new AbstractValueMetaData(classLoader)));
   }

   /**
    * Set the bean constructor
    * 
    * @param constructor the constructor
    */
   @XmlElement(name="constructor", type=AbstractConstructorMetaData.class)
   public void setBeanConstructor(ConstructorMetaData constructor)
   {
      Set<PropertyMetaData> properties = getProperties();
      properties.add(new AbstractPropertyMetaData("constructor", new AbstractValueMetaData(constructor)));
   }
   
   /**
    * Get a bean property
    * 
    * @param name the name
    * @return the value metadata
    */
   public ValueMetaData getBeanProperty(String name)
   {
      if (name == null)
         throw new IllegalArgumentException("Null name");
      PropertyMetaData properties = getProperty("properties");
      if (properties == null)
         return null;
      AbstractMapMetaData map = (AbstractMapMetaData) properties.getValue();
      for (Iterator i = map.entrySet().iterator(); i.hasNext();)
      {
         Map.Entry entry = (Map.Entry) i.next();
         ValueMetaData key = (ValueMetaData) entry.getKey();
         if (key.getUnderlyingValue().equals(name))
         {
            ValueMetaData vmd = (ValueMetaData) entry.getValue();
            return (ValueMetaData) vmd.getUnderlyingValue();
         }
      }
      return null;
   }
   
   /**
    * Add a bean property
    * 
    * @param property the property
    */
   public void addBeanProperty(PropertyMetaData property)
   {
      PropertyMetaData properties = getProperty("properties"); 

      AbstractMapMetaData map;
      if (properties == null)
      {
         map = new AbstractMapMetaData();
         properties = new AbstractPropertyMetaData("properties", map);
         addProperty(properties);
      }
      else
      {
         map = (AbstractMapMetaData) properties.getValue(); 
      }
      
      ValueMetaData valueMetaData = property.getValue();
      valueMetaData = new AbstractValueMetaData(valueMetaData);
      map.put(new AbstractValueMetaData(property.getName()), valueMetaData);
   }

   @XmlElement(name="create", type= AbstractLifecycleMetaData.class)
   public void setBeanCreate(LifecycleMetaData lifecycle)
   {
      Set<PropertyMetaData> properties = getProperties();
      properties.add(new AbstractPropertyMetaData("create", new AbstractValueMetaData(lifecycle)));
   }

   @XmlElement(name="start", type=AbstractLifecycleMetaData.class)
   public void setBeanStart(LifecycleMetaData lifecycle)
   {
      Set<PropertyMetaData> properties = getProperties();
      properties.add(new AbstractPropertyMetaData("start", new AbstractValueMetaData(lifecycle)));
   }

   @XmlTransient
   public void setBean(String bean)
   {
      super.setBean(bean);
   }
}
