package org.jboss.resteasy.test.rx;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;

import jakarta.ws.rs.client.ClientBuilder;

import org.jboss.arquillian.container.test.api.Deployment;
import org.jboss.arquillian.container.test.api.RunAsClient;
import org.jboss.arquillian.junit5.ArquillianExtension;
import org.jboss.resteasy.client.jaxrs.ResteasyClient;
import org.jboss.resteasy.client.jaxrs.internal.CompletionStageRxInvokerProvider;
import org.jboss.resteasy.test.rx.resource.RxCompletionStageResourceImpl;
import org.jboss.resteasy.test.rx.resource.RxScheduledExecutorService;
import org.jboss.resteasy.test.rx.resource.SimpleResource;
import org.jboss.resteasy.test.rx.resource.TestException;
import org.jboss.resteasy.test.rx.resource.TestExceptionMapper;
import org.jboss.resteasy.test.rx.resource.Thing;
import org.jboss.resteasy.utils.PortProviderUtil;
import org.jboss.resteasy.utils.TestUtil;
import org.jboss.shrinkwrap.api.Archive;
import org.jboss.shrinkwrap.api.asset.StringAsset;
import org.jboss.shrinkwrap.api.spec.WebArchive;
import org.junit.jupiter.api.AfterAll;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;

/**
 * @tpSubChapter Reactive classes
 * @tpChapter Integration tests
 * @tpSince RESTEasy 4.0
 *
 *          These tests run synchronously on client, calling a proxy which does a synchronous invocation.
 *          The server creates and returns CompletionStages which run asynchronously.
 */
@ExtendWith(ArquillianExtension.class)
@RunAsClient
public class RxCompletionStageProxyServerAsyncTest {

    private static ResteasyClient client;
    private static SimpleResource proxy;

    private static List<Thing> xThingList = new ArrayList<Thing>();
    private static List<Thing> aThingList = new ArrayList<Thing>();

    static {
        for (int i = 0; i < 3; i++) {
            xThingList.add(new Thing("x"));
        }
        for (int i = 0; i < 3; i++) {
            aThingList.add(new Thing("a"));
        }
    }

    @Deployment
    public static Archive<?> deploy() {
        WebArchive war = TestUtil.prepareArchive(RxCompletionStageProxyServerAsyncTest.class.getSimpleName());
        war.addClass(Thing.class);
        war.addClass(RxScheduledExecutorService.class);
        war.addClass(TestException.class);
        war.setManifest(new StringAsset("Manifest-Version: 1.0\n"
                + "Dependencies: org.jboss.resteasy.resteasy-rxjava2 services\n"));
        return TestUtil.finishContainerPrepare(war, null, RxCompletionStageResourceImpl.class, TestExceptionMapper.class);
    }

    private static String generateURL(String path) {
        return PortProviderUtil.generateURL(path, RxCompletionStageProxyServerAsyncTest.class.getSimpleName());
    }

    //////////////////////////////////////////////////////////////////////////////
    @BeforeAll
    public static void beforeClass() throws Exception {
        client = (ResteasyClient) ClientBuilder.newClient();
        proxy = client.target(generateURL("/")).proxy(SimpleResource.class);
    }

    @AfterAll
    public static void after() throws Exception {
        client.close();
    }

    //////////////////////////////////////////////////////////////////////////////

    @Test
    public void testGet() throws Exception {
        String s = proxy.get();
        Assertions.assertEquals("x", s);
    }

    @Test
    public void testGetThing() throws Exception {
        Thing t = proxy.getThing();
        Assertions.assertEquals(new Thing("x"), t);
    }

    @Test
    public void testGetThingList() throws Exception {
        List<Thing> list = proxy.getThingList();
        Assertions.assertEquals(xThingList, list);
    }

    @Test
    public void testPut() throws Exception {
        String s = proxy.put("a");
        Assertions.assertEquals("a", s);
    }

    @Test
    public void testPutThing() throws Exception {
        Thing t = proxy.putThing("a");
        Assertions.assertEquals(new Thing("a"), t);
    }

    @Test
    public void testPutThingList() throws Exception {
        List<Thing> list = proxy.putThingList("a");
        Assertions.assertEquals(aThingList, list);
    }

    @Test
    public void testPost() throws Exception {
        String s = proxy.post("a");
        Assertions.assertEquals("a", s);
    }

    @Test
    public void testPostThing() throws Exception {
        Thing t = proxy.postThing("a");
        Assertions.assertEquals(new Thing("a"), t);
    }

    @Test
    public void testPostThingList() throws Exception {
        List<Thing> list = proxy.postThingList("a");
        Assertions.assertEquals(aThingList, list);
    }

    @Test
    public void testDelete() throws Exception {
        String s = proxy.delete();
        Assertions.assertEquals("x", s);
    }

    @Test
    public void testDeleteThing() throws Exception {
        Thing t = proxy.deleteThing();
        Assertions.assertEquals(new Thing("x"), t);
    }

    @Test
    public void testDeleteThingList() throws Exception {
        List<Thing> list = proxy.deleteThingList();
        Assertions.assertEquals(xThingList, list);
    }

    @Test
    public void testHead() throws Exception {
        try {
            proxy.head();
        } catch (Exception e) {
            Assertions.assertTrue(throwableContains(e, "Input stream was empty, there is no entity"));
        }
    }

    @Test
    public void testOptions() throws Exception {
        String s = proxy.options();
        Assertions.assertEquals("x", s);
    }

    @Test
    public void testOptionsThing() throws Exception {
        Thing t = proxy.optionsThing();
        Assertions.assertEquals(new Thing("x"), t);
    }

    @Test
    public void testOptionsThingList() throws Exception {
        List<Thing> list = proxy.optionsThingList();
        Assertions.assertEquals(xThingList, list);
    }

    @Test
    public void testUnhandledException() throws Exception {
        try {
            proxy.exceptionUnhandled();
            Assertions.fail("expecting Exception");
        } catch (Exception e) {
            Assertions.assertTrue(e.getMessage().contains("500"));
        }
    }

    @Test
    public void testHandledException() throws Exception {
        try {
            proxy.exceptionHandled();
            Assertions.fail("expecting Exception");
        } catch (Exception e) {
            Assertions.assertTrue(e.getMessage().contains("444"));
        }
    }

    @Test
    public void testGetTwoClients() throws Exception {
        CopyOnWriteArrayList<String> list = new CopyOnWriteArrayList<String>();

        ResteasyClient client1 = (ResteasyClient) ClientBuilder.newClient();
        client1.register(CompletionStageRxInvokerProvider.class);
        SimpleResource proxy1 = client1.target(generateURL("/")).proxy(SimpleResource.class);
        String s1 = proxy1.get();

        ResteasyClient client2 = (ResteasyClient) ClientBuilder.newClient();
        client2.register(CompletionStageRxInvokerProvider.class);
        SimpleResource proxy2 = client2.target(generateURL("/")).proxy(SimpleResource.class);
        String s2 = proxy2.get();

        list.add(s1);
        list.add(s2);
        Assertions.assertEquals(2, list.size());
        for (int i = 0; i < 2; i++) {
            Assertions.assertEquals("x", list.get(i));
        }
        client1.close();
        client2.close();
    }

    @Test
    public void testGetTwoProxies() throws Exception {
        CopyOnWriteArrayList<String> list = new CopyOnWriteArrayList<String>();

        SimpleResource proxy1 = client.target(generateURL("/")).proxy(SimpleResource.class);
        String s1 = proxy1.get();

        SimpleResource proxy2 = client.target(generateURL("/")).proxy(SimpleResource.class);
        String s2 = proxy2.get();

        list.add(s1);
        list.add(s2);
        Assertions.assertEquals(2, list.size());
        for (int i = 0; i < 2; i++) {
            Assertions.assertEquals("x", list.get(i));
        }
    }

    @Test
    public void testGetTwoCompletionStages() throws Exception {
        CopyOnWriteArrayList<String> list = new CopyOnWriteArrayList<String>();

        String s1 = proxy.get();
        String s2 = proxy.get();

        list.add(s1);
        list.add(s2);
        Assertions.assertEquals(2, list.size());
        for (int i = 0; i < 2; i++) {
            Assertions.assertEquals("x", list.get(i));
        }
    }

    private static boolean throwableContains(Throwable t, String s) {
        while (t != null) {
            if (t.getMessage().contains(s)) {
                return true;
            }
            t = t.getCause();
        }
        return false;
    }
}
