/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2010, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.scanning.plugins.visitor;

import java.lang.annotation.Annotation;
import java.lang.annotation.ElementType;
import java.net.URL;

import org.jboss.classloading.spi.visitor.ResourceContext;
import org.jboss.metadata.spi.signature.ConstructorParametersSignature;
import org.jboss.metadata.spi.signature.MethodParametersSignature;
import org.jboss.metadata.spi.signature.Signature;
import org.jboss.reflect.spi.*;
import org.jboss.scanning.plugins.helpers.ResourceOwnerFinder;

/**
 * Class hierarchy resource visitor.
 *
 * @author <a href="mailto:ales.justin@jboss.org">Ales Justin</a>
 */
public abstract class ClassHierarchyResourceVisitor extends ReflectResourceVisitor
{
   private ResourceOwnerFinder finder;
   private boolean checkInterfaces;
   private boolean checkSuper;

   protected ClassHierarchyResourceVisitor(ReflectProvider provider, ResourceOwnerFinder finder)
   {
      super(provider);
      if (finder == null)
         throw new IllegalArgumentException("Null finder");
      this.finder = finder;
   }

   /**
    * Is this class info relevant for scanning.
    *
    * @param classInfo the class info
    * @return true if relevant, false otherwise
    */
   protected abstract boolean isRelevant(ClassInfo classInfo);

   protected void handleClass(ResourceContext resource, ClassInfo classInfo) throws Exception
   {
      if (classInfo == null || isRelevant(classInfo) == false)
         return;

      String className = classInfo.getName();
      if (log.isTraceEnabled())
         log.trace("Scanning class " + className + " for annotations");

      URL ownerURL = finder.findOwnerURL(resource);

      Annotation[] annotations = classInfo.getUnderlyingAnnotations();
      handleAnnotations(ElementType.TYPE, (Signature)null, annotations, className, ownerURL);

      handleMembers(ElementType.CONSTRUCTOR, classInfo.getDeclaredConstructors(), className, ownerURL);
      handleMembers(ElementType.METHOD, classInfo.getDeclaredMethods(), className, ownerURL);
      handleMembers(ElementType.FIELD, classInfo.getDeclaredFields(), className, ownerURL);            

      if (checkInterfaces || checkSuper)
      {
         if (checkInterfaces)
         {
            // interfaces
            ClassInfo[] interfaces = classInfo.getInterfaces();
            if (interfaces != null && interfaces.length > 0)
            {
               for (ClassInfo intf : interfaces)
                  handleClass(intf);
            }
         }

         if (checkSuper)
         {
            // super class
            handleClass(classInfo.getSuperclass());
         }
      }
   }

   /**
    * Handle members for annotations.
    *
    * @param type      where we found the annotations
    * @param members   the member instances
    * @param className the className
    * @param ownerURL    the class owner url
    * @throws Exception for any annotations lookup problems
    */
   protected void handleMembers(ElementType type, AnnotatedInfo[] members, String className, URL ownerURL) throws Exception
   {
      if (members != null && members.length > 0)
      {
         for (AnnotatedInfo ainfo : members)
         {
            if (ainfo instanceof MemberInfo == false)
               throw new IllegalArgumentException("Can only handle member info: " + ainfo);

            Annotation[] annotations = ainfo.getUnderlyingAnnotations();
            MemberInfo member = MemberInfo.class.cast(ainfo);
            handleAnnotations(type, member, annotations, className, ownerURL);
            if (isParametrized(ainfo))
            {
               Annotation[][] paramAnnotations = getParameterAnnotations(member);
               for (int index = 0; index < paramAnnotations.length; index++)
               {
                  Signature signature = getParameterSignature(member, index);
                  handleAnnotations(ElementType.PARAMETER, signature, paramAnnotations[index], className, ownerURL);
               }
            }
         }
      }
   }

   protected boolean isParametrized(AnnotatedInfo member)
   {
      return member instanceof MethodInfo || member instanceof ConstructorInfo;
   }

   protected Annotation[][] getParameterAnnotations(MemberInfo info)
   {
      ParameterInfo[] pinfos;
      if (info instanceof ConstructorInfo)
      {
         ConstructorInfo ci = ConstructorInfo.class.cast(info);
         pinfos = ci.getParameters();
      }
      else if (info instanceof MethodInfo)
      {
         MethodInfo mi = MethodInfo.class.cast(info);
         pinfos = mi.getParameters();
      }
      else
      {
         throw new IllegalArgumentException("Cannot handle info: " + info);
      }

      Annotation[][] values = new Annotation[pinfos.length][];
      for (int i = 0; i < pinfos.length; i++)
      {
         ParameterInfo pi = pinfos[i];
         values[i] = pi.getUnderlyingAnnotations();
      }
      return values;
   }

   protected Signature getParameterSignature(MemberInfo info, int index)
   {
      if (info instanceof ConstructorInfo)
      {
         ConstructorInfo ci = ConstructorInfo.class.cast(info);
         return new ConstructorParametersSignature(ci, index);
      }
      else if (info instanceof MethodInfo)
      {
         MethodInfo mi = MethodInfo.class.cast(info);
         return new MethodParametersSignature(mi, index);
      }
      else
      {
         throw new IllegalArgumentException("Cannot handle info: " + info);
      }
   }

   /**
    * Handle annotations.
    *
    * @param type        where we found the annotations
    * @param member      the member
    * @param annotations the actual annotations
    * @param className   the className
    * @param ownerURL    the class owner url
    * @throws Exception for any annotations lookup problems
    */
   protected void handleAnnotations(ElementType type, MemberInfo member, Annotation[] annotations, String className, URL ownerURL) throws Exception
   {
      Signature signature = null;
      if (member != null)
         signature = Signature.getSignature(member);

      handleAnnotations(type, signature, annotations, className, ownerURL);
   }

   /**
    * Handle annotations.
    *
    * @param type        where we found the annotations
    * @param signature   the signature
    * @param annotations the actual annotations
    * @param className   the className
    * @param ownerURL    the class owner url
    */
   protected void handleAnnotations(ElementType type, Signature signature, Annotation[] annotations, String className, URL ownerURL)
   {
   }

   public void setCheckInterfaces(boolean checkInterfaces)
   {
      this.checkInterfaces = checkInterfaces;
   }

   public void setCheckSuper(boolean checkSuper)
   {
      this.checkSuper = checkSuper;
   }
}