/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2010, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.scanning.plugins;

import java.net.URL;
import java.util.Set;

import org.jboss.classloading.plugins.visitor.FederatedResourceVisitor;
import org.jboss.classloading.spi.dependency.Module;
import org.jboss.classloading.spi.visitor.ResourceFilter;
import org.jboss.deployers.spi.deployer.helpers.AttachmentLocator;
import org.jboss.deployers.structure.spi.DeploymentUnit;
import org.jboss.scanning.plugins.helpers.DelegateResourceFilter;
import org.jboss.scanning.plugins.helpers.VoidScanningHandle;
import org.jboss.scanning.spi.ScanningHandle;
import org.jboss.scanning.spi.ScanningPlugin;

/**
 * Per deployment unit scanner.
 *
 * @author <a href="mailto:ales.justin@jboss.org">Ales Justin</a>
 */
public class DeploymentUnitScanner extends VFSUrlScanner
{
   /** The deployment unit */
   private final DeploymentUnit unit;

   public DeploymentUnitScanner(DeploymentUnit unit, URL... roots)
   {
      super(roots);
      if (unit == null)
         throw new IllegalArgumentException("Null unit");
      this.unit = unit;
   }

   public boolean doScan()
   {
      // we actually don't care here
      // since it's the job of Module to determine what to scan
      // e.g. not all Module impls are VFS based
      return true;
   }

   /**
    * Wrap federated filters with existing.
    * e.g. scanning metadata restrictions
    *
    * @param name the attachment name
    * @param filter the existing federated filter
    * @return filter
    */
   protected ResourceFilter wrap(String name, ResourceFilter filter)
   {
      ResourceFilter ext = unit.getAttachment(name, ResourceFilter.class);
      if (ext != null)
      {
         if (filter != null)
            return new DelegateResourceFilter(ext, filter);
         else
            return ext;
      }
      return filter;
   }

   protected void scan(Set<ScanningPlugin> plugins) throws Exception
   {
      Module module = AttachmentLocator.searchAncestors(unit, Module.class);
      if (module == null)
         throw new IllegalArgumentException("No module found: " + unit);

      ScanningPlugin[] array = plugins.toArray(new ScanningPlugin[plugins.size()]);
      FederatedResourceVisitor visitor = new FederatedResourceVisitor(array, array, getRecurseFilters(array));
      ResourceFilter filter = visitor.getFilter();
      ResourceFilter recurseFilter = visitor.getRecurseFilter();

      filter = wrap(ResourceFilter.class.getName() + ".resource", filter);
      recurseFilter = wrap(ResourceFilter.class.getName() + ".recurse", recurseFilter);

      // by default we don't use VFS,
      // since we already expect it to be VFS based
      URL[] urls;
      if (isToVFS())
         urls = toVFS(getRoots());
      else
         urls = getRoots();

      // something in javassist uses TCL
      ClassLoader tccl = SecurityActions.setContextClassLoader(unit.getClassLoader());
      try
      {
         module.visit(visitor, filter, recurseFilter, urls);
      }
      finally
      {
         SecurityActions.resetContextClassLoader(tccl);
      }
   }

   @SuppressWarnings("unchecked")
   protected ScanningHandle createHandle(ScanningPlugin plugin)
   {
      ScanningHandle handle = plugin.createHandle();
      if (handle != VoidScanningHandle.INSTANCE) // ignore helper dummy/void handle
         unit.addAttachment(plugin.getAttachmentKey(), handle, plugin.getHandleInterface());
      return handle;
   }
}