/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2010, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.scanning.annotations.plugins;

import java.io.Externalizable;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;
import java.lang.annotation.Annotation;
import java.lang.annotation.ElementType;
import java.lang.reflect.AnnotatedElement;
import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.net.URL;
import java.util.*;

import org.jboss.metadata.spi.signature.Signature;
import org.jboss.scanning.annotations.spi.Element;
import org.jboss.scanning.spi.ScanningHandle;
import org.jboss.util.collection.CollectionsFactory;

/**
 * DefaultAnnotationEnvironment.
 *
 * @author <a href="mailto:ales.justin@jboss.com">Ales Justin</a>
 */
public class DefaultAnnotationRepository extends MutableAnnotationRepository implements Externalizable, ScanningHandle<DefaultAnnotationRepository>
{
   /** The serial version UID */
   private static final long serialVersionUID = 1L;
   /** The info map */
   private Map<String, Map<Class<? extends Annotation>, Map<ElementType, Set<ClassSignaturePair>>>> env;
   /** The checked class names */
   private transient Set<String> checkedClassNames = new HashSet<String>();
   /** Should we keep the annotation */
   private boolean keepAnnotations;

   /**
    * Should only be used for de-serialization.
    */
   public DefaultAnnotationRepository()
   {
      super();
   }

   public DefaultAnnotationRepository(ClassLoader classLoader)
   {
      super(classLoader);
      env = new HashMap<String, Map<Class<? extends Annotation>, Map<ElementType, Set<ClassSignaturePair>>>>();
   }

   @Override
   protected ClassLoader getClassLoader()
   {
      return super.getClassLoader();
   }

   public void writeExternal(ObjectOutput out) throws IOException
   {
      out.writeObject(env);
   }

   @SuppressWarnings("unchecked")
   public void readExternal(ObjectInput in) throws IOException, ClassNotFoundException
   {
      env = (Map)in.readObject();
   }

   /**
    * Set the keep annotations flag.
    *
    * @param keepAnnotations the keep annotations flag
    */
   public void setKeepAnnotations(boolean keepAnnotations)
   {
      this.keepAnnotations = keepAnnotations;
   }

   /**
    * Cleanup.
    */
   void cleanup()
   {
      env.clear();
      checkedClassNames.clear();
   }

   /**
    * Get env map.
    *
    * @return the env map
    */
   protected Map<String, Map<Class<? extends Annotation>, Map<ElementType, Set<ClassSignaturePair>>>> getEnv()
   {
      if (env == null)
         throw new IllegalArgumentException("Null env, previously serialized?");

      return env;
   }

   public void merge(DefaultAnnotationRepository subHandle)
   {
      Map<String, Map<Class<? extends Annotation>, Map<ElementType, Set<ClassSignaturePair>>>> env = subHandle.getEnv();
      for (Map.Entry<String, Map<Class<? extends Annotation>, Map<ElementType, Set<ClassSignaturePair>>>> entry : env.entrySet())
      {
         Map<Class<? extends Annotation>, Map<ElementType, Set<ClassSignaturePair>>> etMap = entry.getValue();
         for (Map.Entry<Class<? extends Annotation>, Map<ElementType, Set<ClassSignaturePair>>> et : etMap.entrySet())
         {
            Map<ElementType, Set<ClassSignaturePair>> map = et.getValue();
            for (Map.Entry<ElementType, Set<ClassSignaturePair>> mapE : map.entrySet())
            {
               for (ClassSignaturePair csp : mapE.getValue())
               {
                  putAnnotation(csp.getAnnotation(), et.getKey(), mapE.getKey(), csp.getClassName(), csp.getSignature(), entry.getKey());
               }
            }
         }
      }
   }

   /**
    * Was class name already checked.
    *
    * @param className the class name
    * @return true if already checked, false otherwise
    */
   boolean isAlreadyChecked(String className)
   {
      return checkedClassNames.contains(className);
   }

   /**
    * Put the annotation info.
    *
    * @param annotation the annotation
    * @param type the annotation type
    * @param className the class name
    * @param signature the signature
    * @param ownerURL the class owner url
    */
   void putAnnotation(Annotation annotation, ElementType type, String className, Signature signature, URL ownerURL)
   {
      putAnnotation(annotation, annotation.annotationType(), type, className, signature, ownerURL.getPath());
   }

   /**
    * Put the annotation info.
    *
    * @param annotation the annotation
    * @param annClass the annotation class
    * @param type the annotation type
    * @param className the class name
    * @param signature the signature
    * @param path the owner url path
    */
   void putAnnotation(Annotation annotation, Class<? extends Annotation> annClass, ElementType type, String className, Signature signature, String path)
   {
      if (log.isTraceEnabled())
         log.trace("Adding annotation @" + annClass.getSimpleName() + " for " + className + " at type " + type + ", signature: " + signature);

      // add to checked
      checkedClassNames.add(className);

      Map<String, Map<Class<? extends Annotation>, Map<ElementType, Set<ClassSignaturePair>>>> env = getEnv();

      Map<Class<? extends Annotation>, Map<ElementType, Set<ClassSignaturePair>>> map = env.get(path);
      if (map == null)
      {
         map = new HashMap<Class<? extends Annotation>, Map<ElementType, Set<ClassSignaturePair>>>();
         env.put(path, map);
      }

      Map<ElementType, Set<ClassSignaturePair>> elements = map.get(annClass);
      if (elements == null)
      {
         elements = new HashMap<ElementType, Set<ClassSignaturePair>>();
         map.put(annClass, elements);
      }

      Set<ClassSignaturePair> classes = elements.get(type);
      if (classes == null)
      {
         classes = CollectionsFactory.createLazySet();
         elements.put(type, classes);
      }

      ClassSignaturePair pair;
      if (keepAnnotations)
         pair = new ClassSignaturePair(className, signature, annotation);
      else
         pair = new ClassSignaturePair(className, signature);
      classes.add(pair);
   }

   /**
    * Get matching cs pairs.
    *
    * @param path the owner path
    * @param annClass the annotation class
    * @param type the annotation type
    * @return class names
    */
   protected Set<ClassSignaturePair> getCSPairs(String path, Class<? extends Annotation> annClass, ElementType type)
   {
      Set<ClassSignaturePair> pairs = null;

      if (path == null)
      {
         pairs = new HashSet<ClassSignaturePair>();
         for (Map<Class<? extends Annotation>, Map<ElementType, Set<ClassSignaturePair>>> value : getEnv().values())
         {
            Map<ElementType, Set<ClassSignaturePair>> elements = value.get(annClass);
            if (elements != null)
            {
               Set<ClassSignaturePair> csps = elements.get(type);
               if (csps != null)
                  pairs.addAll(csps);
            }
         }
      }
      else
      {
         Map<Class<? extends Annotation>, Map<ElementType, Set<ClassSignaturePair>>> map = getEnv().get(path);
         if (map != null)
         {
            Map<ElementType, Set<ClassSignaturePair>> elements = map.get(annClass);
            if (elements != null)
               pairs = elements.get(type);
         }
      }

      return (pairs != null) ? pairs : Collections.<ClassSignaturePair>emptySet();
   }

   /**
    * Transform class names into classes.
    *
    * @param <A> the annotation type
    * @param <M> the annotated element type
    * @param path the classpath entry
    * @param type the annotation type
    * @param annClass the annotation class
    * @param aoClass the ao class
    * @return classes
    */
   protected <A extends Annotation, M extends AnnotatedElement> Set<Element<A, M>> transformToElements(
         String path,
         ElementType type,
         Class<A> annClass,
         Class<M> aoClass
   )
   {
      Set<ClassSignaturePair> pairs = getCSPairs(path, annClass, type);
      if (pairs.isEmpty())
         return Collections.emptySet();

      ClassLoader classLoader = getClassLoader();
      Set<Element<A, M>> elements = new HashSet<Element<A, M>>();
      for (ClassSignaturePair pair : pairs)
      {
         String className = pair.getClassName();
         A annotation = annClass.cast(pair.getAnnotation());

         Element<A, M> element;
         if (type == ElementType.TYPE)
            element = new ClassElement<A, M>(classLoader, className, annClass, annotation);
         else if (type == ElementType.PARAMETER)
            element = new ParametersElement<A,M>(classLoader, className, pair.getSignature(), annClass, annotation, aoClass);
         else
            element = new DefaultElement<A,M>(classLoader, className, pair.getSignature(), annClass, annotation, aoClass);
         elements.add(element);
      }
      return elements;
   }

   public <A extends Annotation> Set<Element<A, AnnotatedElement>> getAnnotatedClasses(URL url, Class<A> annotation, ElementType type)
   {
      String path = url != null ? url.getPath() : null;

      if (type == null)
      {
         Set<Element<A, AnnotatedElement>> result = new HashSet<Element<A, AnnotatedElement>>();
         result.addAll(transformToElements(path, ElementType.TYPE, annotation, AnnotatedElement.class));
         result.addAll(transformToElements(path, ElementType.CONSTRUCTOR, annotation, AnnotatedElement.class));
         result.addAll(transformToElements(path, ElementType.METHOD, annotation, AnnotatedElement.class));
         result.addAll(transformToElements(path, ElementType.FIELD, annotation, AnnotatedElement.class));
         result.addAll(transformToElements(path, ElementType.PARAMETER, annotation, AnnotatedElement.class));
         return result;
      }
      else
      {
         return transformToElements(path, type, annotation, AnnotatedElement.class);
      }
   }

   public boolean hasClassAnnotatedWith(Class<? extends Annotation> annotation)
   {
      return getCSPairs(null, annotation, ElementType.TYPE).isEmpty() == false;
   }

   @SuppressWarnings("unchecked")
   public <A extends Annotation> Set<Element<A, Class<?>>> classIsAnnotatedWith(Class<A> annotation)
   {
      return (Set) transformToElements(null, ElementType.TYPE, annotation, Class.class);
   }

   @SuppressWarnings("unchecked")
   public <A extends Annotation> Set<Element<A, Constructor<?>>> classHasConstructorAnnotatedWith(Class<A> annotation)
   {
      return (Set) transformToElements(null, ElementType.CONSTRUCTOR, annotation, Constructor.class);
   }

   public <A extends Annotation> Set<Element<A, Field>> classHasFieldAnnotatedWith(Class<A> annotation)
   {
      return transformToElements(null, ElementType.FIELD, annotation, Field.class);
   }

   public <A extends Annotation> Set<Element<A, Method>> classHasMethodAnnotatedWith(Class<A> annotation)
   {
      return transformToElements(null, ElementType.METHOD, annotation, Method.class);
   }

   public <A extends Annotation> Set<Element<A, AnnotatedElement>> classHasParameterAnnotatedWith(Class<A> annotation)
   {
      return transformToElements(null, ElementType.PARAMETER, annotation, AnnotatedElement.class);
   }

   /**
    * Load the annotation class.
    *
    * @param annotationName the annoation class name
    * @return annotation class
    */
   @SuppressWarnings("unchecked")
   protected Class<Annotation> getAnnotationClass(String annotationName)
   {
      Class<?> clazz = loadClass(annotationName);
      if (Annotation.class.isAssignableFrom(clazz) == false)
         throw new IllegalArgumentException("Annotation name " + annotationName + " doesn't extend Annotation class.");

      return (Class<Annotation>)clazz;
   }

   public boolean hasClassAnnotatedWith(String annotationName)
   {
      return hasClassAnnotatedWith(getAnnotationClass(annotationName));
   }

   public Set<Element<Annotation, Class<?>>> classIsAnnotatedWith(String annotationName)
   {
      return classIsAnnotatedWith(getAnnotationClass(annotationName));
   }

   public Set<Element<Annotation, Constructor<?>>> classHasConstructorAnnotatedWith(String annotationName)
   {
      return classHasConstructorAnnotatedWith(getAnnotationClass(annotationName));
   }

   public Set<Element<Annotation, Field>> classHasFieldAnnotatedWith(String annotationName)
   {
      return classHasFieldAnnotatedWith(getAnnotationClass(annotationName));
   }

   public Set<Element<Annotation, Method>> classHasMethodAnnotatedWith(String annotationName)
   {
      return classHasMethodAnnotatedWith(getAnnotationClass(annotationName));
   }

   public Set<Element<Annotation, AnnotatedElement>> classHasParameterAnnotatedWith(String annotationName)
   {
      return classHasParameterAnnotatedWith(getAnnotationClass(annotationName));
   }
}
