/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2010, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.scanning.hibernate;

import java.lang.annotation.Annotation;
import java.util.ArrayList;
import java.util.List;

import org.jboss.classloading.plugins.visitor.FederatedResourceVisitor;
import org.jboss.classloading.spi.visitor.ResourceContext;
import org.jboss.classloading.spi.visitor.ResourceFilter;
import org.jboss.classloading.spi.visitor.ResourceVisitor;
import org.jboss.deployers.structure.spi.DeploymentUnit;
import org.jboss.scanning.plugins.helpers.DeploymentUtilsFactory;
import org.jboss.scanning.plugins.visitor.ReflectProvider;
import org.jboss.scanning.spi.helpers.AbstractScanningPlugin;

import org.hibernate.ejb.packaging.Scanner;

/**
 * Hibernate's scanning plugin.
 *
 * @author <a href="mailto:ales.justin@jboss.org">Ales Justin</a>
 */
public class HibernateScanningPlugin extends AbstractScanningPlugin<ScannerImpl, Scanner>
{
   /** The scanner */
   private final ScannerImpl scanner;

   /** The visitor */
   private final ResourceVisitor visitor;

   public HibernateScanningPlugin(DeploymentUnit unit, HibernateScanningPluginFactory factory)
   {
      if (factory == null)
         throw new IllegalArgumentException("Null factory");

      this.scanner = new ScannerImpl(unit);

      List<ResourceVisitor> visitors = new ArrayList<ResourceVisitor>();
      // handle packages
      visitors.add(new PackageVisitor(scanner));
      // handle files
      for (String pattern : factory.getPatterns())
      {
         ResourceVisitor rv = new PatternVisitor(pattern, scanner);
         visitors.add(rv);
      }
      // handle classes
      ReflectProvider provider = DeploymentUtilsFactory.getProvider(unit);
      for (Class<? extends Annotation> annotation : factory.getAnnotations())
      {
         ResourceVisitor rv = new SingleAnnotationVisitor(provider, annotation, scanner);
         visitors.add(rv);
      }
      this.visitor = new FederatedResourceVisitor(visitors.toArray(new ResourceVisitor[visitors.size()]));
   }

   protected ScannerImpl doCreateHandle()
   {
      return scanner;
   }

   @Override
   public void cleanupHandle(Scanner handle)
   {
      if (handle instanceof ScannerImpl)
         ScannerImpl.class.cast(handle).cleanup();
   }

   public Class<Scanner> getHandleInterface()
   {
      return Scanner.class;
   }

   public ResourceFilter getFilter()
   {
      return visitor.getFilter();
   }

   public void visit(ResourceContext resource)
   {
      visitor.visit(resource);
   }
}