/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2010, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.scanning.hierarchy.plugins;

import java.io.Externalizable;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;
import java.net.URL;
import java.util.*;

import org.jboss.reflect.plugins.introspection.IntrospectionTypeInfoFactory;
import org.jboss.reflect.spi.TypeInfo;
import org.jboss.reflect.spi.TypeInfoFactory;
import org.jboss.scanning.hierarchy.spi.HierarchyIndex;
import org.jboss.scanning.plugins.helpers.MergeUtils;
import org.jboss.scanning.plugins.helpers.WeakClassLoaderHolder;
import org.jboss.scanning.spi.ScanningHandle;

/**
 * @author <a href="mailto:ales.justin@jboss.org">Ales Justin</a>
 */
public class HierarchyIndexImpl extends WeakClassLoaderHolder implements HierarchyIndex, ScanningHandle<HierarchyIndexImpl>, Externalizable
{
   private static final long serialVersionUID = 1l;
   
   /** The inherited types */
   private Map<String, Map<TypeInfo, Set<TypeInfo>>> cache = new HashMap<String, Map<TypeInfo, Set<TypeInfo>>>();
   /** The type info factory */
   private transient TypeInfoFactory tif;

   /**
    * De-serialization only.
    */
   public HierarchyIndexImpl()
   {
      super();
   }

   public HierarchyIndexImpl(ClassLoader classLoader)
   {
      super(classLoader);
   }

   @Override
   protected ClassLoader getClassLoader()
   {
      return super.getClassLoader();
   }

   void cleanup()
   {
      cache.clear();
   }

   Map<String, Map<TypeInfo, Set<TypeInfo>>> getCache()
   {
      return cache;
   }

   public void merge(HierarchyIndexImpl subHandle)
   {      
      if (tif == null)
         tif = subHandle.tif;

      MergeUtils.doubleMerge(cache, subHandle.getCache());
   }

   void putInfo(String path, TypeInfo key, TypeInfo... classes)
   {
      if (tif == null)
         tif = key.getTypeInfoFactory();

      Map<TypeInfo, Set<TypeInfo>> map = cache.get(path);
      if (map == null)
      {
         map = new HashMap<TypeInfo, Set<TypeInfo>>();
         cache.put(path, map);
      }
      for (TypeInfo superType : classes)
      {
         Set<TypeInfo> infos = map.get(superType);
         if (infos == null)
         {
            infos = new HashSet<TypeInfo>();
            map.put(superType, infos);
         }
         infos.add(key);
      }
   }

   public Set<TypeInfo> getSubTypesOf(URL url, TypeInfo superTypeToLookFor)
   {
      if (url == null)
         throw new IllegalArgumentException("Null url");

      String path = url.getPath();
      Set<TypeInfo> result = null;
      Map<TypeInfo, Set<TypeInfo>> map = cache.get(fixPath(path));
      if (map != null)
         result = map.get(superTypeToLookFor);

      return result == null || result.isEmpty() ? Collections.<TypeInfo>emptySet() : Collections.unmodifiableSet(result);
   }

   public Set<TypeInfo> getSubTypesOf(URL url, Class<?> superTypeToLookFor)
   {
      if (tif == null) // we haven't cached anything yet
         return Collections.emptySet();

      return getSubTypesOf(url, tif.getTypeInfo(superTypeToLookFor));
   }

   @SuppressWarnings({"unchecked", "deprecation"})
   public <T> Set<Class<? extends T>> getSubClassesOf(URL url, Class<T> superTypeToLookFor)
   {
      Set<TypeInfo> classes = getSubTypesOf(url, superTypeToLookFor);
      Set<Class<? extends T>> result = new HashSet<Class<? extends T>>();
      for (TypeInfo ti : classes)
      {
         Class<?> clazz = ti.getType();
         result.add((Class<T>)clazz);
      }
      return result;
   }

   static String fixPath(String path)
   {
      if (path.endsWith("/") == false)
         path += "/";
      return path;
   }

   public void writeExternal(ObjectOutput out) throws IOException
   {
      out.writeObject(cache);
   }

   @SuppressWarnings("unchecked")
   public void readExternal(ObjectInput in) throws IOException, ClassNotFoundException
   {
      cache = (Map) in.readObject();
      tif = new IntrospectionTypeInfoFactory(); // after de-serialization we use Introspection
   }
}
