/*
 * Decompiled with CFR 0.152.
 */
package org.jboss.shrinkwrap.impl.base.nio2.file;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.nio.CharBuffer;
import java.nio.channels.SeekableByteChannel;
import java.nio.charset.Charset;
import java.nio.file.CopyOption;
import java.nio.file.DirectoryNotEmptyException;
import java.nio.file.DirectoryStream;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.FileSystem;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardCopyOption;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.BasicFileAttributeView;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.util.Iterator;
import java.util.logging.Logger;
import org.jboss.shrinkwrap.api.Archive;
import org.jboss.shrinkwrap.api.ShrinkWrap;
import org.jboss.shrinkwrap.api.asset.Asset;
import org.jboss.shrinkwrap.api.asset.EmptyAsset;
import org.jboss.shrinkwrap.api.asset.StringAsset;
import org.jboss.shrinkwrap.api.nio2.file.ShrinkWrapFileSystems;
import org.jboss.shrinkwrap.api.spec.JavaArchive;
import org.jboss.shrinkwrap.impl.base.nio2.file.ShrinkWrapFileAttributeView;
import org.jboss.shrinkwrap.impl.base.nio2.file.ShrinkWrapFileSystem;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;

public class FilesTestCase {
    private static final Logger log = Logger.getLogger(FilesTestCase.class.getName());
    @Rule
    public ExpectedException expectedException = ExpectedException.none();
    private FileSystem fs;

    @Before
    public void createFileSystem() throws IOException {
        String name = "test.jar";
        JavaArchive archive = (JavaArchive)ShrinkWrap.create(JavaArchive.class, (String)"test.jar");
        this.fs = ShrinkWrapFileSystems.newFileSystem((Archive)archive);
    }

    @After
    public void closeFileSystem() throws IOException {
        this.fs.close();
    }

    @Test
    public void delete() throws IOException {
        Archive<?> archive = this.getArchive();
        String pathString = "fileToDelete";
        archive.add((Asset)EmptyAsset.INSTANCE, "fileToDelete");
        Assert.assertTrue((boolean)archive.contains("fileToDelete"));
        Path path = this.fs.getPath("fileToDelete", new String[0]);
        Files.delete(path);
        Assert.assertFalse((boolean)archive.contains("fileToDelete"));
    }

    @Test
    public void deleteNonexistant() throws IOException {
        Archive<?> archive = this.getArchive();
        String pathString = "nonexistant";
        Assert.assertFalse((boolean)archive.contains("nonexistant"));
        Path path = this.fs.getPath("nonexistant", new String[0]);
        boolean gotException = false;
        try {
            Files.delete(path);
        }
        catch (NoSuchFileException nsfe) {
            gotException = true;
        }
        Assert.assertTrue((String)("Request to remove nonexistant path should have thrown " + NoSuchFileException.class.getSimpleName()), (boolean)gotException);
    }

    @Test
    public void deleteIfExists() throws IOException {
        Archive<?> archive = this.getArchive();
        String pathString = "fileToDelete";
        archive.add((Asset)EmptyAsset.INSTANCE, "fileToDelete");
        Assert.assertTrue((boolean)archive.contains("fileToDelete"));
        Path path = this.fs.getPath("fileToDelete", new String[0]);
        boolean deleted = Files.deleteIfExists(path);
        Assert.assertTrue((String)"Did not report deleted", (boolean)deleted);
    }

    @Test
    public void deleteIfExistsDoesntExist() throws IOException {
        String pathString = "fileWhichDoesNotExist";
        Path path = this.fs.getPath("fileWhichDoesNotExist", new String[0]);
        boolean deleted = Files.deleteIfExists(path);
        Assert.assertFalse((String)"Should not report deleted", (boolean)deleted);
    }

    @Test
    public void deleteDirectory() throws IOException {
        String directoryName = "directory";
        Archive archive = this.getArchive().addAsDirectory("directory");
        Assert.assertNull((String)"Test archive should contain the directory, not content", (Object)archive.get("directory").getAsset());
        Path path = this.fs.getPath("directory", new String[0]);
        Files.delete(path);
        Assert.assertFalse((String)"Archive should no longer contain directory ", (boolean)archive.contains("directory"));
    }

    @Test
    public void deleteUnemptyDirectory() throws IOException {
        String directoryName = "/directory";
        String subDirectoryName = "/directory/subdir";
        Archive archive = this.getArchive().addAsDirectory("/directory/subdir");
        Assert.assertNull((String)"Test archive should contain the directory, not content", (Object)archive.get("/directory/subdir").getAsset());
        Path path = this.fs.getPath("/directory", new String[0]);
        boolean gotException = false;
        try {
            Files.delete(path);
        }
        catch (DirectoryNotEmptyException dnee) {
            gotException = true;
        }
        Assert.assertTrue((String)"Should not be able to delete non-empty directory", (boolean)gotException);
    }

    @Test
    public void createDirectory() throws IOException {
        String dirName = "/newDirectory";
        Path dir = this.fs.getPath("/newDirectory", new String[0]);
        Archive<?> archive = this.getArchive();
        Assert.assertFalse((boolean)archive.contains("/newDirectory"));
        Path createdDir = Files.createDirectory(dir, new FileAttribute[]{null});
        Assert.assertTrue((String)"Archive does not contain created directory", (boolean)archive.contains("/newDirectory"));
        Assert.assertTrue((String)"Created path is not a directory", (archive.get("/newDirectory").getAsset() == null ? 1 : 0) != 0);
        Assert.assertEquals((String)"Created directory name was not as expected", (Object)"/newDirectory", (Object)createdDir.toString());
    }

    @Test
    public void createDirectoryRecursiveProhibited() throws IOException {
        String dirName = "/newDirectory/child";
        Path dir = this.fs.getPath("/newDirectory/child", new String[0]);
        Archive<?> archive = this.getArchive();
        Assert.assertFalse((boolean)archive.contains("/newDirectory/child"));
        boolean gotException = false;
        try {
            Files.createDirectory(dir, new FileAttribute[]{null});
        }
        catch (IOException ioe) {
            gotException = true;
        }
        Assert.assertTrue((String)"Should not be able to create directory unless parents are first present", (boolean)gotException);
    }

    @Test
    public void createDirectoriesRecursive() throws IOException {
        String dirName = "/newDirectory/child";
        Path dir = this.fs.getPath("/newDirectory/child", new String[0]);
        Archive<?> archive = this.getArchive();
        Assert.assertFalse((boolean)archive.contains("/newDirectory/child"));
        Path createdDir = Files.createDirectories(dir, new FileAttribute[]{null});
        Assert.assertTrue((String)"Archive does not contain created directory", (boolean)archive.contains("/newDirectory/child"));
        Assert.assertTrue((String)"Created path is not a directory", (archive.get("/newDirectory/child").getAsset() == null ? 1 : 0) != 0);
        Assert.assertEquals((String)"Created directory name was not as expected", (Object)"/newDirectory/child", (Object)createdDir.toString());
    }

    @Test
    public void copyFromInputStreamToPath() throws IOException {
        String contents = "Hello, testing content writing!";
        byte[] bytes = "Hello, testing content writing!".getBytes("UTF-8");
        ByteArrayInputStream in = new ByteArrayInputStream(bytes);
        String pathName = "content";
        Path path = this.fs.getPath("content", new String[0]);
        long bytesCopied = Files.copy(in, path, new CopyOption[0]);
        String roundtrip = new BufferedReader(new InputStreamReader(this.getArchive().get("content").getAsset().openStream())).readLine();
        Assert.assertEquals((String)"Contents after copy were not as expected", (Object)"Hello, testing content writing!", (Object)roundtrip);
        Assert.assertEquals((long)bytes.length, (long)bytesCopied);
    }

    @Test
    public void copyFromInputStreamToExistingPath() throws IOException {
        ByteArrayInputStream in = new ByteArrayInputStream("test".getBytes("UTF-8"));
        String pathName = "content";
        Path path = this.fs.getPath("content", new String[0]);
        this.getArchive().add((Asset)EmptyAsset.INSTANCE, "content");
        boolean gotException = false;
        try {
            Files.copy(in, path, new CopyOption[0]);
        }
        catch (FileAlreadyExistsException faee) {
            gotException = true;
        }
        Assert.assertTrue((String)"Overwrite of existing path should fail", (boolean)gotException);
    }

    @Test
    public void copyFromInputStreamToExistingDirectory() throws IOException {
        ByteArrayInputStream in = new ByteArrayInputStream("test".getBytes("UTF-8"));
        String pathName = "directory";
        Path path = this.fs.getPath("directory", new String[0]);
        this.getArchive().addAsDirectories(new String[]{"directory"});
        boolean gotException = false;
        try {
            Files.copy(in, path, new CopyOption[0]);
        }
        catch (FileAlreadyExistsException faee) {
            gotException = true;
        }
        Assert.assertTrue((String)"Overwrite of existing directory should fail", (boolean)gotException);
    }

    @Test
    public void copyFromInputStreamToExistingNonEmptyDirectoryWithReplaceExistingOption() throws IOException {
        ByteArrayInputStream in = new ByteArrayInputStream("test".getBytes("UTF-8"));
        String dir = "/directory";
        String subdir = "/directory/subdir";
        Path dirPath = this.fs.getPath("/directory", new String[0]);
        this.getArchive().addAsDirectories(new String[]{"/directory/subdir"});
        boolean gotException = false;
        try {
            Files.copy(in, dirPath, StandardCopyOption.REPLACE_EXISTING);
        }
        catch (DirectoryNotEmptyException dnee) {
            gotException = true;
        }
        Assert.assertTrue((String)"Overwrite of existing non-empty dir should fail, even with replace option", (boolean)gotException);
    }

    @Test
    public void copyFromInputStreamToExistingPathWithOverwriteOption() throws IOException {
        String contents = "Hello, testing content writing!";
        byte[] bytes = "Hello, testing content writing!".getBytes("UTF-8");
        ByteArrayInputStream in = new ByteArrayInputStream(bytes);
        String pathName = "content";
        Path path = this.fs.getPath("content", new String[0]);
        this.getArchive().add((Asset)EmptyAsset.INSTANCE, "content");
        long bytesCopied = Files.copy(in, path, StandardCopyOption.REPLACE_EXISTING);
        String roundtrip = new BufferedReader(new InputStreamReader(this.getArchive().get("content").getAsset().openStream())).readLine();
        Assert.assertEquals((String)"Contents after copy were not as expected", (Object)"Hello, testing content writing!", (Object)roundtrip);
        Assert.assertEquals((long)bytes.length, (long)bytesCopied);
    }

    @Test
    public void copyFromPathToOutputStream() throws IOException {
        String contents = "Here we're gonna test reading from the archive and writing the contents to an OutStream";
        String path = "contentPath";
        this.getArchive().add((Asset)new StringAsset("Here we're gonna test reading from the archive and writing the contents to an OutStream"), "contentPath");
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        long bytesCopied = Files.copy(this.fs.getPath("contentPath", new String[0]), out);
        String roundtrip = new String(out.toByteArray(), "UTF-8");
        Assert.assertEquals((String)"Contents after copy were not as expected", (Object)"Here we're gonna test reading from the archive and writing the contents to an OutStream", (Object)roundtrip);
        Assert.assertEquals((long)"Here we're gonna test reading from the archive and writing the contents to an OutStream".length(), (long)bytesCopied);
    }

    @Test
    public void copyFromDirectoryPathToOutputStream() throws IOException {
        String path = "dirPath";
        this.getArchive().addAsDirectories(new String[]{"dirPath"});
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        boolean gotException = false;
        try {
            Files.copy(this.fs.getPath("dirPath", new String[0]), out);
        }
        catch (IllegalArgumentException iae) {
            gotException = true;
        }
        Assert.assertTrue((String)"Call to copy a directory contents to an outstream should not succeed", (boolean)gotException);
    }

    @Test
    public void writeWithAppendOption() throws IOException {
        String initialContent = "initial content";
        String appendContent = " - appended contents";
        String pathName = "content";
        Path path = this.fs.getPath("content", new String[0]);
        Archive<?> archive = this.getArchive();
        archive.add((Asset)new StringAsset("initial content"), "content");
        Files.write(path, " - appended contents".getBytes("UTF-8"), StandardOpenOption.APPEND);
        String newContents = new BufferedReader(new InputStreamReader(archive.get("content").getAsset().openStream())).readLine();
        Assert.assertEquals((String)"New contents was not appended as expected", (Object)"initial content - appended contents", (Object)newContents);
    }

    @Test
    public void createFile() throws IOException {
        String pathName = "fileToCreate";
        Path path = this.fs.getPath("fileToCreate", new String[0]);
        Path newPath = Files.createFile(path, new FileAttribute[]{null});
        Assert.assertEquals((Object)path.toString(), (Object)newPath.toString());
    }

    @Test
    public void createFileWhenFileAlreadyExists() throws IOException {
        String pathName = "fileToCreate";
        this.getArchive().add((Asset)EmptyAsset.INSTANCE, "fileToCreate");
        Path path = this.fs.getPath("fileToCreate", new String[0]);
        boolean gotException = false;
        try {
            Files.createFile(path, new FileAttribute[]{null});
        }
        catch (FileAlreadyExistsException faee) {
            gotException = true;
        }
        Assert.assertTrue((String)"create new file should fail if path already exists", (boolean)gotException);
    }

    @Test
    public void createLink() throws IOException {
        String linkToCreateString = "linkToCreate";
        String existingPathString = "existingPath";
        this.getArchive().add((Asset)EmptyAsset.INSTANCE, "existingPath");
        Path linkToCreatePath = this.fs.getPath("linkToCreate", new String[0]);
        Path existingPath = this.fs.getPath("existingPath", new String[0]);
        boolean gotException = false;
        try {
            Files.createLink(linkToCreatePath, existingPath);
        }
        catch (UnsupportedOperationException ooe) {
            gotException = true;
        }
        Assert.assertTrue((String)"We should not support creation of links", (boolean)gotException);
    }

    @Test
    public void createSymbolicLink() throws IOException {
        String symbolicLinkToCreateString = "linkToCreate";
        String existingPathString = "existingPath";
        this.getArchive().add((Asset)EmptyAsset.INSTANCE, "existingPath");
        Path symbolicLinkToCreatePath = this.fs.getPath("linkToCreate", new String[0]);
        Path existingPath = this.fs.getPath("existingPath", new String[0]);
        boolean gotException = false;
        try {
            Files.createSymbolicLink(symbolicLinkToCreatePath, existingPath, new FileAttribute[0]);
        }
        catch (UnsupportedOperationException ooe) {
            gotException = true;
        }
        Assert.assertTrue((String)"We should not support creation of synbolic links", (boolean)gotException);
    }

    @Test
    public void createTempDirectory() throws IOException {
        String tempDir = "/tmp";
        this.getArchive().addAsDirectories(new String[]{"/tmp"});
        String prefix = "prefix";
        Path tempDirPath = this.fs.getPath("/tmp", new String[0]);
        Path newPath = Files.createTempDirectory(tempDirPath, "prefix", new FileAttribute[0]);
        Assert.assertTrue((String)"temp dir name was not in expected form", (boolean)newPath.toString().startsWith("/tmp/prefix"));
    }

    @Test
    public void existsFalse() throws IOException {
        String pathString = "fileWhichDoesNotExist";
        Path path = this.fs.getPath("fileWhichDoesNotExist", new String[0]);
        boolean exists = Files.exists(path, new LinkOption[0]);
        Assert.assertFalse((String)"Should report exists", (boolean)exists);
    }

    @Test
    public void existsTrue() throws IOException {
        Archive<?> archive = this.getArchive();
        String pathString = "file";
        archive.add((Asset)EmptyAsset.INSTANCE, "file");
        Path path = this.fs.getPath("file", new String[0]);
        boolean exists = Files.exists(path, new LinkOption[0]);
        Assert.assertTrue((String)"Should report exists", (boolean)exists);
    }

    @Test(expected=UnsupportedOperationException.class)
    public void getAttribute() throws IOException {
        Files.getAttribute(this.fs.getPath("file", new String[0]), "basic", new LinkOption[]{null});
    }

    @Test
    public void getFileAttributeView() throws IOException {
        Assert.assertTrue((boolean)(Files.getFileAttributeView(this.fs.getPath("file", new String[0]), BasicFileAttributeView.class, new LinkOption[]{null}) instanceof ShrinkWrapFileAttributeView));
    }

    @Test
    public void getFileStore() throws IOException {
        Assert.assertTrue((String)"Returned file store was not as expected", (this.fs.getFileStores().iterator().next() == Files.getFileStore(this.fs.getPath("path", new String[0])) ? 1 : 0) != 0);
    }

    @Test(expected=UnsupportedOperationException.class)
    public void getLastModifiedTime() throws IOException {
        this.getArchive().add((Asset)EmptyAsset.INSTANCE, "path");
        Files.getLastModifiedTime(this.fs.getPath("path", new String[0]), new LinkOption[]{null});
    }

    @Test(expected=UnsupportedOperationException.class)
    public void getOwner() throws IOException {
        Files.getOwner(this.fs.getPath("path", new String[0]), new LinkOption[]{null});
    }

    @Test(expected=UnsupportedOperationException.class)
    public void getPosixFilePermissions() throws IOException {
        Files.getPosixFilePermissions(this.fs.getPath("path", new String[0]), new LinkOption[]{null});
    }

    @Test
    public void isDirectoryTrue() throws IOException {
        this.getArchive().addAsDirectories(new String[]{"path"});
        Assert.assertTrue((boolean)Files.isDirectory(this.fs.getPath("path", new String[0]), new LinkOption[]{null}));
    }

    @Test
    public void isDirectoryFalse() throws IOException {
        Assert.assertFalse((boolean)Files.isDirectory(this.fs.getPath("path", new String[0]), new LinkOption[]{null}));
    }

    @Test
    public void isExecutable() throws IOException {
        String path = "path";
        this.getArchive().add((Asset)EmptyAsset.INSTANCE, "path");
        Assert.assertTrue((boolean)Files.isExecutable(this.fs.getPath("path", new String[0])));
    }

    @Test
    public void isHidden() throws IOException {
        String path = "path";
        this.getArchive().add((Asset)EmptyAsset.INSTANCE, "path");
        Assert.assertFalse((boolean)Files.isHidden(this.fs.getPath("path", new String[0])));
    }

    @Test
    public void isReadable() throws IOException {
        String path = "path";
        this.getArchive().add((Asset)EmptyAsset.INSTANCE, "path");
        Assert.assertTrue((boolean)Files.isReadable(this.fs.getPath("path", new String[0])));
    }

    @Test
    public void isExecutableNonexistant() throws IOException {
        String path = "path";
        Assert.assertFalse((boolean)Files.isExecutable(this.fs.getPath("path", new String[0])));
    }

    @Test
    public void isReadableNonexistant() throws IOException {
        String path = "path";
        Assert.assertFalse((boolean)Files.isReadable(this.fs.getPath("path", new String[0])));
    }

    @Test
    public void isRegularFile() throws IOException {
        String path = "path";
        this.getArchive().add((Asset)EmptyAsset.INSTANCE, "path");
        Assert.assertTrue((boolean)Files.isRegularFile(this.fs.getPath("path", new String[0]), new LinkOption[0]));
    }

    @Test
    public void isRegularFileFalse() throws IOException {
        String path = "path/";
        Assert.assertFalse((boolean)Files.isRegularFile(this.fs.getPath("path/", new String[0]), new LinkOption[0]));
    }

    @Test
    public void isSameFile() throws IOException {
        String path1 = "path/sub";
        String path2 = "path/sub";
        Assert.assertTrue((boolean)Files.isSameFile(this.fs.getPath("path/sub", new String[0]), this.fs.getPath("path/sub", new String[0])));
    }

    @Test
    public void isSameFileFalse() throws IOException {
        String path1 = "path/sub";
        String path2 = "path/sub/notsame";
        Assert.assertFalse((boolean)Files.isSameFile(this.fs.getPath("path/sub", new String[0]), this.fs.getPath("path/sub/notsame", new String[0])));
    }

    @Test
    public void isSymbolicLink() throws IOException {
        String path = "path";
        Assert.assertFalse((boolean)Files.isSymbolicLink(this.fs.getPath("path", new String[0])));
    }

    @Test
    public void isWritable() throws IOException {
        String path = "path";
        this.getArchive().add((Asset)EmptyAsset.INSTANCE, "path");
        Assert.assertTrue((boolean)Files.isWritable(this.fs.getPath("path", new String[0])));
    }

    @Test
    public void isWritableNonexistant() throws IOException {
        String path = "path";
        Assert.assertFalse((boolean)Files.isWritable(this.fs.getPath("path", new String[0])));
    }

    @Test
    public void move() throws IOException {
        String source = "path";
        String contents = "contents";
        this.getArchive().add((Asset)new StringAsset("contents"), "path");
        String dest = "newPath";
        Path src = this.fs.getPath("path", new String[0]);
        Path dst = this.fs.getPath("newPath", new String[0]);
        Path moved = Files.move(src, dst, new CopyOption[0]);
        Assert.assertEquals((Object)"newPath", (Object)moved.toString());
        String roundtrip = new BufferedReader(new InputStreamReader(this.getArchive().get("newPath").getAsset().openStream())).readLine();
        Assert.assertEquals((String)"Contents not as expected after move", (Object)"contents", (Object)roundtrip);
    }

    @Test
    public void moveDirectory() throws IOException {
        String source = "dir";
        this.getArchive().addAsDirectory("dir");
        String dest = "newPath";
        Path src = this.fs.getPath("dir", new String[0]);
        Path dst = this.fs.getPath("newPath", new String[0]);
        Path moved = Files.move(src, dst, new CopyOption[0]);
        Assert.assertEquals((Object)"newPath", (Object)moved.toString());
        Assert.assertNull((String)"Directory expected after move", (Object)this.getArchive().get("newPath").getAsset());
    }

    @Test
    public void newBufferedReader() throws IOException {
        String path = "path";
        String contents = "contents";
        this.getArchive().add((Asset)new StringAsset("contents"), "path");
        BufferedReader reader = Files.newBufferedReader(this.fs.getPath("path", new String[0]), Charset.defaultCharset());
        CharBuffer buffer = CharBuffer.allocate("contents".length());
        reader.read(buffer);
        reader.close();
        buffer.position(0);
        Assert.assertEquals((String)"Contents not read as expected from the buffered reader", (Object)"contents", (Object)buffer.toString());
    }

    @Test
    public void newBufferedWriter() throws IOException {
        String path = "path";
        String contents = "contents";
        BufferedWriter writer = Files.newBufferedWriter(this.fs.getPath("path", new String[0]), Charset.defaultCharset(), new OpenOption[]{null});
        writer.write("contents");
        writer.close();
        String roundtrip = new BufferedReader(new InputStreamReader(this.getArchive().get("path").getAsset().openStream())).readLine();
        Assert.assertEquals((String)"Contents not written as expected from the buffered writer", (Object)"contents", (Object)roundtrip);
    }

    @Test
    public void newByteChannel() throws IOException {
        String path = "path";
        String contents = "ALR is putting some contents into here.";
        SeekableByteChannel channel = Files.newByteChannel(this.fs.getPath("path", new String[0]), StandardOpenOption.READ, StandardOpenOption.WRITE);
        ByteBuffer writeBuffer = ByteBuffer.wrap("ALR is putting some contents into here.".getBytes());
        channel.write(writeBuffer);
        ByteBuffer readBuffer = ByteBuffer.allocate("ALR is putting some contents into here.".length());
        channel.position(0L);
        channel.read(readBuffer);
        String roundtrip = new String(readBuffer.array());
        String roundTripViaArchive = new BufferedReader(new InputStreamReader(this.getArchive().get("path").getAsset().openStream())).readLine();
        Assert.assertEquals((String)"Contents not read as expected from the channel", (Object)"ALR is putting some contents into here.", (Object)roundtrip);
        Assert.assertEquals((String)"Contents not read as expected from the archive", (Object)"ALR is putting some contents into here.", (Object)roundTripViaArchive);
    }

    @Test(expected=IllegalArgumentException.class)
    public void newByteChannelForReadDoesntExist() throws IOException {
        Files.newByteChannel(this.fs.getPath("path", new String[0]), new OpenOption[]{null});
    }

    @Test
    public void newDirectoryStream() throws IOException {
        String dirs = "a/b/c/d/e";
        this.getArchive().addAsDirectories(new String[]{"a/b/c/d/e"});
        DirectoryStream<Path> stream = Files.newDirectoryStream(this.fs.getPath("/", new String[0]));
        Iterator<Path> paths = stream.iterator();
        int counter = 0;
        while (paths.hasNext()) {
            ++counter;
            Path path = paths.next();
            Assert.assertTrue((boolean)this.getArchive().contains(path.toString()));
        }
        Assert.assertEquals((long)1L, (long)counter);
    }

    @Test
    public void newInputStream() throws IOException {
        String path = "path";
        String contents = "contents";
        this.getArchive().add((Asset)new StringAsset("contents"), "path");
        InputStream in = Files.newInputStream(this.fs.getPath("path", new String[0]), StandardOpenOption.READ);
        byte[] buffer = new byte["contents".length()];
        in.read(buffer);
        in.close();
        String roundtrip = new String(buffer);
        Assert.assertEquals((String)"Contents not read as expected from the instream", (Object)"contents", (Object)roundtrip);
    }

    @Test
    public void newOutputStream() throws IOException {
        String path = "path";
        String contents = "contents";
        OutputStream outStream = Files.newOutputStream(this.fs.getPath("path", new String[0]), StandardOpenOption.WRITE);
        outStream.write("contents".getBytes());
        outStream.close();
        String roundtrip = new BufferedReader(new InputStreamReader(this.getArchive().get("path").getAsset().openStream())).readLine();
        Assert.assertEquals((String)"Contents not read as expected from the outstream", (Object)"contents", (Object)roundtrip);
    }

    @Test
    public void notExistsTrue() throws IOException {
        Assert.assertTrue((boolean)Files.notExists(this.fs.getPath("fake", new String[0]), LinkOption.NOFOLLOW_LINKS));
    }

    @Test
    public void notExistsFalse() throws IOException {
        this.getArchive().add((Asset)EmptyAsset.INSTANCE, "path");
        Assert.assertFalse((boolean)Files.notExists(this.fs.getPath("path", new String[0]), LinkOption.NOFOLLOW_LINKS));
    }

    @Test
    public void probeContentType() throws IOException {
        String path = "path";
        String contents = "contents";
        this.getArchive().add((Asset)new StringAsset("contents"), "path");
        Assert.assertNull((Object)Files.probeContentType(this.fs.getPath("path", new String[0])));
    }

    @Test
    public void readAllBytes() throws IOException {
        String path = "path";
        String contents = "contents";
        this.getArchive().add((Asset)new StringAsset("contents"), "path");
        byte[] bytes = Files.readAllBytes(this.fs.getPath("path", new String[0]));
        String roundtrip = new String(bytes);
        Assert.assertEquals((String)"Contents not read as expected from the readAllBytes", (Object)"contents", (Object)roundtrip);
    }

    @Test
    public void createdDirectoryIsADirectory() throws Exception {
        Path dirPath = this.fs.getPath("dir", new String[0]);
        Files.createDirectory(dirPath, new FileAttribute[0]);
        Assert.assertTrue((String)"Created directory was not a directory", (boolean)Files.isDirectory(dirPath, new LinkOption[0]));
    }

    @Test
    public void createdDirectoryCanBeStreamed() throws Exception {
        Path dirPath = this.fs.getPath("dir", new String[0]);
        Files.createDirectory(dirPath, new FileAttribute[0]);
        Files.createFile(dirPath.resolve("file"), new FileAttribute[0]);
        DirectoryStream<Path> stream = Files.newDirectoryStream(dirPath);
        Iterator<Path> it = stream.iterator();
        Assert.assertEquals((Object)"/dir/file", (Object)it.next().toString());
        Assert.assertFalse((String)"No further elements expected in stream", (boolean)it.hasNext());
    }

    @Test
    public void createdDirectoryStreamThrowsExceptionWhenIsClosed() throws Throwable {
        Path dirPath = this.fs.getPath("dir", new String[0]);
        Files.createDirectory(dirPath, new FileAttribute[0]);
        Files.createFile(dirPath.resolve("file"), new FileAttribute[0]);
        DirectoryStream<Path> stream = Files.newDirectoryStream(dirPath);
        stream.close();
        this.expectedException.expect(IllegalStateException.class);
        Iterator<Path> it = stream.iterator();
    }

    @Test
    public void createdDirectoryStreamThrowsExceptionWhenIteratorWasReturnedBefore() throws Exception {
        Path dirPath = this.fs.getPath("dir", new String[0]);
        Files.createDirectory(dirPath, new FileAttribute[0]);
        Files.createFile(dirPath.resolve("file"), new FileAttribute[0]);
        DirectoryStream<Path> stream = Files.newDirectoryStream(dirPath);
        Iterator<Path> it = stream.iterator();
        this.expectedException.expect(IllegalStateException.class);
        stream.iterator();
    }

    @Test
    public void directoryStreamDoesNotContainSubfolders() throws Exception {
        Files.createDirectories(this.fs.getPath("dir/subdir", new String[0]), new FileAttribute[0]);
        DirectoryStream<Path> stream = Files.newDirectoryStream(this.fs.getPath("/", new String[0]));
        Iterator<Path> it = stream.iterator();
        Assert.assertEquals((Object)"/dir", (Object)it.next().toString());
        Assert.assertFalse((String)"No further elements expected in stream", (boolean)it.hasNext());
    }

    @Test
    public void createdDirectoryCanBeWalked() throws Exception {
        Path dirPath = this.fs.getPath("dir", new String[0]);
        Files.createDirectory(dirPath, new FileAttribute[0]);
        Files.createFile(dirPath.resolve("file"), new FileAttribute[0]);
        final int[] visitFileCalled = new int[1];
        Files.walkFileTree(dirPath, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                visitFileCalled[0] = visitFileCalled[0] + 1;
                return super.visitFile(file, attrs);
            }
        });
        Assert.assertEquals((long)1L, (long)visitFileCalled[0]);
    }

    private Archive<?> getArchive() {
        ShrinkWrapFileSystem swfs = (ShrinkWrapFileSystem)this.fs;
        Archive archive = swfs.getArchive();
        return archive;
    }
}

