/*
 * Decompiled with CFR 0.152.
 */
package org.jboss.shrinkwrap.impl.base.nio2.file;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.nio.CharBuffer;
import java.nio.channels.SeekableByteChannel;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.CopyOption;
import java.nio.file.DirectoryNotEmptyException;
import java.nio.file.DirectoryStream;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.FileSystem;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardCopyOption;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.BasicFileAttributeView;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.util.Iterator;
import org.jboss.shrinkwrap.api.Archive;
import org.jboss.shrinkwrap.api.ShrinkWrap;
import org.jboss.shrinkwrap.api.asset.Asset;
import org.jboss.shrinkwrap.api.asset.EmptyAsset;
import org.jboss.shrinkwrap.api.asset.StringAsset;
import org.jboss.shrinkwrap.api.nio2.file.ShrinkWrapFileSystems;
import org.jboss.shrinkwrap.api.spec.JavaArchive;
import org.jboss.shrinkwrap.impl.base.nio2.file.ShrinkWrapFileAttributeView;
import org.jboss.shrinkwrap.impl.base.nio2.file.ShrinkWrapFileSystem;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

public class FilesTestCase {
    private FileSystem fs;

    @BeforeEach
    public void createFileSystem() throws IOException {
        String name = "test.jar";
        JavaArchive archive = (JavaArchive)ShrinkWrap.create(JavaArchive.class, (String)"test.jar");
        this.fs = ShrinkWrapFileSystems.newFileSystem((Archive)archive);
    }

    @AfterEach
    public void closeFileSystem() throws IOException {
        this.fs.close();
    }

    @Test
    public void delete() throws IOException {
        Archive<?> archive = this.getArchive();
        String pathString = "fileToDelete";
        archive.add((Asset)EmptyAsset.INSTANCE, "fileToDelete");
        Assertions.assertTrue((boolean)archive.contains("fileToDelete"));
        Path path = this.fs.getPath("fileToDelete", new String[0]);
        Files.delete(path);
        Assertions.assertFalse((boolean)archive.contains("fileToDelete"));
    }

    @Test
    public void deleteNonexistent() {
        Archive<?> archive = this.getArchive();
        String pathString = "nonexistent";
        Assertions.assertFalse((boolean)archive.contains("nonexistent"));
        Path path = this.fs.getPath("nonexistent", new String[0]);
        Assertions.assertThrows(NoSuchFileException.class, () -> Files.delete(path), (String)("Request to remove nonexistent path should have thrown " + NoSuchFileException.class.getSimpleName()));
    }

    @Test
    public void deleteIfExists() throws IOException {
        Archive<?> archive = this.getArchive();
        String pathString = "fileToDelete";
        archive.add((Asset)EmptyAsset.INSTANCE, "fileToDelete");
        Assertions.assertTrue((boolean)archive.contains("fileToDelete"));
        Path path = this.fs.getPath("fileToDelete", new String[0]);
        boolean deleted = Files.deleteIfExists(path);
        Assertions.assertTrue((boolean)deleted, (String)"Did not report deleted");
    }

    @Test
    public void deleteIfExistsDoesntExist() throws IOException {
        String pathString = "fileWhichDoesNotExist";
        Path path = this.fs.getPath("fileWhichDoesNotExist", new String[0]);
        boolean deleted = Files.deleteIfExists(path);
        Assertions.assertFalse((boolean)deleted, (String)"Should not report deleted");
    }

    @Test
    public void deleteDirectory() throws IOException {
        String directoryName = "directory";
        Archive archive = this.getArchive().addAsDirectory("directory");
        Assertions.assertNull((Object)archive.get("directory").getAsset(), (String)"Test archive should contain the directory, not content");
        Path path = this.fs.getPath("directory", new String[0]);
        Files.delete(path);
        Assertions.assertFalse((boolean)archive.contains("directory"), (String)"Archive should no longer contain directory ");
    }

    @Test
    public void deleteNonemptyDirectory() {
        String directoryName = "/directory";
        String subDirectoryName = "/directory/subdir";
        Archive archive = this.getArchive().addAsDirectory("/directory/subdir");
        Assertions.assertNull((Object)archive.get("/directory/subdir").getAsset(), (String)"Test archive should contain the directory, not content");
        Path path = this.fs.getPath("/directory", new String[0]);
        Assertions.assertThrows(DirectoryNotEmptyException.class, () -> Files.delete(path), (String)"Should not be able to delete non-empty directory");
    }

    @Test
    public void createDirectory() throws IOException {
        String dirName = "/newDirectory";
        Path dir = this.fs.getPath("/newDirectory", new String[0]);
        Archive<?> archive = this.getArchive();
        Assertions.assertFalse((boolean)archive.contains("/newDirectory"));
        Path createdDir = Files.createDirectory(dir, new FileAttribute[]{null});
        Assertions.assertTrue((boolean)archive.contains("/newDirectory"), (String)"Archive does not contain created directory");
        Assertions.assertNull((Object)archive.get("/newDirectory").getAsset(), (String)"Created path is not a directory");
        Assertions.assertEquals((Object)"/newDirectory", (Object)createdDir.toString(), (String)"Created directory name was not as expected");
    }

    @Test
    public void createDirectoryRecursiveProhibited() {
        String dirName = "/newDirectory/child";
        Path dir = this.fs.getPath("/newDirectory/child", new String[0]);
        Archive<?> archive = this.getArchive();
        Assertions.assertFalse((boolean)archive.contains("/newDirectory/child"));
        Assertions.assertThrows(IOException.class, () -> Files.createDirectory(dir, new FileAttribute[]{null}), (String)"Should not be able to create directory unless parents are first present");
    }

    @Test
    public void createDirectoriesRecursive() throws IOException {
        String dirName = "/newDirectory/child";
        Path dir = this.fs.getPath("/newDirectory/child", new String[0]);
        Archive<?> archive = this.getArchive();
        Assertions.assertFalse((boolean)archive.contains("/newDirectory/child"));
        Path createdDir = Files.createDirectories(dir, new FileAttribute[]{null});
        Assertions.assertTrue((boolean)archive.contains("/newDirectory/child"), (String)"Archive does not contain created directory");
        Assertions.assertNull((Object)archive.get("/newDirectory/child").getAsset(), (String)"Created path is not a directory");
        Assertions.assertEquals((Object)"/newDirectory/child", (Object)createdDir.toString(), (String)"Created directory name was not as expected");
    }

    @Test
    public void copyFromInputStreamToPath() throws IOException {
        String contents = "Hello, testing content writing!";
        byte[] bytes = "Hello, testing content writing!".getBytes(StandardCharsets.UTF_8);
        ByteArrayInputStream in = new ByteArrayInputStream(bytes);
        String pathName = "content";
        Path path = this.fs.getPath("content", new String[0]);
        long bytesCopied = Files.copy(in, path, new CopyOption[0]);
        String roundtrip = new BufferedReader(new InputStreamReader(this.getArchive().get("content").getAsset().openStream())).readLine();
        Assertions.assertEquals((Object)"Hello, testing content writing!", (Object)roundtrip, (String)"Contents after copy were not as expected");
        Assertions.assertEquals((long)bytes.length, (long)bytesCopied);
    }

    @Test
    public void copyFromInputStreamToExistingPath() throws IOException {
        ByteArrayInputStream in = new ByteArrayInputStream("test".getBytes(StandardCharsets.UTF_8));
        String pathName = "content";
        Path path = this.fs.getPath("content", new String[0]);
        this.getArchive().add((Asset)EmptyAsset.INSTANCE, "content");
        Assertions.assertThrows(FileAlreadyExistsException.class, () -> Files.copy(in, path, new CopyOption[0]), (String)"Overwrite of existing path should fail");
    }

    @Test
    public void copyFromInputStreamToExistingDirectory() throws IOException {
        ByteArrayInputStream in = new ByteArrayInputStream("test".getBytes(StandardCharsets.UTF_8));
        String pathName = "directory";
        Path path = this.fs.getPath("directory", new String[0]);
        this.getArchive().addAsDirectories(new String[]{"directory"});
        Assertions.assertThrows(FileAlreadyExistsException.class, () -> Files.copy(in, path, new CopyOption[0]), (String)"Overwrite of existing directory should fail");
    }

    @Test
    public void copyFromInputStreamToExistingNonEmptyDirectoryWithReplaceExistingOption() throws IOException {
        ByteArrayInputStream in = new ByteArrayInputStream("test".getBytes(StandardCharsets.UTF_8));
        String dir = "/directory";
        String subdir = "/directory/subdir";
        Path dirPath = this.fs.getPath("/directory", new String[0]);
        this.getArchive().addAsDirectories(new String[]{"/directory/subdir"});
        Assertions.assertThrows(DirectoryNotEmptyException.class, () -> Files.copy(in, dirPath, StandardCopyOption.REPLACE_EXISTING), (String)"Overwrite of existing non-empty dir should fail, even with replace option");
    }

    @Test
    public void copyFromInputStreamToExistingPathWithOverwriteOption() throws IOException {
        String contents = "Hello, testing content writing!";
        byte[] bytes = "Hello, testing content writing!".getBytes(StandardCharsets.UTF_8);
        ByteArrayInputStream in = new ByteArrayInputStream(bytes);
        String pathName = "content";
        Path path = this.fs.getPath("content", new String[0]);
        this.getArchive().add((Asset)EmptyAsset.INSTANCE, "content");
        long bytesCopied = Files.copy(in, path, StandardCopyOption.REPLACE_EXISTING);
        String roundtrip = new BufferedReader(new InputStreamReader(this.getArchive().get("content").getAsset().openStream())).readLine();
        Assertions.assertEquals((Object)"Hello, testing content writing!", (Object)roundtrip, (String)"Contents after copy were not as expected");
        Assertions.assertEquals((long)bytes.length, (long)bytesCopied);
    }

    @Test
    public void copyFromPathToOutputStream() throws IOException {
        String contents = "Here we're gonna test reading from the archive and writing the contents to an OutStream";
        String path = "contentPath";
        this.getArchive().add((Asset)new StringAsset("Here we're gonna test reading from the archive and writing the contents to an OutStream"), "contentPath");
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        long bytesCopied = Files.copy(this.fs.getPath("contentPath", new String[0]), out);
        String roundtrip = new String(out.toByteArray(), StandardCharsets.UTF_8);
        Assertions.assertEquals((Object)"Here we're gonna test reading from the archive and writing the contents to an OutStream", (Object)roundtrip, (String)"Contents after copy were not as expected");
        Assertions.assertEquals((long)"Here we're gonna test reading from the archive and writing the contents to an OutStream".length(), (long)bytesCopied);
    }

    @Test
    public void copyFromDirectoryPathToOutputStream() {
        String path = "dirPath";
        this.getArchive().addAsDirectories(new String[]{"dirPath"});
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        Assertions.assertThrows(IllegalArgumentException.class, () -> Files.copy(this.fs.getPath("dirPath", new String[0]), out), (String)"Call to copy a directory contents to an output stream should not succeed");
    }

    @Test
    public void writeWithAppendOption() throws IOException {
        String initialContent = "initial content";
        String appendContent = " - appended contents";
        String pathName = "content";
        Path path = this.fs.getPath("content", new String[0]);
        Archive<?> archive = this.getArchive();
        archive.add((Asset)new StringAsset("initial content"), "content");
        Files.write(path, " - appended contents".getBytes(StandardCharsets.UTF_8), StandardOpenOption.APPEND);
        String newContents = new BufferedReader(new InputStreamReader(archive.get("content").getAsset().openStream())).readLine();
        Assertions.assertEquals((Object)"initial content - appended contents", (Object)newContents, (String)"New contents was not appended as expected");
    }

    @Test
    public void createFile() throws IOException {
        String pathName = "fileToCreate";
        Path path = this.fs.getPath("fileToCreate", new String[0]);
        Path newPath = Files.createFile(path, new FileAttribute[]{null});
        Assertions.assertEquals((Object)path.toString(), (Object)newPath.toString());
    }

    @Test
    public void createFileWhenFileAlreadyExists() {
        String pathName = "fileToCreate";
        this.getArchive().add((Asset)EmptyAsset.INSTANCE, "fileToCreate");
        Path path = this.fs.getPath("fileToCreate", new String[0]);
        Assertions.assertThrows(FileAlreadyExistsException.class, () -> Files.createFile(path, new FileAttribute[]{null}), (String)"Create new file should fail if path already exists");
    }

    @Test
    public void createLink() {
        String linkToCreateString = "linkToCreate";
        String existingPathString = "existingPath";
        this.getArchive().add((Asset)EmptyAsset.INSTANCE, "existingPath");
        Path linkToCreatePath = this.fs.getPath("linkToCreate", new String[0]);
        Path existingPath = this.fs.getPath("existingPath", new String[0]);
        Assertions.assertThrows(UnsupportedOperationException.class, () -> Files.createLink(linkToCreatePath, existingPath), (String)"We should not support creation of links");
    }

    @Test
    public void createSymbolicLink() {
        String symbolicLinkToCreateString = "linkToCreate";
        String existingPathString = "existingPath";
        this.getArchive().add((Asset)EmptyAsset.INSTANCE, "existingPath");
        Path symbolicLinkToCreatePath = this.fs.getPath("linkToCreate", new String[0]);
        Path existingPath = this.fs.getPath("existingPath", new String[0]);
        Assertions.assertThrows(UnsupportedOperationException.class, () -> Files.createSymbolicLink(symbolicLinkToCreatePath, existingPath, new FileAttribute[0]), (String)"We should not support creation of symbolic links");
    }

    @Test
    public void createTempDirectory() throws IOException {
        String tempDir = "/tmp";
        this.getArchive().addAsDirectories(new String[]{"/tmp"});
        String prefix = "prefix";
        Path tempDirPath = this.fs.getPath("/tmp", new String[0]);
        Path newPath = Files.createTempDirectory(tempDirPath, "prefix", new FileAttribute[0]);
        Assertions.assertTrue((boolean)newPath.toString().startsWith("/tmp/prefix"), (String)"temp dir name was not in expected form");
    }

    @Test
    public void existsFalse() {
        Path path = this.fs.getPath("fileWhichDoesNotExist", new String[0]);
        Assertions.assertFalse((boolean)Files.exists(path, new LinkOption[0]), (String)"Should report exists");
    }

    @Test
    public void existsTrue() {
        Archive<?> archive = this.getArchive();
        String pathString = "file";
        archive.add((Asset)EmptyAsset.INSTANCE, "file");
        Path path = this.fs.getPath("file", new String[0]);
        Assertions.assertTrue((boolean)Files.exists(path, new LinkOption[0]), (String)"Should report exists");
    }

    @Test
    public void getAttribute() {
        Assertions.assertThrows(UnsupportedOperationException.class, () -> Files.getAttribute(this.fs.getPath("file", new String[0]), "basic", new LinkOption[]{null}));
    }

    @Test
    public void getFileAttributeView() {
        Assertions.assertInstanceOf(ShrinkWrapFileAttributeView.class, (Object)Files.getFileAttributeView(this.fs.getPath("file", new String[0]), BasicFileAttributeView.class, new LinkOption[]{null}));
    }

    @Test
    public void getFileStore() throws IOException {
        Assertions.assertSame((Object)this.fs.getFileStores().iterator().next(), (Object)Files.getFileStore(this.fs.getPath("path", new String[0])), (String)"Returned file store was not as expected");
    }

    @Test
    public void getLastModifiedTime() {
        this.getArchive().add((Asset)EmptyAsset.INSTANCE, "path");
        Assertions.assertThrows(UnsupportedOperationException.class, () -> Files.getLastModifiedTime(this.fs.getPath("path", new String[0]), new LinkOption[]{null}));
    }

    @Test
    public void getOwner() {
        Assertions.assertThrows(UnsupportedOperationException.class, () -> Files.getOwner(this.fs.getPath("path", new String[0]), new LinkOption[]{null}));
    }

    @Test
    public void getPosixFilePermissions() {
        Assertions.assertThrows(UnsupportedOperationException.class, () -> Files.getPosixFilePermissions(this.fs.getPath("path", new String[0]), new LinkOption[]{null}));
    }

    @Test
    public void isDirectoryTrue() {
        this.getArchive().addAsDirectories(new String[]{"path"});
        Assertions.assertTrue((boolean)Files.isDirectory(this.fs.getPath("path", new String[0]), new LinkOption[]{null}));
    }

    @Test
    public void isDirectoryFalse() {
        Assertions.assertFalse((boolean)Files.isDirectory(this.fs.getPath("path", new String[0]), new LinkOption[]{null}));
    }

    @Test
    public void isExecutable() {
        String path = "path";
        this.getArchive().add((Asset)EmptyAsset.INSTANCE, "path");
        Assertions.assertTrue((boolean)Files.isExecutable(this.fs.getPath("path", new String[0])));
    }

    @Test
    public void isHidden() throws IOException {
        String path = "path";
        this.getArchive().add((Asset)EmptyAsset.INSTANCE, "path");
        Assertions.assertFalse((boolean)Files.isHidden(this.fs.getPath("path", new String[0])));
    }

    @Test
    public void isReadable() {
        String path = "path";
        this.getArchive().add((Asset)EmptyAsset.INSTANCE, "path");
        Assertions.assertTrue((boolean)Files.isReadable(this.fs.getPath("path", new String[0])));
    }

    @Test
    public void isExecutableNonexistent() {
        String path = "path";
        Assertions.assertFalse((boolean)Files.isExecutable(this.fs.getPath("path", new String[0])));
    }

    @Test
    public void isReadableNonexistent() {
        String path = "path";
        Assertions.assertFalse((boolean)Files.isReadable(this.fs.getPath("path", new String[0])));
    }

    @Test
    public void isRegularFile() {
        String path = "path";
        this.getArchive().add((Asset)EmptyAsset.INSTANCE, "path");
        Assertions.assertTrue((boolean)Files.isRegularFile(this.fs.getPath("path", new String[0]), new LinkOption[0]));
    }

    @Test
    public void isRegularFileFalse() {
        String path = "path/";
        Assertions.assertFalse((boolean)Files.isRegularFile(this.fs.getPath("path/", new String[0]), new LinkOption[0]));
    }

    @Test
    public void isSameFile() throws IOException {
        String path1 = "path/sub";
        String path2 = "path/sub";
        Assertions.assertTrue((boolean)Files.isSameFile(this.fs.getPath("path/sub", new String[0]), this.fs.getPath("path/sub", new String[0])));
    }

    @Test
    public void isSameFileFalse() throws IOException {
        String path1 = "path/sub";
        String path2 = "path/sub/notsame";
        Assertions.assertFalse((boolean)Files.isSameFile(this.fs.getPath("path/sub", new String[0]), this.fs.getPath("path/sub/notsame", new String[0])));
    }

    @Test
    public void isSymbolicLink() {
        String path = "path";
        Assertions.assertFalse((boolean)Files.isSymbolicLink(this.fs.getPath("path", new String[0])));
    }

    @Test
    public void isWritable() {
        String path = "path";
        this.getArchive().add((Asset)EmptyAsset.INSTANCE, "path");
        Assertions.assertTrue((boolean)Files.isWritable(this.fs.getPath("path", new String[0])));
    }

    @Test
    public void isWritableNonexistent() {
        String path = "path";
        Assertions.assertFalse((boolean)Files.isWritable(this.fs.getPath("path", new String[0])));
    }

    @Test
    public void move() throws IOException {
        String source = "path";
        String contents = "contents";
        this.getArchive().add((Asset)new StringAsset("contents"), "path");
        String dest = "newPath";
        Path src = this.fs.getPath("path", new String[0]);
        Path dst = this.fs.getPath("newPath", new String[0]);
        Path moved = Files.move(src, dst, new CopyOption[0]);
        Assertions.assertEquals((Object)"newPath", (Object)moved.toString());
        String roundtrip = new BufferedReader(new InputStreamReader(this.getArchive().get("newPath").getAsset().openStream())).readLine();
        Assertions.assertEquals((Object)"contents", (Object)roundtrip, (String)"Contents not as expected after move");
    }

    @Test
    public void moveDirectory() throws IOException {
        String source = "dir";
        this.getArchive().addAsDirectory("dir");
        String dest = "newPath";
        Path src = this.fs.getPath("dir", new String[0]);
        Path dst = this.fs.getPath("newPath", new String[0]);
        Path moved = Files.move(src, dst, new CopyOption[0]);
        Assertions.assertEquals((Object)"newPath", (Object)moved.toString());
        Assertions.assertNull((Object)this.getArchive().get("newPath").getAsset(), (String)"Directory expected after move");
    }

    @Test
    public void newBufferedReader() throws IOException {
        String path = "path";
        String contents = "contents";
        this.getArchive().add((Asset)new StringAsset("contents"), "path");
        BufferedReader reader = Files.newBufferedReader(this.fs.getPath("path", new String[0]), Charset.defaultCharset());
        CharBuffer buffer = CharBuffer.allocate("contents".length());
        reader.read(buffer);
        reader.close();
        buffer.position(0);
        Assertions.assertEquals((Object)"contents", (Object)buffer.toString(), (String)"Contents not read as expected from the buffered reader");
    }

    @Test
    public void newBufferedWriter() throws IOException {
        String path = "path";
        String contents = "contents";
        BufferedWriter writer = Files.newBufferedWriter(this.fs.getPath("path", new String[0]), Charset.defaultCharset(), new OpenOption[]{null});
        writer.write("contents");
        writer.close();
        String roundtrip = new BufferedReader(new InputStreamReader(this.getArchive().get("path").getAsset().openStream())).readLine();
        Assertions.assertEquals((Object)"contents", (Object)roundtrip, (String)"Contents not written as expected from the buffered writer");
    }

    @Test
    public void newByteChannel() throws IOException {
        String path = "path";
        String contents = "ALR is putting some contents into here.";
        SeekableByteChannel channel = Files.newByteChannel(this.fs.getPath("path", new String[0]), StandardOpenOption.READ, StandardOpenOption.WRITE);
        ByteBuffer writeBuffer = ByteBuffer.wrap("ALR is putting some contents into here.".getBytes());
        channel.write(writeBuffer);
        ByteBuffer readBuffer = ByteBuffer.allocate("ALR is putting some contents into here.".length());
        channel.position(0L);
        channel.read(readBuffer);
        String roundtrip = new String(readBuffer.array());
        String roundTripViaArchive = new BufferedReader(new InputStreamReader(this.getArchive().get("path").getAsset().openStream())).readLine();
        Assertions.assertEquals((Object)"ALR is putting some contents into here.", (Object)roundtrip, (String)"Contents not read as expected from the channel");
        Assertions.assertEquals((Object)"ALR is putting some contents into here.", (Object)roundTripViaArchive, (String)"Contents not read as expected from the archive");
    }

    @Test
    public void newByteChannelForReadDoesntExist() {
        Assertions.assertThrows(IllegalArgumentException.class, () -> Files.newByteChannel(this.fs.getPath("path", new String[0]), new OpenOption[]{null}));
    }

    @Test
    public void newDirectoryStream() throws IOException {
        String dirs = "a/b/c/d/e";
        this.getArchive().addAsDirectories(new String[]{"a/b/c/d/e"});
        DirectoryStream<Path> stream = Files.newDirectoryStream(this.fs.getPath("/", new String[0]));
        Iterator<Path> paths = stream.iterator();
        int counter = 0;
        while (paths.hasNext()) {
            ++counter;
            Path path = paths.next();
            Assertions.assertTrue((boolean)this.getArchive().contains(path.toString()));
        }
        Assertions.assertEquals((int)1, (int)counter);
    }

    @Test
    public void newInputStream() throws IOException {
        String path = "path";
        String contents = "contents";
        this.getArchive().add((Asset)new StringAsset("contents"), "path");
        InputStream in = Files.newInputStream(this.fs.getPath("path", new String[0]), StandardOpenOption.READ);
        byte[] buffer = new byte["contents".length()];
        in.read(buffer);
        in.close();
        String roundtrip = new String(buffer);
        Assertions.assertEquals((Object)"contents", (Object)roundtrip, (String)"Contents not read as expected from the instream");
    }

    @Test
    public void newOutputStream() throws IOException {
        String path = "path";
        String contents = "contents";
        OutputStream outStream = Files.newOutputStream(this.fs.getPath("path", new String[0]), StandardOpenOption.WRITE);
        outStream.write("contents".getBytes());
        outStream.close();
        String roundtrip = new BufferedReader(new InputStreamReader(this.getArchive().get("path").getAsset().openStream())).readLine();
        Assertions.assertEquals((Object)"contents", (Object)roundtrip, (String)"Contents not read as expected from the outstream");
    }

    @Test
    public void notExistsTrue() {
        Assertions.assertTrue((boolean)Files.notExists(this.fs.getPath("fake", new String[0]), LinkOption.NOFOLLOW_LINKS));
    }

    @Test
    public void notExistsFalse() {
        this.getArchive().add((Asset)EmptyAsset.INSTANCE, "path");
        Assertions.assertFalse((boolean)Files.notExists(this.fs.getPath("path", new String[0]), LinkOption.NOFOLLOW_LINKS));
    }

    @Test
    public void probeContentType() throws IOException {
        String path = "path";
        String contents = "contents";
        this.getArchive().add((Asset)new StringAsset("contents"), "path");
        Assertions.assertNull((Object)Files.probeContentType(this.fs.getPath("path", new String[0])));
    }

    @Test
    public void readAllBytes() throws IOException {
        String path = "path";
        String contents = "contents";
        this.getArchive().add((Asset)new StringAsset("contents"), "path");
        byte[] bytes = Files.readAllBytes(this.fs.getPath("path", new String[0]));
        String roundtrip = new String(bytes);
        Assertions.assertEquals((Object)"contents", (Object)roundtrip, (String)"Contents not read as expected from the readAllBytes");
    }

    @Test
    public void createdDirectoryIsADirectory() throws Exception {
        Path dirPath = this.fs.getPath("dir", new String[0]);
        Files.createDirectory(dirPath, new FileAttribute[0]);
        Assertions.assertTrue((boolean)Files.isDirectory(dirPath, new LinkOption[0]), (String)"Created directory was not a directory");
    }

    @Test
    public void createdDirectoryCanBeStreamed() throws Exception {
        Path dirPath = this.fs.getPath("dir", new String[0]);
        Files.createDirectory(dirPath, new FileAttribute[0]);
        Files.createFile(dirPath.resolve("file"), new FileAttribute[0]);
        DirectoryStream<Path> stream = Files.newDirectoryStream(dirPath);
        Iterator<Path> it = stream.iterator();
        Assertions.assertEquals((Object)"/dir/file", (Object)it.next().toString());
        Assertions.assertFalse((boolean)it.hasNext(), (String)"No further elements expected in stream");
    }

    @Test
    public void createdDirectoryStreamThrowsExceptionWhenIsClosed() throws Throwable {
        Path dirPath = this.fs.getPath("dir", new String[0]);
        Files.createDirectory(dirPath, new FileAttribute[0]);
        Files.createFile(dirPath.resolve("file"), new FileAttribute[0]);
        DirectoryStream<Path> stream = Files.newDirectoryStream(dirPath);
        stream.close();
        Assertions.assertThrows(IllegalStateException.class, stream::iterator);
    }

    @Test
    public void createdDirectoryStreamThrowsExceptionWhenIteratorWasReturnedBefore() throws Exception {
        Path dirPath = this.fs.getPath("dir", new String[0]);
        Files.createDirectory(dirPath, new FileAttribute[0]);
        Files.createFile(dirPath.resolve("file"), new FileAttribute[0]);
        DirectoryStream<Path> stream = Files.newDirectoryStream(dirPath);
        Iterator<Path> it = stream.iterator();
        Assertions.assertThrows(IllegalStateException.class, stream::iterator);
    }

    @Test
    public void directoryStreamDoesNotContainSubfolders() throws Exception {
        Files.createDirectories(this.fs.getPath("dir/subdir", new String[0]), new FileAttribute[0]);
        DirectoryStream<Path> stream = Files.newDirectoryStream(this.fs.getPath("/", new String[0]));
        Iterator<Path> it = stream.iterator();
        Assertions.assertEquals((Object)"/dir", (Object)it.next().toString());
        Assertions.assertFalse((boolean)it.hasNext(), (String)"No further elements expected in stream");
    }

    @Test
    public void createdDirectoryCanBeWalked() throws Exception {
        Path dirPath = this.fs.getPath("dir", new String[0]);
        Files.createDirectory(dirPath, new FileAttribute[0]);
        Files.createFile(dirPath.resolve("file"), new FileAttribute[0]);
        final int[] visitFileCalled = new int[1];
        Files.walkFileTree(dirPath, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                visitFileCalled[0] = visitFileCalled[0] + 1;
                return super.visitFile(file, attrs);
            }
        });
        Assertions.assertEquals((int)1, (int)visitFileCalled[0]);
    }

    private Archive<?> getArchive() {
        ShrinkWrapFileSystem swfs = (ShrinkWrapFileSystem)this.fs;
        Archive archive = swfs.getArchive();
        return archive;
    }
}

