/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.bpel.tests.samples;

import junit.framework.Test;

import org.jboss.soa.bpel.tests.RiftSawTest;
import org.jboss.soa.bpel.tests.RiftSawTestSetup;
import org.jboss.soa.bpel.tests.samples.atm.AtmService;
import org.jboss.soa.bpel.tests.samples.atm.FrontEnd;
import org.jboss.soa.bpel.tests.samples.atm.InsufficientFunds;
import org.jboss.soa.bpel.tests.samples.atm.InsufficientFunds_Exception;
import org.jboss.soa.bpel.tests.samples.atm.UnauthorizedAccess_Exception;

/**
 * Test for common ATM usage scenarios.
 * 
 * @author Juan Cantu
 * @author Alejandro Guizar
 */
public class TutorialBPELAtmTestCase extends RiftSawTest {

	private FrontEnd frontEnd;

	private static final String TEST_NAME = "Tutorial_bpel_atm";

	public TutorialBPELAtmTestCase() {
		super(TEST_NAME);
	}

	public static Test suite() {
		return new RiftSawTestSetup(TutorialBPELAtmTestCase.class, TEST_NAME,
				"Tutorial_bpel_atm.war,Tutorial_bpel_atm-1.jar");
	}

	protected void setUp() throws Exception {
		AtmService frontEndService = new AtmService();

		// obtain dynamic proxy for web service port
		frontEnd = frontEndService.getFrontEndPort();
	}

	public void testConnect() {
		// connect to bank
		int ticketNumber = frontEnd.connect();
		assertTrue(ticketNumber > 0);

		// check atm is connected
		String status = frontEnd.status(ticketNumber);
		assertEquals("connected", status);

		// disconnect from bank
		frontEnd.disconnect(ticketNumber);
	}

	public void testLogOnAuthorized() {
		// connect to bank
		int ticketNumber = frontEnd.connect();

		// begin customer session
		final String customerName = "koen";
		try {
			frontEnd.logOn(ticketNumber, customerName);
		}
		catch (UnauthorizedAccess_Exception e) {
			fail("log on of authorized customer should succeed");
		}

		// end customer session
		frontEnd.logOff(customerName);

		// disconnect from bank
		frontEnd.disconnect(ticketNumber);
	}

	public void testLogOnUnauthorized() {
		// connect to bank
		int ticketNumber = frontEnd.connect();

		// begin customer session
		final String customerName = "nobody";
		try {
			frontEnd.logOn(ticketNumber, customerName);
			fail("log on of unauthorized customer should fail");
		}
		catch (UnauthorizedAccess_Exception e) {
			assertEquals(customerName, e.getFaultInfo().getCustomerName());
		}

		// disconnect from bank
		frontEnd.disconnect(ticketNumber);
	}

	public void testDeposit() throws UnauthorizedAccess_Exception {
		// connect to bank
		int ticketNumber = frontEnd.connect();

		// begin customer session
		final String customerName = "tom";
		frontEnd.logOn(ticketNumber, customerName);

		// get current balance
		double previousBalance = frontEnd.getBalance(customerName);

		// deposit some funds
		double newBalance = frontEnd.deposit(customerName, 10);
		// check the new balance is correct
		assertEquals(previousBalance + 10, newBalance, 0);

		// end customer session
		frontEnd.logOff(customerName);

		// disconnect from bank
		frontEnd.disconnect(ticketNumber);
	}

	public void testWithdrawUnderBalance() throws UnauthorizedAccess_Exception {
		// connect to bank
		int ticketNumber = frontEnd.connect();

		// begin customer session
		final String customerName = "tom";
		frontEnd.logOn(ticketNumber, customerName);

		// get current balance
		double previousBalance = frontEnd.getBalance(customerName);

		// withdraw some funds
		try {
			double newBalance = frontEnd.withdraw(customerName, 10);
			// check new balance is correct
			assertEquals(previousBalance - 10, newBalance, 0);
		}
		catch (InsufficientFunds_Exception e) {
			fail("withdraw under balance should succeed");
		}

		// end customer session
		frontEnd.logOff(customerName);

		// disconnect from bank
		frontEnd.disconnect(ticketNumber);
	}

	public void testWithdrawOverBalance() throws UnauthorizedAccess_Exception {
		// connect to bank
		int ticketNumber = frontEnd.connect();

		// begin customer session
		final String customerName = "fady";
		frontEnd.logOn(ticketNumber, customerName);

		// get current balance
		double previousBalance = frontEnd.getBalance(customerName);

		// try to withdraw an amount greater than current balance
		try {
			frontEnd.withdraw(customerName, previousBalance + 1);
			fail("withdraw over balance should fail");
		}
		catch (InsufficientFunds_Exception e) {
			InsufficientFunds faultInfo = e.getFaultInfo();
			assertEquals(customerName, faultInfo.getCustomerName());
			// check account balance has not changed
			assertEquals(previousBalance, faultInfo.getAmount(), 0);
		}

		// end customer session
		frontEnd.logOff(customerName);

		// disconnect from bank
		frontEnd.disconnect(ticketNumber);
	}
}