/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.bpel.runtime.ws;

import javax.wsdl.*;
import javax.wsdl.extensions.soap.SOAPAddress;
import javax.xml.namespace.QName;
import java.net.URL;
import java.util.Collection;
import java.util.List;
import java.util.Map;

/**
 * WSDL helper class
 */
public final class WSDLParser
{

  private Definition wsdlDefinition;
  private int dfsDepth = 0;
  
  public WSDLParser(Definition wsdlDefinition)
  {
    this.wsdlDefinition = wsdlDefinition;
  }

  public void reset()
  {
    dfsDepth = 0;
  }

  public Operation getRPCOperation(QName service, String port, QName payloadName)
  {
    reset();
    Operation result = _getRPCOperation(this.wsdlDefinition, service, port, payloadName);
    if (result == null)
      throw new IllegalArgumentException("Unable to find operation for element "+payloadName);
    return result;
  }

  @SuppressWarnings("unchecked")
  private Operation _getRPCOperation(Definition wsdl, QName serviceName, String portName, QName payloadName)
  {
    if(++dfsDepth>50) // easier then retaining references
      throw new IllegalStateException("Recursive loop detected. DFS depth reached limit");

    // namespace / java.util.List of imports
    Collection<List<Import>> imports = wsdl.getImports().values();
    for(List<Import> importNS : imports)
    {
      for(Import wsdlImport : importNS)
      {
        Operation result = _getRPCOperation(wsdlImport.getDefinition(), serviceName, portName, payloadName);
        if(result!=null)
          return result;
      }
    }

    // resolve the port & operation
    Service service = wsdl.getService(serviceName);
    if (service == null)
      return null; // service might be defined elsewhere

    Port port = service.getPort(portName);
    if (port == null)
      throw new IllegalArgumentException("service "+serviceName+" contains no such port "+portName);

    Binding binding = port.getBinding();
    PortType portType = binding.getPortType();
    List<Operation> operations = portType.getOperations();
    for(Operation op : operations)
    {
        if(op.getName().equals(payloadName.getLocalPart()))
          return op;
    }

    return null;
  }

  public Operation getDocLitOperation(QName service, String port, QName payloadName)
  {
    reset();
    Operation result = _getDocLitOperation(this.wsdlDefinition, service, port, payloadName);
    if (result == null)
      throw new IllegalArgumentException("Unable to find operation for element "+payloadName);
    return result;
  }

  @SuppressWarnings("unchecked")
  private Operation _getDocLitOperation(Definition wsdl, QName serviceName, String portName, QName payloadName)
  {
    if(++dfsDepth>50) // easier then retaining references
      throw new IllegalStateException("Recursive loop detected. DFS depth reached limit");

    // namespace / java.util.List of imports
    Collection<List<Import>> imports = wsdl.getImports().values();
    for(List<Import> importNS : imports)
    {
      for(Import wsdlImport : importNS)
      {
        Operation result = _getDocLitOperation(wsdlImport.getDefinition(), serviceName, portName, payloadName);
        if(result!=null)
          return result;
      }
    }

    // resolve the port & operation
    Service service = wsdl.getService(serviceName);
    if (service == null)
      return null; // service might be defined elsewhere

    Port port = service.getPort(portName);
    if (port == null)
      throw new IllegalArgumentException("service "+serviceName+" contains no such port "+portName);

    Binding binding = port.getBinding();
    PortType portType = binding.getPortType();
    List<Operation> operations = portType.getOperations();
    for(Operation op : operations)
    {
      Message message = op.getInput().getMessage();
      Collection<Part> parts = message.getParts().values();
      for(Part part : parts)
      {
        if(part.getElementName().equals(payloadName))
          return op;
      }
    }

    return null;
  }

  public URL getServiceLocationURL(QName serviceQName, String portName)
  {
    reset();
    return _getServiceLocationURL(this.wsdlDefinition, serviceQName, portName);
  }

  public URL _getServiceLocationURL(Definition wsdl, QName serviceQName, String portName)
  {
    URL match = null;
    dfsDepth++;

    if(dfsDepth>50) // easier then retaining references
        throw new IllegalStateException("Recursive loop detected. DFS depth reached limit");

    // namespace / java.util.List of imports
    Map<String, List<Import>> imports = wsdl.getImports();
    for(String ns : imports.keySet())
    {
      List<Import> importNS = imports.get(ns);
      for(Import wsdlImport : importNS)
      {
        URL result = _getServiceLocationURL(wsdlImport.getDefinition(), serviceQName, portName);
        if(result!=null)
        {
          match = result;
          break;
        }
      }

      if(match!=null) break;
    }

    if(match!=null) // DFS results
        return match;

    try
    {
      Service service = wsdl.getService(serviceQName);
      Port port = null;
      SOAPAddress soapAddress = null;

      // --

      if(service!=null)
      {
        port = service.getPort(portName);
        if(port!=null)
        {
          for(Object obj : port.getExtensibilityElements())
          {
            if(obj instanceof SOAPAddress)
            {
              soapAddress = (SOAPAddress)obj;
            }
          }

        }
      }

      // --

      if(soapAddress!=null)
        match = new URL(soapAddress.getLocationURI());
    }
    catch (Exception e)
    {
      throw new RuntimeException("Failed to parse " + wsdl, e);
    }

    return match;
  }
}
