/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.bpel.uddi300;

import java.rmi.RemoteException;
import java.util.Map;
import java.util.Properties;

import javax.xml.namespace.QName;

import org.apache.commons.configuration.ConfigurationException;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.juddi.api_v3.AccessPointType;
import org.apache.juddi.v3.annotations.AnnotationProcessor;
import org.apache.juddi.v3.client.config.TokenResolver;
import org.apache.juddi.v3.client.config.UDDIClerk;
import org.apache.juddi.v3.client.config.UDDIClerkManager;
import org.apache.juddi.v3.client.transport.TransportException;
import org.jboss.soa.bpel.runtime.engine.ode.UDDIRegistration;
import org.jboss.soa.bpel.runtime.ws.EndpointMetaData;
import org.jboss.soa.bpel.runtime.ws.WSDLReference;
import org.uddi.api_v3.AccessPoint;
import org.uddi.api_v3.BindingTemplate;
import org.uddi.api_v3.BindingTemplates;
import org.uddi.api_v3.BusinessService;
import org.uddi.api_v3.CategoryBag;
import org.uddi.api_v3.Description;
import org.uddi.api_v3.Name;
import org.uddi.api_v3.TModelInstanceDetails;
import org.uddi.api_v3.TModelInstanceInfo;
import org.uddi.v3_service.DispositionReportFaultMessage;

/**
 * 
 * * @author Kurt T Stam <kurt.stam@jboss.com>
 *
 */
public class UDDI300RegistrationImpl extends AnnotationProcessor implements UDDIRegistration {

	protected static final Log log = LogFactory.getLog(UDDI300RegistrationImpl.class);
	@SuppressWarnings("unused")
	private Properties properties = new Properties();
	private final static String BPEL_UDDI_CLERK = "bpel.uddi.clerk";
	private final static String DEFAULT_BPEL_UDDI_CLERK = "BPELClerk";
	public static final String DEFAULT_SERVICE_KEY_FORMAT   = "uddi:${keyDomain}:bpel-services-";
	public static final String DEFAULT_BINDING_KEY_FORMAT   = "uddi:${keyDomain}:bindings-";
	//Can be set in the uddi.xml property section
	public static final String LANG                         = "lang";
	public static final String SERVICE_DESCRIPTION          = "serviceDescription";
	public static final String SERVICE_CATEGORY_BAG         = "serviceCategoryBag";
	public static final String BINDING_DESCRIPTION          = "bindingDescription";
	public static final String BINDING_CATEGORY_BAG         = "bindingCategoryBag";
	public static final String BINDING_TMODEL_KEYS          = "bindingTModelKeys";
	public static final String SERVICE_KEY_FORMAT           = "serviceKeyFormat";
	public static final String BINDING_KEY_FORMAT           = "bindingKeyFormat";
	
	public UDDI300RegistrationImpl(Properties properties) {
			super();
			this.properties = properties;
	}
	 
	private String getClerkName() {
	      return (properties.getProperty(BPEL_UDDI_CLERK, DEFAULT_BPEL_UDDI_CLERK));
	}
	/**
	 * Registers a BPEL ServiceEndpointReference (EPR) into a UDDI registry using the jUDDI client code.
	 * If the serviceKey does not already exist we register the service along with the EPR.
	 * 
	 * @param EPR
	 * @param metaData
	 * @param wsdlRef
	 */
	public void registerEPR(EndpointMetaData metaData, WSDLReference wsdlRef) {
		boolean selfRegister = false;
		try {
			selfRegister = UDDIClerkManager.getClientConfig().isRegisterOnStartup();
			if (selfRegister) {
				Map<String,UDDIClerk> clerks = UDDIClerkManager.getClientConfig().getUDDIClerks();
				UDDIClerk bpelClerk = clerks.get(getClerkName());
				if (bpelClerk!=null) {
					try {
						BusinessService service = createBusinessService(bpelClerk, metaData, wsdlRef);
						bpelClerk.register(service, bpelClerk.getUDDINode().getApiNode());
					} catch (Exception e) {
						log.error("Unable to register service " + metaData.getServiceName()
								+ " ." + e.getMessage(),e);
					} catch (Throwable t) {
						log.error("Unable to register service " + metaData.getServiceName()
								+ " ." + t.getMessage(),t);
					}
				} else {
					throw new ConfigurationException("Could not find UDDI Clerk named "+ getClerkName());
				}
			}
		} catch (ConfigurationException ce) {
			if (selfRegister) {
				log.error(ce.getMessage(),ce);
			} else {
				log.info(ce.getMessage());
			}
		}
	}
	/**
	 * UnRegisters the binding from the UDDI Registry. 
	 * @param EPR
	 * @param metaData
	 * @param wsdlRef
	 */
	public void unRegisterEPR(QName service, String port) {
		boolean selfRegister = false;
		try {
			selfRegister = UDDIClerkManager.getClientConfig().isRegisterOnStartup();
			if (selfRegister) {
				Map<String,UDDIClerk> clerks = UDDIClerkManager.getClientConfig().getUDDIClerks();
				UDDIClerk bpelClerk = clerks.get(getClerkName());
				String bindingName = service.getLocalPart() + "-" + port;
				if (bpelClerk!=null) {
					try {
						Properties properties = bpelClerk.getUDDINode().getProperties();
						//Constructing the bindingKey
						String bindingKey = TokenResolver.replaceTokens(DEFAULT_BINDING_KEY_FORMAT + bindingName, properties);
						if (properties.containsKey(BINDING_KEY_FORMAT)) {
							bindingKey = TokenResolver.replaceTokens((String)properties.get(BINDING_KEY_FORMAT) + bindingName, properties);
						}
						bindingKey = bindingKey.toLowerCase();
						//Lookup the binding
						BindingTemplate binding = null;
						try {
							binding = bpelClerk.findServiceBinding(bindingKey, bpelClerk.getUDDINode().getApiNode());
							//TODO Kurt: Just delete this binding -not all bindings- upgrade the jUDDI client to support this.
							BusinessService businessService = bpelClerk.findService(binding.getServiceKey(), bpelClerk.getUDDINode().getApiNode());
							bpelClerk.unRegister(businessService, bpelClerk.getUDDINode().getApiNode());
						} catch (Exception e) {
							log.warn("Could not find BindingTemplate with key " + bindingKey + " for unRegistration.");
						}
					} catch (Exception e) {
						log.error("Unable to unRegister EPR " + bindingName
								+ " ." + e.getMessage(),e);
					} catch (Throwable t) {
						log.error("Unable to unRegister EPR " + bindingName
								+ " ." + t.getMessage(),t);
					}
				} else {
					throw new ConfigurationException("Could not find UDDI Clerk named "+ getClerkName());
				}
			}
		} catch (ConfigurationException ce) {
			if (selfRegister) {
				log.error(ce.getMessage(),ce);
			} else {
				log.info(ce.getMessage());
			}
		}
	}
	
	private BusinessService createBusinessService(UDDIClerk clerk,
			EndpointMetaData metaData, WSDLReference wsdlRef) 
		throws DispositionReportFaultMessage, RemoteException, ConfigurationException, TransportException {
		Properties properties = clerk.getUDDINode().getProperties();
		//Constructing the serviceKey
		String serviceKey = TokenResolver.replaceTokens(DEFAULT_SERVICE_KEY_FORMAT + metaData.getServiceName().getLocalPart(), properties);
		if (properties.containsKey(SERVICE_KEY_FORMAT)) {
			serviceKey = TokenResolver.replaceTokens((String)properties.get(SERVICE_KEY_FORMAT) + metaData.getServiceName(), properties);
		}
		BusinessService service = null;
		try {
			//Checking if this serviceKey already exist
			service = clerk.findService(serviceKey, clerk.getUDDINode().getApiNode());
			log.debug("Service " + metaData.getServiceName() + " already present in the UDDI Registry");
		} catch (Exception e) {
			//If it does not exist construct service information
			log.debug("Constructing Service UDDI Information for " + metaData.getServiceName());
			service = new BusinessService();
			service.setBusinessKey(TokenResolver.replaceTokens("uddi:${keyDomain}:${businessKey}", properties));
			service.setServiceKey(serviceKey);
			if (properties.containsKey(SERVICE_DESCRIPTION)) {
				Description description = new Description();
				String lang = "en";
				if (properties.containsKey(LANG)) lang = properties.getProperty(LANG);
				description.setLang(lang);
				description.setValue(properties.getProperty(SERVICE_DESCRIPTION));
				Name serviceName = new Name();
				serviceName.setLang(lang);
				serviceName.setValue(metaData.getServiceName().getLocalPart());
				service.getName().add(serviceName);
				service.getDescription().add(description);
			}
			//default categoryBag on the binding
			if (properties.containsKey(SERVICE_CATEGORY_BAG)) {
				String defaultCategoryBag = properties.getProperty(SERVICE_CATEGORY_BAG);
				log.info("Adding categoryBag: " + defaultCategoryBag);
				CategoryBag categoryBag = parseCategoryBag(defaultCategoryBag);
		        service.setCategoryBag(categoryBag);
			}
		}
		//Construct bindingTemplate
		BindingTemplate binding = createBindingTemplate(clerk, metaData, wsdlRef);
		//Add the bindingTemplate on the service
		if (service.getBindingTemplates()==null) {
			BindingTemplates bindingTemplates = new BindingTemplates();
			service.setBindingTemplates(bindingTemplates);
		}
		service.getBindingTemplates().getBindingTemplate().add(binding);
		return service;
	}
	
	private BindingTemplate createBindingTemplate(UDDIClerk clerk,
			EndpointMetaData metaData, WSDLReference wsdlRef) {
		
		Properties properties = clerk.getUDDINode().getProperties();
		BindingTemplate bindingTemplate = new BindingTemplate();
		//Constructing the bindingKey
		String bindingKey = TokenResolver.replaceTokens(DEFAULT_BINDING_KEY_FORMAT + metaData.getServiceName().getLocalPart() + "-"
				+ metaData.getPortName(), properties);
		if (properties.containsKey(BINDING_KEY_FORMAT)) {
			bindingKey = TokenResolver.replaceTokens((String)properties.get(BINDING_KEY_FORMAT) + metaData.getServiceName().getLocalPart() + "-"
					+ metaData.getPortName(), properties);
		}
		
		bindingTemplate.setBindingKey(bindingKey);
		
		String lang = "en";
		if (properties.containsKey(LANG)) lang = properties.getProperty(LANG);
		if (properties.containsKey(BINDING_DESCRIPTION)) {
			Description bindingDescription = new Description();
			bindingDescription.setLang(lang);
			bindingDescription.setValue(properties.getProperty(BINDING_DESCRIPTION));
			bindingTemplate.getDescription().add(bindingDescription);
		}
		
		AccessPoint accessPoint = new AccessPoint();
		accessPoint.setUseType(AccessPointType.WSDL_DEPLOYMENT.toString());
		accessPoint.setValue(wsdlRef.getWsdlFileURL().toString());
		bindingTemplate.setAccessPoint(accessPoint);
		
		//default tModelKeys on the binding
		String defaultTModelKeys = properties.getProperty("tModelKeys");
		if (defaultTModelKeys!=null &&  !"".equals(defaultTModelKeys)) {
			String[] tModelKeys= defaultTModelKeys.split(",");
			for (String tModelKey : tModelKeys) {
				TModelInstanceInfo instanceInfo = new TModelInstanceInfo();
				instanceInfo.setTModelKey(tModelKey);
				if (bindingTemplate.getTModelInstanceDetails()==null) {
					bindingTemplate.setTModelInstanceDetails(new TModelInstanceDetails());
				}
				bindingTemplate.getTModelInstanceDetails().getTModelInstanceInfo().add(instanceInfo);
			}
		}
		//default categoryBag on the binding
		String defaultCategoryBag = properties.getProperty("bindingCategoryBag");
		if (defaultCategoryBag!=null &&  !"".equals(defaultCategoryBag)) {
			log.info("Adding categoryBag: " + defaultCategoryBag);
			CategoryBag categoryBag = parseCategoryBag(defaultCategoryBag);
	        bindingTemplate.setCategoryBag(categoryBag);
		}
		
		return bindingTemplate;
	}
	
	
	
}
