/*
 * JBoss, Home of Professional Open Source
 * Copyright 2009, Red Hat Middleware LLC, and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 */
package org.jboss.soa.bpel.runtime.engine.service;

import org.jboss.logging.Logger;
import org.jboss.soa.bpel.runtime.engine.BPELEngine;
import org.jboss.soa.bpel.runtime.engine.BPELEngineFactory;
import org.jboss.util.naming.NonSerializableFactory;

import javax.naming.*;

/**
 * This is the JBoss Service managing the BPEL engine.
 *
 * @author gbrown
 *
 */
public class BPELEngineService extends org.jboss.system.ServiceMBeanSupport implements BPELEngineServiceMBean {

  private final static Logger logger = Logger.getLogger(BPELEngineService.class);

  private String m_jndiName=null;
  private String m_pmapiJndiName=null;
  private BPELEngine m_bpelEngine=null;
  
  /**
   * The default constructor.
   */
  public BPELEngineService() {
  }

  /**
   * This method starts the service, instantiating the
   * BPEL engine and registering it with JNDI.
   */
  protected void startService() throws Exception {
    logger.info("Starting JBoss BPEL Engine");

    m_bpelEngine = BPELEngineFactory.getInstance();

    rebind();

    m_bpelEngine.init();

    rebindPMAPI();
  }

  /**
   * This method stops the service, closing the
   * BPEL engine and unregistering it from JNDI.
   */
  protected void stopService() throws Exception {
    logger.info("Stopping JBoss BPEL Engine");

    unbind(m_jndiName);

    m_bpelEngine.close();

    m_bpelEngine = null;
  }

  /**
   * This method returns the JNDI name used to register the
   * BPEL engine.
   */
  public String getJndiName() {
    return m_jndiName;
  }

  /**
   * This method sets the JNDI name associated with the
   * BPEL engine.
   *
   * @param jndiName The JNDI name
   * @throws NamingException Failed to register BPEL engine against
   * 					the JNDI name
   */
  public void setJndiName(String jndiName) throws NamingException {
    String oldName = m_jndiName;
    m_jndiName = jndiName;

    if (super.getState() == STARTED) {
      unbind(oldName);
      try {
        rebind();
      } catch(Exception e) {
        NamingException ne = new NamingException("Failed to update jndiName");
        ne.setRootCause(e);
        throw ne;
      }
    }
  }

  /**
   * This method returns the JNDI name used to register the
   * BPEL Process Management API.
   */
  public String getPmapiJndiName() {
    return m_jndiName;
  }

  /**
   * This method sets the JNDI name associated with the
   * BPEL Process Management API.
   *
   * @param jndiName The JNDI name
   * @throws NamingException Failed to register BPEL Process Management API against
   * 					the JNDI name
   */
  public void setPmapiJndiName(String jndiName) throws NamingException {
    String oldName = m_pmapiJndiName;
    m_pmapiJndiName = jndiName;

    if (super.getState() == STARTED) {
    
      if (oldName != null) {
    	  unbind(oldName);
      }
      
      try {
        rebindPMAPI();
      } catch(Exception e) {
        NamingException ne = new NamingException("Failed to update PMAPI jndiName");
        ne.setRootCause(e);
        throw ne;
      }
    }
  }

  private void rebind() throws NamingException {
    InitialContext rootCtx = new InitialContext();
    Name fullName = rootCtx.getNameParser("").parse(m_jndiName);
    logger.info("fullName="+fullName);
    NonSerializableFactory.rebind(fullName, m_bpelEngine, true);
  }

  private void unbind(String jndiName) {
    try {
      InitialContext rootCtx = new InitialContext();
      rootCtx.unbind(jndiName);
      NonSerializableFactory.unbind(jndiName);
    } catch(NamingException e) {
      logger.error("Failed to unbind map", e);
    }
  }

  private void rebindPMAPI() throws NamingException {
	if (m_pmapiJndiName != null) {
	    InitialContext rootCtx = new InitialContext();
	    Name fullName = rootCtx.getNameParser("").parse(m_pmapiJndiName);
	    logger.info("pmapi fullName="+fullName);
	    NonSerializableFactory.rebind(fullName, m_bpelEngine.getManagementInterface(), true);
	}
  }
}
