package org.jboss.testharness.impl.packaging;

import static org.jboss.testharness.impl.packaging.PackagingType.EAR;
import static org.jboss.testharness.impl.packaging.PackagingType.WAR;

import java.io.File;
import java.io.FilenameFilter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.apache.log4j.Logger;
import org.jboss.testharness.api.Configuration;
import org.jboss.testharness.api.ResourceDescriptor;
import org.jboss.testharness.impl.packaging.ear.EarArtifactDescriptor;
import org.jboss.testharness.impl.packaging.ear.EjbJarXml;
import org.jboss.testharness.impl.packaging.jsr299.BeansXml;
import org.jboss.testharness.impl.packaging.jsr299.JSR299ArtifactDescriptor;
import org.jboss.testharness.impl.packaging.jsr299.TCKArtifactDescriptor;
import org.jboss.testharness.impl.packaging.war.WarArtifactDescriptor;

public class ArtifactGenerator
{
   
   private static class ArtifactProcessor
   {
      
      private static final Logger log = Logger.getLogger(ArtifactProcessor.class);
      
      private final boolean unit;
      private final boolean runLocally;
      private final boolean addDeclaringPackage;
      private final String beansXml;
      private final String ejbJarXml;
      private final PackagingType packagingType;
      private final Collection<ResourceDescriptor> resources;
      private final Collection<Class<?>> classes;
      private final Class<? extends Throwable> expectedDeploymentException;
      private final Set<ResourceDescriptor> extraLibraries;
      private final String[] packages;
      
      
      private final Class<?> declaringClass;
      private final boolean standalone;

      public ArtifactProcessor(Class<?> declaringClass, boolean standalone, String extraLibrariesDirectory)
      {
         this.standalone = standalone;
         this.declaringClass = declaringClass;
         
         if (declaringClass.isAnnotationPresent(Artifact.class))
         {
            this.addDeclaringPackage = declaringClass.getAnnotation(Artifact.class).addCurrentPackage();
         }
         else
         {
            throw new IllegalStateException("Unable to find @Artifact on " + declaringClass);
         }
         
         if (declaringClass.isAnnotationPresent(BeansXml.class))
         {
            this.beansXml = asAbsolutePath(declaringClass.getAnnotation(BeansXml.class).value());
         }
         else
         {
            this.beansXml = null;
         }
         
         if (declaringClass.isAnnotationPresent(Packaging.class))
         {
            this.packagingType = declaringClass.getAnnotation(Packaging.class).value();
         }
         else
         {
            this.packagingType = WAR;
         }
         
         if (declaringClass.isAnnotationPresent(EjbJarXml.class))
         {
            this.ejbJarXml = asAbsolutePath(declaringClass.getAnnotation(EjbJarXml.class).value());	
         }
         else
         {
            this.ejbJarXml = null;
         }
        	 
         
         if (declaringClass.isAnnotationPresent(IntegrationTest.class))
         {
            this.unit = false;
            this.runLocally = declaringClass.getAnnotation(IntegrationTest.class).runLocally(); 
         }
         else
         {
            this.unit = true;
            this.runLocally = false;
         }
         
         if (declaringClass.isAnnotationPresent(Resources.class))
         {
            this.resources = asResourceDescriptors(declaringClass.getAnnotation(Resources.class).value());
         }
         else
         {
            this.resources = Collections.emptyList();
         }
         
         if (declaringClass.isAnnotationPresent(Classes.class))
         {
            this.classes = Arrays.asList(declaringClass.getAnnotation(Classes.class).value());
            this.packages = declaringClass.getAnnotation(Classes.class).packages();
         }
         else
         {
            this.classes = Collections.emptyList();
            this.packages = new String[0];
         }
         if (declaringClass.isAnnotationPresent(ExpectedDeploymentException.class))
         {
            this.expectedDeploymentException = declaringClass.getAnnotation(ExpectedDeploymentException.class).value();
         }
         else
         {
            this.expectedDeploymentException = null;
         }
         if (extraLibrariesDirectory != null)
         {
            File directory = new File(extraLibrariesDirectory);
            this.extraLibraries = new HashSet<ResourceDescriptor>();
            if (directory.isDirectory())
            {
               for (File file : directory.listFiles(new FilenameFilter()
                  {
                     
                     public boolean accept(File dir, String name)
                     {
                        return name.endsWith(".jar");
                     }
                     
                  }))
               {
                  try
                  {
                     this.extraLibraries.add(new ResourceDescriptorImpl(file.getName(), file.toURI().toURL()));
                  }
                  catch (IOException e)
                  {
                     log.warn("Unable to load extra library", e);
                  }
               }
            }
         }
         else
         {
            this.extraLibraries = Collections.emptySet();
         }
      }
      
      public TCKArtifactDescriptor createArtifact()
      {
         final TCKArtifactDescriptor artifact = newArtifact(packagingType, declaringClass, beansXml, ejbJarXml, standalone, addDeclaringPackage);
         artifact.setUnit(unit);
         artifact.setRunLocally(runLocally);
         artifact.setExpectedDeploymentException(expectedDeploymentException);
         artifact.getClasses().addAll(classes);
         // Annoying hack
         artifact.getResources().removeAll(resources);
         artifact.getResources().addAll(resources);
         artifact.getLibraries().addAll(extraLibraries);
         for (String packageName : packages)
         {
            artifact.addPackage(packageName, false);
         }
         return artifact;
      }
      
      private Collection<ResourceDescriptor> asResourceDescriptors(Resource[] resources)
      {
         List<ResourceDescriptor> resourceDescriptorImpls = new ArrayList<ResourceDescriptor>();
         for (Resource resource : resources)
         {
            resourceDescriptorImpls.add(new ResourceDescriptorImpl(resource.destination(), asAbsolutePath(resource.source())));
         }
         return resourceDescriptorImpls;
      }
      
      private String asAbsolutePath(String path)
      {
         if (path.startsWith("/"))
         {
            return path.substring(1);
         }
         else
         {
            return declaringClass.getPackage().getName().replace(".", "/") + "/" + path;
         }
      }
      
      private static TCKArtifactDescriptor newArtifact(PackagingType packagingType, Class<?> declaringClass, String beansXml, String ejbJarXml, boolean standalone, boolean addDeclaringPackage)
      {
         TCKArtifactDescriptor artifact;
         if (!standalone && packagingType.equals(WAR))
         {
               artifact = new WarArtifactDescriptor(declaringClass, beansXml).initialize();
               if (ejbJarXml != null)
               {
                  artifact.getResources().add(new ResourceDescriptorImpl("WEB-INF/ejb-jar.xml", ejbJarXml));
               }
         } 
         else if (!standalone && packagingType.equals(EAR))
         {
            artifact = new EarArtifactDescriptor(declaringClass, beansXml, ejbJarXml).initialize();
         }
         else
         {
            artifact = new JSR299ArtifactDescriptor(declaringClass, beansXml).initialize();
         }
         
         if (addDeclaringPackage)
         {
            artifact.addPackage(declaringClass.getPackage());
         }
         return artifact;
      }
      
      public boolean isUnit()
      {
         return unit;
      };
      
   }
   
   private static final Logger log = Logger.getLogger(ArtifactGenerator.class);
   
   private final Configuration configuration;
   
   public ArtifactGenerator(Configuration configuration)
   {
      if (configuration == null)
      {
         throw new IllegalArgumentException("configuration must not be null");
      }
      this.configuration = configuration;
   }

   public List<ArtifactDescriptor> createArtifacts(String packageName)
   {
      Set<Class<?>> classes = new HashSet<Class<?>>();
      classes.addAll(new ArtifactScanner(packageName, null).getClasses());
      List<ArtifactDescriptor> artifacts = new ArrayList<ArtifactDescriptor>();
      for (Class<?> clazz : classes)
      {
         artifacts.add(createArtifact(clazz));
      }
      return artifacts;
   }
   
   public void dumpArtifacts(String basePackageName) 
   {
      if (basePackageName == null)
      {
         throw new IllegalArgumentException("Cannot dump artifacts as no package to dump from set. Specify org.jboss.testharness.testPackage");
      }
      List<ArtifactDescriptor> artifacts = createArtifacts(basePackageName);
      File file = new File(configuration.getOutputDirectory());
      if (!file.exists())
      {
         file.mkdirs();
      }
      else if (file.isFile())
      {
         throw new IllegalStateException("Cannot use debug directory " + configuration.getOutputDirectory() + ", it already exists");
      }
      log.info("Writing artifacts to " + configuration.getOutputDirectory());
      for (ArtifactDescriptor artifact : artifacts)
      {
         try
         {
            artifact.writeArtifactToDisk(configuration.getOutputDirectory());
            log.info("Written artifact to disk " + artifact);
         }
         catch (IOException e)
         {
            log.warn("Error writing artifact to disk " + artifact, e);
         }
      }
   }

   public TCKArtifactDescriptor createArtifact(Class<?> declaringClass)
   {
      if (declaringClass.isAnnotationPresent(Artifact.class))
      {
         return new ArtifactProcessor(declaringClass, configuration.isStandalone(), configuration.getLibraryDirectory()).createArtifact();
      }
      else
      {
         return null;
      }
   }
   
}
