package org.jboss.testharness.impl.packaging.ear;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.PrintWriter;
import java.io.Writer;
import java.util.HashSet;
import java.util.Set;

import org.apache.log4j.Logger;
import org.jboss.testharness.api.ResourceDescriptor;
import org.jboss.testharness.impl.packaging.ResourceDescriptorImpl;
import org.jboss.testharness.impl.packaging.jsr299.JSR299ArtifactDescriptor;
import org.jboss.testharness.impl.packaging.war.WarArtifactDescriptor;

public class EarArtifactDescriptor extends JSR299ArtifactDescriptor
{
   
   private static final Logger log = Logger.getLogger(EarArtifactDescriptor.class);
   
   public static final String APPLICATION_XML_DESTINATION = "META-INF/application.xml";
   
   private final EjbJarArtifactDescriptor ejbJar;
   private final WarArtifactDescriptor war;
   
   public EarArtifactDescriptor(Class<?> declaringClass, String beansXmlSourceFileName, String ejbJarXml)
   {
      super(declaringClass, null);
      this.ejbJar = new EjbJarArtifactDescriptor(declaringClass, beansXmlSourceFileName, ejbJarXml).initialize();
      this.war = new WarArtifactDescriptor(declaringClass, null)
      {
         
         @Override
         protected boolean isAddBeansXml()
         {
            return false;
         }
         
         @Override
         public boolean isLibrariesSupported()
         {
            return false;
         }
         
      }.initialize();
      
   }
   
   @Override
   public EarArtifactDescriptor initialize()
   {
      super.initialize();
      InputStream applicationXml = null;
      try
      {
         applicationXml = getApplicationXml(getDeclaringClass());
         getResources().add(new ResourceDescriptorImpl(APPLICATION_XML_DESTINATION, applicationXml));
      }
      catch (IOException e)
      {
         log.warn("Error generating ear", e);
      }
      finally
      {
         if (applicationXml != null)
         {
            try
            {
               applicationXml.close();
            }
            catch (IOException e1)
            {
               log.error("Error generating ear", e1);
            }
         }
      }
      return this;
   }
   
   @Override
   public Set<Class<?>> getClasses()
   {
      return getEjbJar().getClasses();
   }
   
   public EjbJarArtifactDescriptor getEjbJar()
   {
      return ejbJar;
   }
   
   public WarArtifactDescriptor getWar()
   {
      return war;
   }
   
   @Override
   public String toString()
   {
      return "ear, " + super.toString();
   }
   
   @Override
   public String getExtension()
   {
      return ".ear";
   }
   
   @Override
   public boolean isLibrariesSupported()
   {
      return true;
   }
   
   @Override
   protected String getLibrariesRoot()
   {
      return "lib";
   }
   
   @Override
   protected boolean isAddBeansXml()
   {
      return false;
   }
   
   @Override
   public void create() throws IOException
   {
      for (ResourceDescriptor resource : new HashSet<ResourceDescriptor>(getResources()))
      {
         if (resource.getName().startsWith("war"))
         {
            getResources().remove(resource);
            war.getResources().add(new ResourceDescriptorImpl(resource.getName().substring(3), resource.getSource()));
         }
      }
      getResources().add(new ResourceDescriptorImpl(war.getDefaultName(), war.getJar()));
      getResources().add(new ResourceDescriptorImpl(ejbJar.getDefaultName(), ejbJar.getJar()));
      super.create();
   }
   
   private static InputStream getApplicationXml(Class<?> declaringClass) throws IOException
   {
      ByteArrayOutputStream os = new ByteArrayOutputStream();
      try
      {
         Writer writer = new PrintWriter(os);
         writer.write("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n");
         writer.write("<application xmlns=\"http://java.sun.com/xml/ns/javaee\" \n" + 
                   "\t\txmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" \n" +
                   "\t\txsi:schemaLocation=\"http://java.sun.com/xml/ns/javaee http://java.sun.com/xml/ns/javaee/application_5.xsd\" \n" +
                   "\t\tversion=\"5\">\n\n");
         writer.write("\t<display-name>JSR-99 TCK: " + declaringClass.getName() + "</display-name>\n\n");
         writer.write("\t<module>\n" +
              "\t\t<web>\n" +
                  "\t\t\t<web-uri>" + declaringClass.getName() +".war</web-uri>\n" + 
                  "\t\t\t<context-root>/" + declaringClass.getName() + "</context-root>\n" +
              "\t\t</web>\n" +
          "\t</module>\n\n");
         writer.write("\t<module>\n" +
                   "\t\t<ejb>" + declaringClass.getName() +".jar</ejb>\n" + 
           "\t</module>\n\n");
         writer.write("</application>\n");
         writer.flush();
         return new ByteArrayInputStream(os.toByteArray());
      }
      finally
      {
         os.close();
      }
   }
}
