/**
 * 
 */
package org.jboss.testharness.impl.packaging;

import static org.jboss.testharness.impl.util.Reflections.loadResources;

import java.io.File;
import java.io.IOException;
import java.lang.annotation.Annotation;
import java.net.URL;
import java.net.URLDecoder;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.Set;
import java.util.zip.ZipEntry;
import java.util.zip.ZipException;
import java.util.zip.ZipFile;

import org.apache.log4j.Logger;
import org.jboss.testharness.impl.util.Reflections;

public class ArtifactScanner
{
   private static final Logger log = Logger.getLogger(ArtifactScanner.class);

   private final String packageName;
   private final String packageNameAsPath;
   private final Class<? extends Annotation> annotationType;
   
   private final Set<Class<?>> classes = new HashSet<Class<?>>();
   
   public ArtifactScanner(String packageName, Class<? extends Annotation> annotation)
   {
      this.packageName = packageName;
      this.packageNameAsPath = packageName.replace(".", "/");
      this.annotationType = annotation;
   }
   
   private void scan()
   {
      try
      {
         Set<String> paths = new HashSet<String>();
         
         for (URL url : loadResources(packageNameAsPath))
         {
            String urlPath = url.getFile();
            urlPath = URLDecoder.decode(urlPath, "UTF-8");
            if ( urlPath.startsWith("file:") )
            {
                  urlPath = urlPath.substring(5);
            }
            if ( urlPath.indexOf('!')>0 )
            {
               urlPath = urlPath.substring(0, urlPath.indexOf('!'));
            }
            paths.add(urlPath);
         }
         handle(paths);
      }
      catch (IOException ioe) 
      {
         log.warn("could not read: " + this.packageName, ioe);
      }
   }
   
   private void handle(Set<String> paths) throws IOException
   {
      for ( String urlPath: paths )
      {
         log.trace("scanning: " + urlPath);
         handle(new File(urlPath), packageName.substring(0, packageName.lastIndexOf(".")));
         
      }
   }
   
   private void handleArchiveByFile(File file) throws IOException
   {
      try
      {
         log.trace("archive: " + file);
         ZipFile zip = new ZipFile(file);
         Enumeration<? extends ZipEntry> entries = zip.entries();
         while ( entries.hasMoreElements() )
         {
            ZipEntry entry = entries.nextElement();
            String name = entry.getName();
            if (name.startsWith(packageNameAsPath) && name.endsWith(".class"))
            {
               addClass(name.replace(".class", "").replace("/", "."));
            }
         }
      }
      catch (ZipException e)
      {
         throw new RuntimeException("Error handling file " + file, e);
      }
   }
   
   private void handle(File file, String packageName) throws IOException
   {
      if ( file.isDirectory() )
      {
         packageName = packageName + "." + file.getName();
         for ( File child: file.listFiles() )
         {
            handle(child, packageName);
         }
      }
      else if (file.getName().endsWith(".class") && ! file.getName().contains("$"))
      {
         addClass(packageName + "." + file.getName().replace(".class", ""));
         
      }
      else if (file.getName().endsWith(".jar"))
      {
         handleArchiveByFile(file);
      }
   }
   
   private void addClass(String className)
   {
      Class<?> clazz = Reflections.loadClass(className);
      if (clazz != null && clazz.isAnnotationPresent(Artifact.class))
      {
         if (annotationType != null && clazz.isAnnotationPresent(annotationType))
         {
            classes.add(clazz);
         }
         else
         {
            classes.add(clazz);
         }
      }
      else if (clazz == null)
      {
         log.warn("Unable to load class " + className);
      }
   }
   
   public Set<Class<?>> getClasses()
   {
      scan();
      return classes;
   }
   
}