package org.jboss.testharness;

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;

import org.apache.log4j.Logger;
import org.jboss.testharness.api.Configuration;
import org.jboss.testharness.api.DeploymentException;
import org.jboss.testharness.api.TestResult;
import org.jboss.testharness.api.TestResult.Status;
import org.jboss.testharness.impl.ConfigurationImpl;
import org.jboss.testharness.impl.packaging.ArtifactGenerator;
import org.jboss.testharness.impl.packaging.TCKArtifact;
import org.testng.IHookCallBack;
import org.testng.IHookable;
import org.testng.ITestContext;
import org.testng.ITestResult;
import org.testng.annotations.AfterClass;
import org.testng.annotations.AfterSuite;
import org.testng.annotations.BeforeClass;
import org.testng.annotations.BeforeSuite;

public abstract class AbstractTest implements IHookable
{

   private static Logger log = Logger.getLogger(AbstractTest.class);

   private static boolean inContainer = false;

   public static boolean isInContainer()
   {
      return inContainer;
   }

   public static void setInContainer(boolean inContainer)
   {
      AbstractTest.inContainer = inContainer;
   }

   private TCKArtifact artifact;
   private DeploymentException deploymentException;
   private boolean skipTest = false;

   private boolean isSuiteDeployingTestsToContainer()
   {
      return !isInContainer() && (!getCurrentConfiguration().isStandalone() || getCurrentConfiguration().isRunIntegrationTests());
   }

   private void generateArtifact() throws IOException
   {
      // If we are in the container, the artifact is already built
      if (!isInContainer())
      {
         ArtifactGenerator generator = new ArtifactGenerator(getCurrentConfiguration());
         artifact = postCreate(generator.createArtifact(this.getClass()));
      }
   }
   
   protected TCKArtifact postCreate(TCKArtifact artifact) throws IOException
   {
      return artifact;
   }

   private boolean isDeployToContainerNeeded()
   {
      /*
       * If this isn't running inside the container AND there is an artifact to
       * deploy AND EITHER we are in standalone mode and it isn't a unit test OR
       * we aren't in standalone mode THEN we need to deploy
       */
      return !isInContainer() && artifact != null && ((getCurrentConfiguration().isStandalone() && !artifact.isUnit() && getCurrentConfiguration().isRunIntegrationTests()) || !getCurrentConfiguration().isStandalone());
   }

   private void deployArtifact()
   {
      try
      {
         if (isDeployToContainerNeeded())
         {
            InputStream jar = null;
            try
            {
               jar = artifact.getJarAsStream();
               if (!getCurrentConfiguration().getContainers().deploy(jar, artifact.getDefaultName()))
               {
                  this.deploymentException = handleDeploymentFailure(getCurrentConfiguration().getContainers().getDeploymentException());
               }
            }
            finally
            {
               if (jar != null)
               {
                  jar.close();
               }
            }
         }
         else if (artifact != null && artifact.isUnit())
         {
            Set<Class<?>> classes = artifact.getClasses();
            if (!getCurrentConfiguration().getStandaloneContainers().deploy(classes, xmlResourcesAsList()))
            {
               this.deploymentException = handleDeploymentFailure(getCurrentConfiguration().getStandaloneContainers().getDeploymentException());
            }
         }
      }
      catch (IOException e)
      {
         throw new RuntimeException("Error connecting to the container", e);
      }
      if (artifact != null && artifact.getExpectedDeploymentException() != null)
      {
         if (deploymentException != null)
         {
            if (isThrowablePresent(artifact.getExpectedDeploymentException(), deploymentException.getCause()))
            {
               // We expect this exception, so ignore it
               deploymentException = null;
               skipTest = true;
            }
         }
         else
         {
            this.deploymentException = handleDeploymentFailure(new DeploymentException(artifact.getDefaultName(), new ExpectedException("Expected exception " + artifact.getExpectedDeploymentException() + " but none was thrown")));
         }
      }
   }

   protected DeploymentException handleDeploymentFailure(DeploymentException deploymentException)
   {
      return deploymentException;
   }

   private List<URL> xmlResourcesAsList()
   {
      List<URL> urlList = new ArrayList<URL>();
      if (artifact.getXmlConfig() != null)
      {
         urlList.add(artifact.getXmlConfig().getSource());
      }
      return urlList;
   }

   private void undeployArtifact() throws Exception
   {
      if (isDeployToContainerNeeded())
      {
         getCurrentConfiguration().getContainers().undeploy(artifact.getDefaultName());
      }
      if (getCurrentConfiguration().isStandalone() && artifact != null && artifact.isUnit())
      {
         getCurrentConfiguration().getStandaloneContainers().undeploy();
      }
   }

   private void checkAssertionsEnabled()
   {
      boolean assertionsEnabled = false;
      try
      {
         assert false;
      }
      catch (AssertionError error)
      {
         assertionsEnabled = true;
      }
      if (!assertionsEnabled)
      {
         throw new IllegalStateException("Assertions must be enabled!");
      }
   }

   @BeforeSuite(alwaysRun = true, groups = "scaffold")
   public void beforeSuite(ITestContext context) throws Exception
   {
      if (isSuiteDeployingTestsToContainer())
      {
         getCurrentConfiguration().getContainers().setup();
      }
      if (getCurrentConfiguration().isStandalone())
      {
         getCurrentConfiguration().getStandaloneContainers().setup();
      }
      checkAssertionsEnabled();
   }

   @AfterSuite(alwaysRun = true, groups = "scaffold")
   public void afterSuite() throws Exception
   {
      if (isSuiteDeployingTestsToContainer())
      {
         getCurrentConfiguration().getContainers().cleanup();
      }
      if (getCurrentConfiguration().isStandalone())
      {
         getCurrentConfiguration().getStandaloneContainers().cleanup();
      }
   }

   @BeforeClass(alwaysRun = true, groups = "scaffold")
   public void beforeClass() throws Throwable
   {
      generateArtifact();
      deployArtifact();

   }

   @AfterClass(alwaysRun = true, groups = "scaffold")
   public void afterClass() throws Exception
   {
      undeployArtifact();
      this.artifact = null;
      this.deploymentException = null;
      skipTest = false;
   }

   public void beforeMethod()
   {

   }

   public void afterMethod()
   {

   }

   public final void run(IHookCallBack callback, ITestResult testResult)
   {
      if (artifact == null && !isInContainer())
      {
         log.warn("Non @Artifact-test for testcase " + testResult.getMethod());
      }
      if (deploymentException != null)
      {
         testResult.setThrowable(deploymentException.getCause());
      }
      else if ((!isDeployToContainerNeeded() || artifact.isRunLocally()) && !skipTest)
      {
         try
         {
            beforeMethod();
            callback.runTestMethod(testResult);
         }
         finally
         {
            afterMethod();
            if (!getCurrentConfiguration().isStandalone() && !isInContainer() && !artifact.isRunLocally())
            {
               log.warn("Running testcase locally " + testResult.getMethod());
            }
         }
      }
      else if (!skipTest)
      {
         try
         {
            TestResult result = getCurrentConfiguration().getInContainerTestLauncher().launchTest(new TestNGInvocationContext(testResult, callback));
            if (result.getStatus().equals(Status.FAILED) || result.getStatus().equals(Status.SKIPPED))
            {
               testResult.setThrowable(result.getThrowable());
               testResult.setStatus(ITestResult.FAILURE);
            }
         }
         catch (IOException e)
         {
            throw new RuntimeException("Error connecting to the container", e);
         }
      }
   }

   protected Configuration getCurrentConfiguration()
   {
      return ConfigurationImpl.get();
   }

   protected String getContextPath()
   {
      return "http://" + getCurrentConfiguration().getHost() + "/" + this.getClass().getName() + "/";
   }

   protected boolean isThrowablePresent(Class<? extends Throwable> throwableType, Throwable throwable)
   {
      if (throwable == null)
      {
         return false;
      }
      else if (throwableType.isAssignableFrom(throwable.getClass()))
      {
         return true;
      }
      else
      {
         return isThrowablePresent(throwableType, throwable.getCause());
      }
   }
}