package org.jboss.testharness.impl.packaging;

import static org.jboss.testharness.impl.util.LogUtil.logger;
import static org.jboss.testharness.impl.util.Reflections.loadClass;
import static org.jboss.testharness.impl.util.Reflections.loadResourceAsStream;
import static org.jboss.testharness.impl.util.Reflections.loadResources;
import static org.jboss.testharness.impl.util.Strings.isEmpty;

import java.io.BufferedOutputStream;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URL;
import java.net.URLClassLoader;
import java.net.URLDecoder;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.Random;
import java.util.Set;
import java.util.jar.Attributes;
import java.util.jar.JarEntry;
import java.util.jar.JarOutputStream;
import java.util.jar.Manifest;
import java.util.logging.Level;
import java.util.zip.ZipEntry;
import java.util.zip.ZipException;
import java.util.zip.ZipFile;

import org.jboss.testharness.api.ResourceDescriptor;
import org.jboss.testharness.impl.ConfigurationFactory;
import org.jboss.testharness.impl.util.Files;

public class ArtifactDescriptor
{
   /**
    * Implementation of scanner which can scan a {@link URLClassLoader}
    * 
    * @author Thomas Heute
    * @author Gavin King
    * @author Norman Richards
    * @author Pete Muir
    */
   private static class URLPackageScanner
   {

      private final String packageName;

      private final String packageNamePath;
      private final boolean addRecursively;

      private final Set<String> classes = new HashSet<String>();

      public URLPackageScanner(Package pkg, boolean addRecursively)
      {
         this(pkg.getName(), addRecursively);
      }

      public URLPackageScanner(String packageName, boolean addRecursively)
      {
         this.packageName = packageName;
         this.packageNamePath = packageName.replace(".", "/");
         this.addRecursively = addRecursively;
      }

      private void scanPackage()
      {
         try
         {
            Set<String> paths = new HashSet<String>();

            for (URL url : loadResources(packageNamePath))
            {
               String urlPath = url.getFile();
               urlPath = URLDecoder.decode(urlPath, "UTF-8");
               if (urlPath.startsWith("file:"))
               {
                  urlPath = urlPath.substring(5);
               }
               if (urlPath.indexOf('!') > 0)
               {
                  urlPath = urlPath.substring(0, urlPath.indexOf('!'));
               }
               paths.add(urlPath);
            }
            handle(paths);
         }
         catch (IOException ioe)
         {
            logger().log(Level.WARNING, "could not read: " + packageName, ioe);
         }
      }

      private void handleArchiveByFile(File file) throws IOException
      {
         try
         {
            logger().finer("archive: " + file);
            ZipFile zip = new ZipFile(file);
            Enumeration<? extends ZipEntry> entries = zip.entries();
            while (entries.hasMoreElements())
            {
               ZipEntry entry = entries.nextElement();
               String name = entry.getName();
               if (name.startsWith(packageNamePath) && name.endsWith(".class") && (addRecursively || !name.substring(packageNamePath.length() + 1).contains("/")))
               {
                  String className = name.replace("/", ".").replace(".class", "");
                  classes.add(className);
               }
            }
         }
         catch (ZipException e)
         {
            throw new RuntimeException("Error handling file " + file, e);
         }
      }

      private void handle(Set<String> paths) throws IOException
      {
         for (String urlPath : paths)
         {
            logger().finest("scanning: " + urlPath);
            File file = new File(urlPath);
            if (file.isDirectory())
            {
               handle(file, packageName);
            }
            else
            {
               handleArchiveByFile(file);
            }
         }
      }

      private void handle(File file, String packageName)
      {
         for (File child : file.listFiles())
         {
            if (!child.isDirectory() && child.getName().endsWith(".class"))
            {
               classes.add(packageName + "." + child.getName().substring(0, child.getName().lastIndexOf(".class")));
            }
            else if (child.isDirectory() && addRecursively)
            {
               handle(child, packageName + "." + child.getName());
            }
         }
      }

      public Set<String> getClassNames()
      {
         scanPackage();
         return classes;
      }

   }

   private static class JarCreator
   {

      private final File root;

      public JarCreator(File root)
      {
         this.root = root;
      }

      public InputStream jar() throws IOException
      {
         ByteArrayOutputStream byteArrayOutputStream = null;
         JarOutputStream jarOutputStream = null;
         try
         {
            byteArrayOutputStream = new ByteArrayOutputStream();
            Manifest manifest = new Manifest();
            Attributes attributes = manifest.getMainAttributes();
            attributes.putValue("Created-By", "JSR-299 TCK Harness");
            attributes.put(Attributes.Name.MANIFEST_VERSION, "1.0");
            jarOutputStream = new JarOutputStream(byteArrayOutputStream, manifest);
            jar(root, jarOutputStream);
            jarOutputStream.close();
            return new ByteArrayInputStream(byteArrayOutputStream.toByteArray());
         }
         finally
         {
            if (jarOutputStream != null)
            {
               jarOutputStream.close();
            }
            else if (byteArrayOutputStream != null)
            {
               byteArrayOutputStream.close();
            }
         }
      }

      private void jar(File directory, JarOutputStream jarOutputStream) throws IOException
      {
         File[] children = directory.listFiles();
         // loop through dirList, and zip the files
         for (File child : children)
         {
            if (child.isDirectory())
            {
               jar(child, jarOutputStream);
            }
            else
            {
               FileInputStream fis = new FileInputStream(child);
               JarEntry jarEntry = new JarEntry(child.getPath().substring(root.getPath().length() + 1).replaceAll("\\" + File.separator, "/"));
               jarOutputStream.putNextEntry(jarEntry);
               Files.copy(fis, jarOutputStream);
               fis.close();
            }
         }
      }

   }

   public static final Random random = new Random(System.currentTimeMillis());

   private final Set<Class<?>> classes;
   private final Set<ResourceDescriptor> resources;
   private final Set<ResourceDescriptor> libraries;
   private final Class<?> declaringClass;

   // Cache
   private InputStream jar;
   private File explodedJar;

   private File classesRoot;
   private String classesRootName = "/";
   private File librariesRoot;
   private String librariesRootName = "/";
   private String extension = ".jar";
   private boolean librariesSupported = false;
   private boolean extrasSupported = true;

   public ArtifactDescriptor(Class<?> declaringClass)
   {
      classes = new HashSet<Class<?>>();
      resources = new HashSet<ResourceDescriptor>();
      this.declaringClass = declaringClass;
      this.libraries = new HashSet<ResourceDescriptor>();
   }

   public ArtifactDescriptor initialize()
   {
      if (extrasSupported)
      {
         for (String packageName : ConfigurationFactory.get().getExtraPackages())
         {
            addPackage(packageName, false);
         }
         getResources().addAll(ConfigurationFactory.get().getExtraResources());
      }
      return this;
   }

   public InputStream getJarAsStream() throws IOException
   {
      if (jar == null)
      {
         jar = new JarCreator(getExplodedJar()).jar();
      }
      return jar;
   }

   public URL getJar() throws IOException
   {
      InputStream is = null;
      try
      {
         is = getJarAsStream();
         File file = File.createTempFile(ArtifactDescriptor.class.getCanonicalName(), ".jar");
         file.deleteOnExit();
         Files.copy(is, file);
         is.close();
         return file.toURI().toURL();
      }
      finally
      {
         if (is != null)
         {
            is.close();
         }
      }
   }

   public File getExplodedJar() throws IOException
   {
      if (explodedJar == null)
      {
         create();
      }
      return explodedJar;
   }

   public void create() throws IOException
   {
      explodedJar = null;
      jar = null;

      File root = new File(System.getProperty("java.io.tmpdir") + "/" + getClass().getName() + "." + random.nextInt());
      root.mkdir();
      root.deleteOnExit();
      for (Class<?> clazz : classes)
      {
         copyClass(clazz, getClassesRoot(root));
      }
      for (ResourceDescriptor resourceDescriptor : resources)
      {
         copyResource(resourceDescriptor, root);
      }
      if (isLibrariesSupported())
      {
         for (ResourceDescriptor resourceDescriptor : libraries)
         {
            copyResource(resourceDescriptor, getLibraryRoot(root));
         }
      }
      this.explodedJar = root;
   }

   private static void copyClass(Class<?> clazz, File root) throws IOException
   {
      copyClass(clazz.getName(), root);
   }

   private static void copyResource(ResourceDescriptor resourceDescriptorImpl, File root) throws IOException
   {
      String directoryName;
      String fileName;
      if (resourceDescriptorImpl.getName().lastIndexOf("/") > 0)
      {
         directoryName = resourceDescriptorImpl.getName().substring(0, resourceDescriptorImpl.getName().lastIndexOf("/"));
         fileName = resourceDescriptorImpl.getName().substring(resourceDescriptorImpl.getName().lastIndexOf("/") + 1);
      }
      else
      {
         directoryName = "";
         fileName = resourceDescriptorImpl.getName();
      }
      if (isEmpty(fileName))
      {
         throw new IllegalArgumentException("Unable to determine source file name of " + resourceDescriptorImpl);
      }
      File directory = makeDirectoryStructure(root, directoryName);
      File file = new File(directory, fileName);
      file.createNewFile();
      file.deleteOnExit();
      Files.copy(resourceDescriptorImpl.getSource().openStream(), file);
   }

   private static void copyClass(String className, File root) throws IOException
   {
      InputStream clazzStream = null;
      try
      {
         String classFilePathName = getClassFileName(className);
         String directoryName = classFilePathName.substring(0, classFilePathName.lastIndexOf("/"));
         String classFileName = classFilePathName.substring(classFilePathName.lastIndexOf("/") + 1);
         File packageDirectory = makeDirectoryStructure(root, directoryName);
         File classFile = new File(packageDirectory, classFileName);
         classFile.createNewFile();
         classFile.deleteOnExit();
         clazzStream = loadResourceAsStream(classFilePathName);
         if (clazzStream == null)
         {
            throw new IllegalStateException("Eror loading " + className + " (" + classFilePathName + ")");
         }
         Files.copy(clazzStream, classFile);
      }
      finally
      {
         if (clazzStream != null)
         {
            clazzStream.close();
         }
      }
   }

   private static File makeDirectoryStructure(File root, String directoryName)
   {
      for (String directory : directoryName.split("\\/"))
      {
         root = new File(root, directory);
         root.mkdir();
         root.deleteOnExit();
      }
      return root;
   }

   public void writeArtifactToDisk(String outputDirectory) throws IOException
   {
      writeArtifactToDisk(outputDirectory, getDefaultName());
   }

   public void writeArtifactToDisk(String outputDirectory, String fileName) throws IOException
   {
      OutputStream os = null;
      InputStream jar = null;
      try
      {
         File file = new File(outputDirectory, fileName);
         file.createNewFile();
         os = new BufferedOutputStream(new FileOutputStream(file));
         jar = getJarAsStream();
         Files.copy(jar, os);
      }
      finally
      {
         if (os != null)
         {
            os.close();
         }
         if (jar != null)
         {
            jar.close();
         }
      }
   }

   public String getDefaultName()
   {
      return declaringClass.getName() + getExtension();
   }

   private static String getClassFileName(String className)
   {
      return className.replace('.', '/') + ".class";
   }

   public Set<Class<?>> getClasses()
   {
      return classes;
   }

   public Set<ResourceDescriptor> getResources()
   {
      return resources;
   }

   public void addPackage(Package pkg)
   {
      addPackage(pkg.getName(), false);
   }

   public void addPackage(String packageName, boolean addRecursively)
   {
      URLPackageScanner packageScanner = new URLPackageScanner(packageName, addRecursively);
      for (String className : packageScanner.getClassNames())
      {
         Class<?> clazz = loadClass(className);
         if (clazz == null)
         {
            System.out.println("Class null: " + className);
         }
         getClasses().add(clazz);
      }
   }

   public File getClassesRoot(File archiveRoot)
   {
      if (classesRoot == null)
      {
         classesRoot = makeDirectoryStructure(archiveRoot, getClassesRoot());
      }
      return classesRoot;
   }

   public String getClassesRoot()
   {
      return classesRootName;
   }

   public void setClassesRoot(String classesRoot)
   {
      this.classesRootName = classesRoot;
   }

   @Override
   public String toString()
   {
      return "Declared by: " + declaringClass.getName() + " Classes: " + classes + " Resources: " + resources;
   }

   public Class<?> getDeclaringClass()
   {
      return declaringClass;
   }

   public String getExtension()
   {
      return extension;
   }

   public void setExtension(String extension)
   {
      this.extension = extension;
   }
   
   public boolean isExtrasSupported()
   {
      return extrasSupported;
   }
   
   public void setExtrasSupported(boolean extrasSupported)
   {
      this.extrasSupported = extrasSupported;
   }

   public Set<ResourceDescriptor> getLibraries()
   {
      return libraries;
   }

   public File getLibraryRoot(File archiveRoot)
   {
      if (librariesRoot == null)
      {
         librariesRoot = makeDirectoryStructure(archiveRoot, getLibrariesRoot());
      }
      return librariesRoot;
   }

   public String getLibrariesRoot()
   {
      return librariesRootName;
   }

   public void setLibrariesRoot(String librariesRoot)
   {
      this.librariesRootName = librariesRoot;
   }

   public boolean isLibrariesSupported()
   {
      return librariesSupported;
   }

   public void setLibrariesSupported(boolean librariesSupported)
   {
      this.librariesSupported = librariesSupported;
   }
}
