package org.jboss.testharness.impl.packaging;

import static java.util.logging.Level.WARNING;
import static org.jboss.testharness.impl.packaging.PackagingType.EAR;
import static org.jboss.testharness.impl.packaging.PackagingType.WAR;
import static org.jboss.testharness.impl.util.LogUtil.logger;

import java.io.File;
import java.io.FilenameFilter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.jboss.testharness.api.Configuration;
import org.jboss.testharness.api.ResourceDescriptor;
import org.jboss.testharness.impl.packaging.ear.EarArtifactDescriptor;
import org.jboss.testharness.impl.packaging.ear.EjbJarArtifactDescriptor;
import org.jboss.testharness.impl.packaging.ear.EjbJarXml;
import org.jboss.testharness.impl.packaging.ear.PersistenceXml;
import org.jboss.testharness.impl.packaging.jsr299.BeansXml;
import org.jboss.testharness.impl.packaging.jsr299.Extension;
import org.jboss.testharness.impl.packaging.jsr299.JSR299ArtifactDescriptor;
import org.jboss.testharness.impl.packaging.jsr303.JSR303ArtifactDescriptor;
import org.jboss.testharness.impl.packaging.jsr303.ValidationXml;
import org.jboss.testharness.impl.packaging.war.WarArtifactDescriptor;
import org.jboss.testharness.impl.packaging.war.WebXml;

/**
 * @author Pete Muir
 */
public class ArtifactGenerator
{

   private static class ArtifactProcessor
   {

      private final boolean unit;
      private final boolean runLocally;
      private final boolean addDeclaringPackage;
      private final String beansXml;
      private final String extension;
      private final String validationXml;
      private final String ejbJarXml;
      private final String webXml;
      private final String persistenceXml;
      private final ArtifactType artifactType;
      private final PackagingType packagingType;
      private final Collection<ResourceDescriptor> resources;
      private final Collection<Class<?>> classes;
      private final Class<? extends Throwable> expectedDeploymentException;
      private final Set<ResourceDescriptor> extraLibraries;
      private final String[] packages;
      private final Class<?> declaringClass;
      private final boolean standalone;

      public ArtifactProcessor(Class<?> declaringClass, boolean standalone, String extraLibrariesDirectory)
      {
         this.standalone = standalone;
         this.declaringClass = declaringClass;

         if (declaringClass.isAnnotationPresent(Artifact.class))
         {
            Artifact artifactAnn = declaringClass.getAnnotation(Artifact.class);
            this.addDeclaringPackage = artifactAnn.addCurrentPackage();
            this.artifactType = artifactAnn.artifactType();
         }
         else
         {
            throw new IllegalStateException("Unable to find @Artifact on " + declaringClass);
         }

         if (declaringClass.isAnnotationPresent(BeansXml.class))
         {
            this.beansXml = asAbsolutePath(declaringClass.getAnnotation(BeansXml.class).value());
         }
         else
         {
            this.beansXml = null;
         }
         
         if (declaringClass.isAnnotationPresent(Extension.class))
         {
            this.extension = asAbsolutePath(declaringClass.getAnnotation(Extension.class).value());
         }
         else
         {
            this.extension = null;
         }

         if (declaringClass.isAnnotationPresent(ValidationXml.class))
         {
            this.validationXml = asAbsolutePath(declaringClass.getAnnotation(ValidationXml.class).value());
         }
         else
         {
            this.validationXml = null;
         }

         if (declaringClass.isAnnotationPresent(Packaging.class))
         {
            this.packagingType = declaringClass.getAnnotation(Packaging.class).value();
         }
         else
         {
            this.packagingType = WAR;
         }

         if (declaringClass.isAnnotationPresent(EjbJarXml.class))
         {
            this.ejbJarXml = asAbsolutePath(declaringClass.getAnnotation(EjbJarXml.class).value());
         }
         else
         {
            this.ejbJarXml = null;
         }
         
         if (declaringClass.isAnnotationPresent(WebXml.class))
         {
            this.webXml = asAbsolutePath(declaringClass.getAnnotation(WebXml.class).value());
         }
         else
         {
            this.webXml = null;
         }

         if (declaringClass.isAnnotationPresent(PersistenceXml.class))
         {
            this.persistenceXml = asAbsolutePath(declaringClass.getAnnotation(PersistenceXml.class).value());
         }
         else
         {
            this.persistenceXml = null;
         }

         if (declaringClass.isAnnotationPresent(IntegrationTest.class))
         {
            this.unit = false;
            this.runLocally = declaringClass.getAnnotation(IntegrationTest.class).runLocally();
         }
         else
         {
            this.unit = true;
            this.runLocally = false;
         }

         this.resources = new ArrayList<ResourceDescriptor>();
         if (declaringClass.isAnnotationPresent(Resource.class))
         {
            this.resources.add(asResourceDescriptor(declaringClass.getAnnotation(Resource.class)));
         }

         if (declaringClass.isAnnotationPresent(Resources.class))
         {
            this.resources.addAll(asResourceDescriptors(declaringClass.getAnnotation(Resources.class).value()));
         }

         if (declaringClass.isAnnotationPresent(Classes.class))
         {
            this.classes = Arrays.asList(declaringClass.getAnnotation(Classes.class).value());
            this.packages = declaringClass.getAnnotation(Classes.class).packages();
         }
         else
         {
            this.classes = Collections.emptyList();
            this.packages = new String[0];
         }
         if (declaringClass.isAnnotationPresent(ExpectedDeploymentException.class))
         {
            this.expectedDeploymentException = declaringClass.getAnnotation(ExpectedDeploymentException.class).value();
         }
         else
         {
            this.expectedDeploymentException = null;
         }
         if (extraLibrariesDirectory != null)
         {
            File directory = new File(extraLibrariesDirectory);
            this.extraLibraries = new HashSet<ResourceDescriptor>();
            if (directory.isDirectory())
            {
               for (File file : directory.listFiles(new FilenameFilter()
               {

                  public boolean accept(File dir, String name)
                  {
                     return name.endsWith(".jar");
                  }

               }))
               {
                  try
                  {
                     this.extraLibraries.add(new ResourceDescriptorImpl(file.getName(), file.toURI().toURL()));
                  }
                  catch (IOException e)
                  {
                     logger().log(WARNING, "Unable to load extra library", e);
                  }
               }
            }
         }
         else
         {
            this.extraLibraries = Collections.emptySet();
         }
      }

      public TCKArtifact createArtifact()
      {
         final TCKArtifact artifact = newArtifact(artifactType, packagingType, declaringClass, beansXml, extension, validationXml, ejbJarXml, webXml, persistenceXml, standalone, addDeclaringPackage);
         artifact.setUnit(unit);
         artifact.setRunLocally(runLocally);
         artifact.setExpectedDeploymentException(expectedDeploymentException);
         artifact.getClasses().addAll(classes);
         // Annoying hack
         artifact.getResources().removeAll(resources);
         artifact.getResources().addAll(resources);
         artifact.getLibraries().addAll(extraLibraries);
         for (String packageName : packages)
         {
            artifact.addPackage(packageName, false);
         }
         return artifact;
      }

      private Collection<ResourceDescriptor> asResourceDescriptors(Resource[] resources)
      {
         List<ResourceDescriptor> resourceDescriptorImpls = new ArrayList<ResourceDescriptor>();
         for (Resource resource : resources)
         {
            resourceDescriptorImpls.add(asResourceDescriptor(resource));
         }
         return resourceDescriptorImpls;
      }

      private ResourceDescriptor asResourceDescriptor(Resource resource)
      {
         return new ResourceDescriptorImpl(resource.destination(), asAbsolutePath(resource.source()));
      }

      private String asAbsolutePath(String path)
      {
         if (path.startsWith("/"))
         {
            return path.substring(1);
         }
         else
         {
            return declaringClass.getPackage().getName().replace(".", "/") + "/" + path;
         }
      }

      private static TCKArtifact newArtifact(ArtifactType artifactType, PackagingType packagingType, Class<?> declaringClass, String beansXml, String extension, String validationXml, String ejbJarXml, String webXml, String persistenceXml, boolean standalone, boolean addDeclaringPackage)
      {
         TCKArtifact artifact;

         if (!standalone && packagingType.equals(WAR))
         {
            artifact = createAndInitalizeWarArtifact(artifactType, declaringClass, beansXml, extension, validationXml, ejbJarXml, webXml, persistenceXml, addDeclaringPackage);
         }
         else if (!standalone && packagingType.equals(EAR))
         {
            artifact = createAndInitalizeEarArtifact(artifactType, declaringClass, beansXml, extension, validationXml, ejbJarXml, webXml, persistenceXml, addDeclaringPackage);
         }
         else
         {
            artifact = createBaseArtifact(artifactType, declaringClass, beansXml, extension, validationXml);
            if (addDeclaringPackage)
            {
               artifact.addPackage(declaringClass.getPackage());
            }
            artifact.initialize();
         }
         return artifact;
      }

      private static TCKArtifact createAndInitalizeEarArtifact(ArtifactType artifactType, Class<?> declaringClass, String beansXml, String extension, String validationXml, String ejbJarXml, String webXml, String persistenceXml, boolean addDeclaringPackage)
      {
         TCKArtifact ejbJar = EjbJarArtifactDescriptor.createEjbJarArtifact(createBaseArtifact(artifactType, declaringClass, beansXml, extension, validationXml), ejbJarXml, persistenceXml);
         if (addDeclaringPackage)
         {
            ejbJar.addPackage(declaringClass.getPackage());
         }
         TCKArtifact war = WarArtifactDescriptor.createWarArtifact(createBaseArtifact(artifactType, declaringClass, null, null, validationXml), webXml, null);
         TCKArtifact ear = EarArtifactDescriptor.createEarArtifact(createBaseArtifact(artifactType, declaringClass, null, null, validationXml), ejbJar, war);
         ear.initialize();

         return ear;
      }

      private static TCKArtifact createAndInitalizeWarArtifact(ArtifactType artifactType, Class<?> declaringClass, String beansXml, String extension, String validationXml, String ejbJarXml, String webXml, String persistenceXml, boolean addDeclaringPackage)
      {

         TCKArtifact artifact = createBaseArtifact(artifactType, declaringClass, beansXml, extension, validationXml);
         artifact = WarArtifactDescriptor.createWarArtifact(artifact, webXml, persistenceXml).initialize();
         if (ejbJarXml != null)
         {
            artifact.getResources().add(new ResourceDescriptorImpl("WEB-INF/ejb-jar.xml", ejbJarXml));
         }
         if (addDeclaringPackage)
         {
            artifact.addPackage(declaringClass.getPackage());
         }
         return artifact;
      }

      private static TCKArtifact createBaseArtifact(ArtifactType artifactType, Class<?> declaringClass, String beansXml, String extension, String validationXml)
      {
         TCKArtifact artifact;
         if (ArtifactType.JSR303.equals(artifactType))
         {
            artifact = new JSR303ArtifactDescriptor(declaringClass, validationXml);
         }
         else
         {
            artifact = new JSR299ArtifactDescriptor(declaringClass, beansXml, extension);
         }
         return artifact;
      }

      public boolean isUnit()
      {
         return unit;
      }
   }

   private final Configuration configuration;

   public ArtifactGenerator(Configuration configuration)
   {
      if (configuration == null)
      {
         throw new IllegalArgumentException("configuration must not be null");
      }
      this.configuration = configuration;
   }

   public List<TCKArtifact> createArtifacts(String packageName)
   {
      Set<Class<?>> classes = new HashSet<Class<?>>();
      classes.addAll(new ArtifactScanner(packageName, null).getClasses());
      List<TCKArtifact> artifacts = new ArrayList<TCKArtifact>();
      for (Class<?> clazz : classes)
      {
         artifacts.add(createArtifact(clazz));
      }
      return artifacts;
   }

   public void dumpArtifacts(String basePackageName)
   {
      if (basePackageName == null)
      {
         throw new IllegalArgumentException("Cannot dump artifacts as no package to dump from set. Specify org.jboss.testharness.testPackage");
      }
      List<TCKArtifact> artifacts = createArtifacts(basePackageName);
      File file = new File(configuration.getOutputDirectory());
      if (!file.exists())
      {
         file.mkdirs();
      }
      else if (file.isFile())
      {
         throw new IllegalStateException("Cannot use debug directory " + configuration.getOutputDirectory() + ", it already exists");
      }
      logger().info("Writing " + artifacts.size() + " artifact(s) to " + configuration.getOutputDirectory());
      for (TCKArtifact artifact : artifacts)
      {
         try
         {
            artifact.writeArtifactToDisk(configuration.getOutputDirectory());
            logger().info("Written artifact to disk " + artifact);
         }
         catch (IOException e)
         {
            logger().log(WARNING, "Error writing artifact to disk " + artifact, e);
         }
      }
   }

   public TCKArtifact createArtifact(Class<?> declaringClass)
   {
      if (declaringClass.isAnnotationPresent(Artifact.class))
      {
         return new ArtifactProcessor(declaringClass, configuration.isStandalone(), configuration.getLibraryDirectory()).createArtifact();
      }
      else
      {
         return null;
      }
   }

}
