package org.jboss.testharness.impl.packaging.ear;
import static java.util.logging.Level.WARNING;
import static org.jboss.testharness.impl.util.LogUtil.logger;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.PrintWriter;
import java.io.Writer;

import org.jboss.testharness.impl.packaging.ResourceDescriptorImpl;
import org.jboss.testharness.impl.packaging.TCKArtifact;

public class EarArtifactDescriptor
{

   public static final String APPLICATION_XML_DESTINATION = "META-INF/application.xml";

   private final TCKArtifact ejbJar;
   private final TCKArtifact war;
   private final TCKArtifact ear;

   public static TCKArtifact createEarArtifact(TCKArtifact ear, TCKArtifact ejbJar, TCKArtifact war)
   {
      EarArtifactDescriptor descriptor = new EarArtifactDescriptor(ear, ejbJar, war);
      return descriptor.ear;
   }

   private EarArtifactDescriptor(TCKArtifact ear, TCKArtifact ejbJar, TCKArtifact war)
   {
      this.ear = ear;
      this.ejbJar = ejbJar;
      this.war = war;
      this.ear.setExtrasSupported(false);
      this.war.setExtrasSupported(false);
      this.war.setLibrariesSupported(false);
      this.war.skipIncludeXmlConfig(true);
      this.war.addToClassPath(ejbJar.getDefaultName());
      this.ear.getClasses().clear();
      this.ear.setExtension(".ear");
      this.ear.setLibrariesSupported(true);
      this.ear.setLibrariesRoot("lib");
      this.ear.skipIncludeXmlConfig(true);
      this.war.setIncludeDeploymentProperties(false);
      this.ear.setSkipBeansXmlConfig(true);
      this.ear.setIncludeDeploymentProperties(false);
      this.ejbJar.initialize();
      this.war.initialize();
      createContainingJars();
      createApplicationXml();
   }

   private TCKArtifact createApplicationXml()
   {
      InputStream applicationXml = null;
      try
      {
         applicationXml = getApplicationXml(ear.getDeclaringClass());
         ear.getResources().add(new ResourceDescriptorImpl(APPLICATION_XML_DESTINATION, applicationXml));
      }
      catch (IOException e)
      {
         logger().log(WARNING, "Error generating ear", e);
      }
      finally
      {
         if (applicationXml != null)
         {
            try
            {
               applicationXml.close();
            }
            catch (IOException e1)
            {
               logger().log(WARNING, "Error generating ear", e1);
            }
         }
      }
      return ear;
   }

   public TCKArtifact getEarJar()
   {
      return ear;
   }

   public TCKArtifact getEjbJar()
   {
      return ejbJar;
   }

   public TCKArtifact getWar()
   {
      return war;
   }

   @Override
   public String toString()
   {
      return "EarArtifactDescriptor{" + "ejbJar=" + ejbJar + ", war=" + war + ", ear=" + ear + '}';
   }

   public void createContainingJars()
   {
      try
      {
         ear.getResources().add(new ResourceDescriptorImpl(war.getDefaultName(), war.getJar()));
         ear.getResources().add(new ResourceDescriptorImpl(ejbJar.getDefaultName(), ejbJar.getJar()));
      }
      catch (IOException ioe)
      {
         String message = "Error generating ear";
         logger().log(WARNING, message, ioe);
         throw new RuntimeException(message);
      }
   }

   private static InputStream getApplicationXml(Class<?> declaringClass) throws IOException
   {
      ByteArrayOutputStream os = new ByteArrayOutputStream();
      try
      {
         Writer writer = new PrintWriter(os);
         writer.write("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n");
         writer.write("<application xmlns=\"http://java.sun.com/xml/ns/javaee\" \n" + "\t\txmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" \n" + "\t\txsi:schemaLocation=\"http://java.sun.com/xml/ns/javaee http://java.sun.com/xml/ns/javaee/application_5.xsd\" \n" + "\t\tversion=\"5\">\n\n");
         writer.write("\t<display-name>JSR-99 TCK: " + declaringClass.getName() + "</display-name>\n\n");
         writer.write("\t<module>\n" + "\t\t<web>\n" + "\t\t\t<web-uri>" + declaringClass.getName() + ".war</web-uri>\n" + "\t\t\t<context-root>/" + declaringClass.getName() + "</context-root>\n" + "\t\t</web>\n" + "\t</module>\n\n");
         writer.write("\t<module>\n" + "\t\t<ejb>" + declaringClass.getName() + ".jar</ejb>\n" + "\t</module>\n\n");
         writer.write("</application>\n");
         writer.flush();
         return new ByteArrayInputStream(os.toByteArray());
      }
      finally
      {
         os.close();
      }
   }
}
