/*
* JBoss, Home of Professional Open Source
* Copyright 2005, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.test;

import java.io.File;
import java.io.FileOutputStream;
import java.io.PrintStream;
import java.lang.ref.WeakReference;
import java.lang.reflect.Field;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.ArrayList;
import java.util.StringTokenizer;

import org.jboss.profiler.jvmti.JVMTIInterface;

import junit.framework.TestCase;

/**
 * 
 * A Regular MemoryTestCase not using any container side.
 * 
 * @author <a href="clebert.suconic@jboss.com">Clebert Suconic</a>
 */
public class JBossMemoryTestCase extends TestCase
{
   public JBossMemoryTestCase()
   {
      super();
   }

   public JBossMemoryTestCase(String name)
   {
      super(name);
   }

   protected static ClassLoader newClassLoader(Class anyUserClass) throws Exception
   {
       URL classLocation =  anyUserClass.getProtectionDomain().getCodeSource().getLocation();
       StringTokenizer tokenString = new StringTokenizer(System.getProperty("java.class.path"),File.pathSeparator);
       String pathIgnore = System.getProperty("java.home");
       if (pathIgnore==null)
       {
          pathIgnore = classLocation.toString();
       } 
       
       ArrayList<URL> urls = new ArrayList<URL>();
       while (tokenString.hasMoreElements())
       {
          String value = tokenString.nextToken();
          URL itemLocation = new File(value).toURL();
          if (!itemLocation.equals(classLocation) && itemLocation.toString().indexOf(pathIgnore)>=0)
          {
             //System.out.println("Location:" + itemLocation);
             urls.add(itemLocation);
          }
       }
       
       URL[] urlArray= urls.toArray(new URL[urls.size()]);
       
       ClassLoader masterClassLoader = URLClassLoader.newInstance(urlArray,null);
       
       
       ClassLoader appClassLoader = URLClassLoader.newInstance(new URL[] {classLocation},masterClassLoader);
       
       return appClassLoader;
    }
   
   private static void resetObject(Object object, Field[] fields)
   {
      for (int fieldN=0;fieldN<fields.length;fieldN++)
      {
         try
         {
            System.out.print("        Field["+fieldN+"] "+fields[fieldN].getName());
            fields[fieldN].set(object,null);
            System.out.println("...done");
         }
         catch (Exception e)
         {
   
            System.out.println("...error " + e.getMessage());
            //System.out.println("Exception " + e.getMessage() + " happened during setField");
         }
      }
   }

   /**
    * If you started your class with -agentlib:jbossAgent in case of leakage (if className still loaded) a file (reportFile) will be created, and a heapSnapshot(./snapshot,mem)
    * 
    * @param weakReferenceOnLoader A weakReference to the created ClassLoader. If there is no references to this classLoader this reference will be cleared
    * @param className The class name supposed to be unloade.
    * @param reportHTMLFile the report file 
    * @throws Exception
    */
   protected void checkUnload(WeakReference weakReferenceOnLoader, String className, String reportHTMLFile) throws Exception
   {
      JVMTIInterface jvmti = new JVMTIInterface();
      if (jvmti.isActive())
      {
         jvmti.forceGC();
         Class clazz = jvmti.getClassByName(className);
         if (clazz!=null)
         {
            jvmti.heapSnapshot("snapshot", "mem");
            clazz=null;
            
            String report =jvmti.exploreClassReferences(className, 10, true, false, false, false, false);
            
            System.out.println(report);
            File outputfile = new File(reportHTMLFile);
            FileOutputStream outfile = new FileOutputStream(outputfile);
            PrintStream realoutput = new PrintStream(outfile);
            realoutput.println(report);
            realoutput.close();
            
            
            jvmti.forceGC();
            
            clazz = jvmti.getClassByName(className);
            
            if (clazz==null)
            {
                System.out.println("Attention: After clearing every field on AspectManager, GC could release the classLoader");
            }
            
            fail ("Class " + className + " still referenced. Look at report for more details");
         }
      }
      else
      {
         System.gc();
         Thread.sleep(1000);
      }
      assertNull("The classLoader is supposed to be released. Something is holding a reference. If you activate -agentlib:jbossAgent this testcase will generate a report with referenceHolders.",weakReferenceOnLoader.get());
   }
   private Field[] getDeclaredFields(Class clazz)
   {
      ArrayList<Field> list = new ArrayList<Field>();
      for (Class classIteration = clazz;classIteration!=null;classIteration=classIteration.getSuperclass())
      {
          Field[] fields = classIteration.getDeclaredFields();
          for (int i = 0; i < fields.length; i++)
          {
             fields[i].setAccessible(true);
             list.add(fields[i]);
          }
          
      }
      
      return list.toArray(new Field[list.size()]);
   }

   /** 
    * This method could be used for debug purposes
    * 
    * @param className the class name
    */
   protected void clearEverySingleFieldOnInstances(String className)
   {
      System.out.println("Clearing " + className);
      JVMTIInterface jvmti = new JVMTIInterface();
      Class classes[] = jvmti.getLoadedClasses();
      Object objects[] = null;
      
      for (int i=0;i<classes.length;i++)
      {
         if (classes[i].getName().equals(className))
         {
            Field fields[] = getDeclaredFields(classes[i]);
            objects = jvmti.getAllObjects(classes[i]);
            for (int j=0;j<objects.length;j++)
            {
               resetObject(objects[j], fields);
            }
            if (objects.length==0)
            {
               resetObject(null, fields);
            }
         }
      }
      classes= null;
      objects = null;
   }
   

}
