/******************************************************************************
 * JBoss, a division of Red Hat                                               *
 * Copyright 2006, Red Hat Middleware, LLC, and individual                    *
 * contributors as indicated by the @authors tag. See the                     *
 * copyright.txt in the distribution for a full listing of                    *
 * individual contributors.                                                   *
 *                                                                            *
 * This is free software; you can redistribute it and/or modify it            *
 * under the terms of the GNU Lesser General Public License as                *
 * published by the Free Software Foundation; either version 2.1 of           *
 * the License, or (at your option) any later version.                        *
 *                                                                            *
 * This software is distributed in the hope that it will be useful,           *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of             *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU           *
 * Lesser General Public License for more details.                            *
 *                                                                            *
 * You should have received a copy of the GNU Lesser General Public           *
 * License along with this software; if not, write to the Free                *
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA         *
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.                   *
 ******************************************************************************/
package org.jboss.unit.mc;

import org.jboss.unit.spi.pojo.TestLifeCycle;
import org.jboss.unit.spi.pojo.TestCaseLifeCycleException;
import org.jboss.unit.spi.pojo.TestCase;
import org.jboss.unit.mc.api.annotations.Bootstrap;
import org.jboss.kernel.plugins.bootstrap.basic.BasicBootstrap;
import org.jboss.kernel.plugins.deployment.xml.BeanXMLDeployer;
import org.jboss.kernel.spi.deployment.KernelDeployment;
import org.jboss.kernel.spi.dependency.KernelControllerContext;
import org.jboss.beans.metadata.plugins.AbstractBeanMetaData;
import org.jboss.beans.metadata.plugins.AbstractConstructorMetaData;
import org.jboss.beans.metadata.plugins.AbstractValueMetaData;

import java.util.Map;
import java.net.URL;

/**
 * @author <a href="mailto:julien@jboss.org">Julien Viet</a>
 * @version $Revision: 1.1 $
 */
public class MCTestLifeCycle extends BasicBootstrap implements TestLifeCycle
{

   /** . */
   private TestLifeCycle delegate;

   /** . */
   private BeanXMLDeployer beanDeployer;

   /** . */
   private KernelDeployment deployment;

   public MCTestLifeCycle(TestLifeCycle delegate)
   {
      this.delegate = delegate;
   }

   public TestCase newTestCase(String name) throws TestCaseLifeCycleException
   {
      return delegate.newTestCase(name);
   }

   public void testCaseParametrize(TestCase testCase, Map<String, String> parametrization) throws TestCaseLifeCycleException
   {
      delegate.testCaseParametrize(testCase, parametrization);
   }

   public void testCaseCreate(TestCase testCase) throws TestCaseLifeCycleException
   {
      Class<? extends Object> testClass = testCase.getPOJO().getClass();

      //
      Bootstrap bootstrapAnnotation = testClass.getAnnotation(Bootstrap.class);
      if (bootstrapAnnotation == null)
      {
         throw new TestCaseLifeCycleException("No Bootstrap annotation found");
      }

      URL url = testClass.getResource(bootstrapAnnotation.resourceName());
      if (url == null)
      {
         throw new TestCaseLifeCycleException("No URL found for bean xml " + bootstrapAnnotation.resourceName());
      }

      // setup MC
      run();

      try
      {
         String beanName = bootstrapAnnotation.beanName();

         //
         AbstractBeanMetaData bmd = new AbstractBeanMetaData(beanName, testClass.getName());
         AbstractConstructorMetaData cmd = new AbstractConstructorMetaData();
         bmd.setConstructor(cmd);
         TestCaseFactory factory = new TestCaseFactory(testCase.getPOJO());
         AbstractValueMetaData vmd = new AbstractValueMetaData(factory);
         cmd.setFactory(vmd);
         cmd.setFactoryMethod("getTestCase");
         getKernel().getController().install(bmd);
      }
      catch (Throwable throwable)
      {
         // Perform some MC throwable unwrap
         if (deployment != null)
         {
            // This may be MC saying that something went wrong
            if (throwable instanceof IllegalStateException)
            {
               // log.error("Test runtime context startup failed", throwable);
               for (KernelControllerContext kernelControllerContext : deployment.getInstalledContexts())
               {
                  Throwable t = kernelControllerContext.getError();
                  if (t != null)
                  {
                     throwable = t;
                     break;
                  }
               }
            }

            // Undeploy all beans
            if (deployment != null)
            {
               beanDeployer.undeploy(deployment);
            }
         }

         //
         throw new TestCaseLifeCycleException(throwable);
      }

      try
      {
         beanDeployer = new BeanXMLDeployer(getKernel());

         //
         deployment = beanDeployer.deploy(url);

         //
         beanDeployer.validate(deployment);
      }
      catch (Throwable throwable)
      {
         throw new TestCaseLifeCycleException(throwable);
      }

      //
      delegate.testCaseCreate(testCase);
   }

   public void testCaseInvoke(TestCase testCase) throws TestCaseLifeCycleException
   {
      delegate.testCaseInvoke(testCase);
   }

   public void testCaseDestroy(TestCase testCase)
   {
      try
      {
         delegate.testCaseDestroy(testCase);
      }
      catch (Throwable t)
      {
         t.printStackTrace();
      }

      // Clean up MC
      if (deployment != null)
      {
         beanDeployer.undeploy(deployment);
      }
   }
}
