/*
* JBoss, a division of Red Hat
* Copyright 2006, Red Hat Middleware, LLC, and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/

package org.jboss.unit.tooling.ant;

import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.Project;
import org.apache.tools.ant.taskdefs.Java;
import org.apache.tools.ant.types.Environment;
import org.apache.tools.ant.types.Path;
import org.jboss.unit.tooling.JavaBuilder;
import org.jboss.unit.tooling.TaskExecutingThread;
import org.jboss.unit.tooling.TestFailedException;
import org.jboss.unit.tooling.ant.configuration.ExcludeType;
import org.jboss.unit.tooling.ant.configuration.IncludeType;
import org.jboss.unit.tooling.ant.configuration.ParameterType;
import org.jboss.unit.tooling.ant.configuration.PropertyType;
import org.jboss.unit.tooling.ant.configuration.ReportsType;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;

/**
 * @author <a href="mailto:boleslaw dot dawidowicz at redhat anotherdot com">Boleslaw Dawidowicz</a>
 * @version $Revision: 0.1 $
 */
public class TestsType
{
   private String config;

   private boolean fork = true;

   private List<IncludeType> includes = new LinkedList<IncludeType>();

   private List<ExcludeType> excludes = new LinkedList<ExcludeType>();

   private ReportsType reports;

   private Project project;

   private Path path;

   private String suiteName;

   private List<PropertyType> properties = new ArrayList<PropertyType>();

   private Set<Environment.Variable> sysproperties = new HashSet<Environment.Variable>();

   private Set<ParameterType> parameters = new HashSet<ParameterType>();

   private JBossUnitTask jbossUnitTask;

   public TestsType()
   {
   }


   public void execute() throws BuildException
   {
      //Evaluate all the include filters

      Java javaTask = createJava();

      try
      {
         System.out.println("Invoking java: " + javaTask.getCommandLine().toString());
         TaskExecutingThread executor = new TaskExecutingThread(javaTask);
         executor.run();


      }
      catch (Exception e)
      {
         throw new BuildException("Testsuite executed with test failures"); 
      }

   }

   private Java createJava()
   {
      Java javaTask = null;
      try
      {

         JavaBuilder javaBuilder = new JavaBuilder(getProject());

         addArguments(javaBuilder);

         // Update settings using environment variables
         javaBuilder.applyVariables(getProject().getProperties());

         javaTask = javaBuilder.buildJava();

         // Apply task classpath
         if (getClasspath() != null)
         {
            Path javaPath = javaTask.createClasspath();
            javaPath.add(getClasspath());
         }

         // Set system properties
         for (Environment.Variable sysproperty : sysproperties)
         {
            javaTask.addSysproperty(sysproperty);
         }


      }
      catch (BuildException e)
      {
         throw new IllegalStateException("Failed to create <java> ant task: ", e);
      }

      return javaTask;
   }


   /**
    * Set all needed settings on JavaBuilder object based on passed configuration
    * @param javaBuilder
    */
   private void addArguments(JavaBuilder javaBuilder)
   {
      javaBuilder.setConfig(config);

      // JPDA
      javaBuilder.setJPDA(jbossUnitTask.isJpda());
      javaBuilder.setJPDAport(Integer.decode(jbossUnitTask.getJpdaPort()));
      javaBuilder.setSuspend(jbossUnitTask.isJpdaSuspend());

      // JUL
      javaBuilder.setUseJUL(jbossUnitTask.isUseJUL());

      // Assertions
      javaBuilder.setAssertions(jbossUnitTask.isAssertions());

      // Fork
      javaBuilder.setFork(isFork());

      // Fail on Error
      javaBuilder.setFailOnError(jbossUnitTask.isFailOnError());


      if (getSuiteName() != null)
      {
         javaBuilder.setSuiteName(getSuiteName());
      }

      if (getReports() != null && getReports().getXMLReports() != null)
      {
         javaBuilder.setXmlReportDir(getReports().getXMLReports().getToDir());
      }

      if (getReports() != null && getReports().getHTMLReports() != null)
      {
         javaBuilder.setHtmlReportDir(getReports().getHTMLReports().getToDir());
      }
      
      for (IncludeType include : includes)
      {
         //Check attributes consistency

         if (include.getId() != null && (include.getIds() != null || include.getKeywords() != null))
         {
            throw new IllegalStateException("The use of attribute 'id' in <include> tag should be exclusive");
         }


         if (include.getIds() != null)
         {
            javaBuilder.addUnparsedIncludeIDs(include.getIds());
         }
         if (include.getKeywords() != null)
         {
            javaBuilder.addUnparsedIncludeKeywords(include.getKeywords());
         }
         if (include.getId() != null)
         {
            javaBuilder.addUnparsedIncludeIDs(include.getId());
         }
      }

      for (ExcludeType exclude : excludes)
      {

         if (exclude.getId() != null && (exclude.getIds() != null || exclude.getKeywords() != null))
         {
            throw new IllegalStateException("The use of attribute 'id' in <exclude> tag should be exclusive");
         }

         if (exclude.getIds() != null)
         {
            javaBuilder.addUnparsedExcludeIDs(exclude.getIds());
         }
         if (exclude.getKeywords() != null)
         {
            javaBuilder.addUnparsedExcludeKeywords(exclude.getKeywords());
         }
         if (exclude.getId() != null)
         {
            javaBuilder.addUnparsedExcludeIDs(exclude.getId());
         }
      }

      // Properties

      if (properties.size() > 0)
      {

         for (PropertyType property : properties)
         {
            javaBuilder.addProperty(property.getName(),property.getValue());
         }

         javaBuilder.updateFQNProperties((Hashtable<String,String>)getProject().getProperties());
      }


      //Parameters

      for (ParameterType parameter : parameters)
      {
         javaBuilder.addParameter(parameter.getName(), parameter.getValues());
      }

      javaBuilder.updateFQNParameters((Hashtable<String,String>)getProject().getProperties());

   }

   public Set<Environment.Variable> getSysproperties()
   {
      return sysproperties;
   }

   public void setSysproperties(Set<Environment.Variable> sysproperties)
   {
      this.sysproperties = sysproperties;
   }

   public void addConfiguredInclude(IncludeType include)
   {
      this.includes.add(include);
   }

   public void addConfiguredExclude(ExcludeType exclude)
   {
      this.excludes.add(exclude);
   }

   public void addConfiguredReports(ReportsType reports)
   {
      this.reports = reports;
   }

   public void addConfiguredProperty(PropertyType property)
   {
      this.properties.add(property);
   }

   public void addConfiguredParameter(ParameterType param)
   {
      this.parameters.add(param);
   }

   public Set<ParameterType> getParameters()
   {
      return parameters;
   }

   public void setParameters(Set<ParameterType> parameters)
   {
      this.parameters = parameters;
   }

   public String getConfig()
   {
      return config;
   }

   public void setConfig(String config)
   {
      this.config = config;
   }


   public Path getClasspath()
   {
      return path;
   }

   public void addConfiguredClasspath(Path path)
   {
      this.path = path;
   }


   public boolean isFork()
   {
      return fork;
   }

   public void setFork(boolean fork)
   {
      this.fork = fork;
   }

   public Project getProject()
   {
      return project;
   }

   public void setProject(Project project)
   {
      this.project = project;
   }

   public ReportsType getReports()
   {
      return reports;
   }

   public void setReports(ReportsType reports)
   {
      this.reports = reports;
   }

   public String getSuiteName()
   {
      return suiteName;
   }

   public void setSuiteName(String suiteName)
   {
      this.suiteName = suiteName;
   }

   public JBossUnitTask getParent()
   {
      return jbossUnitTask;
   }

   public void setParent(JBossUnitTask jbossUnitTask)
   {
      this.jbossUnitTask = jbossUnitTask;
   }
}
