/**
 * Copyright (C) 2012 JBoss Inc
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.jbpm.formModeler.core.processing;


import org.jbpm.formModeler.api.model.Field;

import java.util.Map;

public interface FieldHandler {

    /**
     * Determine the list of class names this field can generate. That is, normally,
     * a field can generate multiple outputs (an input text can generate Strings,
     * Integers, ...)
     *
     * @return the set of class types that can be generated by this handler.
     */
    public String[] getCompatibleClassNames();

    /**
     * Read a parameter value (normally from a request), and translate it to
     * an object with desired class (that must be one of the returned by this handler)
     *
     * @return a object with desired class
     * @throws Exception
     */
    public Object getValue(Field field, String inputName, Map parametersMap, Map filesMap, String desiredClassName, Object previousValue) throws Exception;

    /**
     * Determine the value as a parameter map for a given input value. This is like the inverse operation of getValue()
     *
     * @param objectValue Object value to represent
     * @param pattern     Pattern to apply if any
     * @return a Map representing the parameter values expected inside a request that would cause the form
     *         to generate given object value as a result.
     */
    public Map getParamValue(String inputName, Object objectValue, String pattern);


    /**
     * When rendering a form, if field is handled by this handler, determine the
     * page that renders the input(s)
     *
     * @return a page to include
     */
    public String getPageToIncludeForRendering();

    /**
     * When rendering a form, if field is handled by this handler, determine the
     * page that renders the displaying of the value
     *
     * @return a page to include
     */
    public String getPageToIncludeForDisplaying();

    /**
     * Determine if supplied value is empty
     *
     * @param value
     * @return true if value can be considered empty
     */
    public boolean isEmpty(Object value);
}
